# EAF Module Testing

This directory contains all tests for the Entity Attributes Field (EAF) module.

## Test Environment Setup

The test environment has been configured with:

- **PHPUnit 11.5.42** (via drupal/core-dev ^11.2)
- **Test directory structure** organized by test type
- **Helper trait** for common test operations
- **Example test** to verify setup

## Running Tests

### Using the Test Runner Script

```bash
# Run all unit tests
ddev exec /var/www/html/web/modules/mycontrib/eaf/run-tests.sh unit

# Run kernel tests (when implemented)
ddev exec /var/www/html/web/modules/mycontrib/eaf/run-tests.sh kernel

# Run functional tests (when implemented)
ddev exec /var/www/html/web/modules/mycontrib/eaf/run-tests.sh functional

# Run all tests
ddev exec /var/www/html/web/modules/mycontrib/eaf/run-tests.sh all

# Run with coverage report
ddev exec /var/www/html/web/modules/mycontrib/eaf/run-tests.sh unit --coverage
```

### Using PHPUnit Directly

```bash
# Run specific test file
ddev exec /var/www/html/vendor/bin/phpunit \
  --bootstrap /var/www/html/web/core/tests/bootstrap.php \
  /var/www/html/web/modules/mycontrib/eaf/tests/src/Unit/ExampleUnitTest.php

# Run all unit tests
ddev exec /var/www/html/vendor/bin/phpunit \
  --bootstrap /var/www/html/web/core/tests/bootstrap.php \
  /var/www/html/web/modules/mycontrib/eaf/tests/src/Unit

# Run with coverage
ddev exec /var/www/html/vendor/bin/phpunit \
  --bootstrap /var/www/html/web/core/tests/bootstrap.php \
  --coverage-html /var/www/html/web/modules/mycontrib/eaf/coverage \
  /var/www/html/web/modules/mycontrib/eaf/tests/src/Unit
```

## Directory Structure

```
tests/
├── fixtures/                           # Test fixtures and sample data
├── src/
│   ├── Unit/                          # Unit tests (no database)
│   │   ├── Plugin/
│   │   │   ├── EntityAttribute/       # Attribute plugin tests
│   │   │   ├── Field/
│   │   │   │   ├── FieldType/        # Field type tests
│   │   │   │   ├── FieldWidget/      # Widget tests
│   │   │   │   └── FieldFormatter/   # Formatter tests
│   │   ├── EntityAttributePluginBaseTest.php
│   │   ├── FieldAttributeServiceTest.php
│   │   └── ExampleUnitTest.php
│   ├── Kernel/                        # Kernel tests (with database)
│   │   ├── EntityAttributePluginManagerTest.php
│   │   ├── EntityAttributesFieldItemListTest.php
│   │   ├── AttributesStorageKernelTest.php
│   │   └── FieldAttributeServiceKernelTest.php
│   ├── Functional/                    # Browser-based functional tests
│   │   ├── AttributesFieldConfigurationTest.php
│   │   ├── AttributesFieldWidgetTest.php
│   │   ├── EntityAttributesRenderTest.php
│   │   ├── FieldAttributesRenderTest.php
│   │   └── FieldItemAttributesRenderTest.php
│   ├── FunctionalJavascript/         # JavaScript functional tests
│   │   └── AccessibilityTest.php
│   └── Traits/                        # Reusable test traits
│       └── AttributeFieldTestTrait.php
└── README.md (this file)
```

## Test Types

### Unit Tests
- Test isolated class logic without dependencies
- Mock all external dependencies
- Fast execution
- Located in: `tests/src/Unit/`

### Kernel Tests
- Test with minimal Drupal installation
- Database available
- Service integration testing
- Located in: `tests/src/Kernel/`

### Functional Tests
- Full browser testing
- Complete Drupal installation
- Test user interactions
- Located in: `tests/src/Functional/`

### FunctionalJavascript Tests
- Browser testing with JavaScript support
- Requires Selenium/ChromeDriver
- Test AJAX interactions
- Located in: `tests/src/FunctionalJavascript/`

## Test Coverage Goals

- **Overall Target:** 95%+ code coverage
- **Plugin Base Classes:** 100%
- **Field Type/Widget:** 100%
- **Services:** 95%
- **Hooks/Preprocessors:** 85%

See `../../.claude/EAF_TESTING_PLAN.md` for complete testing strategy.

## Writing New Tests

### Unit Test Example

```php
<?php

declare(strict_types=1);

namespace Drupal\Tests\eaf\Unit;

use Drupal\Tests\UnitTestCase;

/**
 * Tests MyClass functionality.
 *
 * @group eaf
 * @coversDefaultClass \Drupal\eaf\MyClass
 */
final class MyClassTest extends UnitTestCase {

  /**
   * @covers ::myMethod
   */
  public function testMyMethod(): void {
    // Test implementation
  }
}
```

### Using the Test Trait

```php
<?php

use Drupal\Tests\eaf\Traits\AttributeFieldTestTrait;
use Drupal\Tests\BrowserTestBase;

class MyFunctionalTest extends BrowserTestBase {

  use AttributeFieldTestTrait;

  public function testSomething(): void {
    // Use trait methods
    $field = $this->createAttributeField('node', 'article');
    $this->enableAttributePlugins($field, ['entity_css_class']);
    // ... rest of test
  }
}
```

## Continuous Integration

The test suite is designed to run in CI/CD pipelines:

1. **Static Analysis** - PHPStan, PHP_CodeSniffer
2. **Unit Tests** - Fast, no dependencies
3. **Kernel Tests** - Database required
4. **Functional Tests** - Full Drupal + browser

## Troubleshooting

### Tests not found
Ensure you're running from the correct directory or using full paths.

### Database errors in Kernel tests
Make sure `SIMPLETEST_DB` environment variable is set in `phpunit.xml`.

### Browser tests failing
Check that `SIMPLETEST_BASE_URL` points to accessible site URL.

### Permission errors
Ensure `web/sites/simpletest/browser_output` directory exists and is writable.

## Contributing

When adding new features:
1. Write tests first (TDD)
2. Ensure all tests pass
3. Maintain or improve coverage
4. Follow Drupal coding standards

---

*For complete testing plan, see: `.claude/EAF_TESTING_PLAN.md`*
