<?php

declare(strict_types=1);

namespace Drupal\Tests\eaf\Unit;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eaf\EntityAttributePluginBase;
use Drupal\eaf\EntityAttributePluginManagerInterface;
use Drupal\Tests\UnitTestCase;

/**
 * Tests EntityAttributePluginBase functionality.
 *
 * @coversDefaultClass \Drupal\eaf\EntityAttributePluginBase
 * @group eaf
 */
final class EntityAttributePluginBaseTest extends UnitTestCase {

  /**
   * The plugin manager mock.
   *
   * @var \Drupal\eaf\EntityAttributePluginManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->pluginManager = $this->createMock(EntityAttributePluginManagerInterface::class);
  }

  /**
   * Creates a test plugin instance.
   *
   * @param array $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   Plugin ID.
   * @param array $plugin_definition
   *   Plugin definition.
   *
   * @return \Drupal\eaf\EntityAttributePluginBase
   *   The plugin instance.
   */
  protected function createPlugin(array $configuration = [], string $plugin_id = 'test_plugin', array $plugin_definition = []): EntityAttributePluginBase {
    return new class($configuration, $plugin_id, $plugin_definition, $this->pluginManager) extends EntityAttributePluginBase {
      public function prepareFormElement(array $element, mixed $value): array {
        return parent::prepareFormElement($element, $value);
      }
    };
  }

  /**
   * @covers ::__construct
   */
  public function testConstruct(): void {
    $configuration = ['foo' => 'bar'];
    $plugin_id = 'test_plugin';
    $plugin_definition = ['label' => new TranslatableMarkup('Test')];

    $plugin = $this->createPlugin($configuration, $plugin_id, $plugin_definition);

    $this->assertInstanceOf(EntityAttributePluginBase::class, $plugin);
  }

  /**
   * @covers ::getSettings
   * @dataProvider settingsDataProvider
   */
  public function testGetSettings(array $definition, array $expected): void {
    $plugin = $this->createPlugin([], 'test', $definition);
    $this->assertEquals($expected, $plugin->getSettings());
  }

  /**
   * Data provider for testGetSettings.
   */
  public static function settingsDataProvider(): array {
    return [
      'with settings' => [
        ['settings' => ['weight' => 10, 'parent' => 'foo']],
        ['weight' => 10, 'parent' => 'foo'],
      ],
      'empty settings' => [
        ['settings' => []],
        [],
      ],
    ];
  }

  /**
   * @covers ::formElement
   */
  public function testFormElement(): void {
    $definition = [
      'label' => new TranslatableMarkup('Test Label'),
      'settings' => ['weight' => 5],
    ];
    $plugin = $this->createPlugin([], 'test', $definition);

    $element = $plugin->formElement();

    $this->assertEquals('Test Label', (string) $element['#title']);
    $this->assertNull($element['#default_value']);
    $this->assertEquals(5, $element['#weight']);
    $this->assertSame($plugin, $element['#plugin']);
  }

  /**
   * @covers ::id
   */
  public function testId(): void {
    $plugin = $this->createPlugin([], 'my_plugin_id');
    $this->assertEquals('my_plugin_id', $plugin->id());
  }

  /**
   * @covers ::label
   * @dataProvider labelDataProvider
   */
  public function testLabel(array $definition, $expected): void {
    $plugin = $this->createPlugin([], 'test', $definition);
    $result = $plugin->label();

    if ($expected === NULL) {
      $this->assertNull($result);
    }
    else {
      $this->assertEquals($expected, (string) $result);
    }
  }

  /**
   * Data provider for testLabel.
   */
  public static function labelDataProvider(): array {
    return [
      'with label' => [
        ['label' => new TranslatableMarkup('My Label')],
        'My Label',
      ],
      'without label' => [
        [],
        NULL,
      ],
    ];
  }

  /**
   * @covers ::getDefaultValue
   */
  public function testGetDefaultValue(): void {
    $plugin = $this->createPlugin();
    $this->assertNull($plugin->getDefaultValue());
  }

  /**
   * @covers ::getParentPluginId
   * @dataProvider parentPluginDataProvider
   */
  public function testGetParentPluginId(array $settings, $expected): void {
    $definition = ['settings' => $settings];
    $plugin = $this->createPlugin([], 'test', $definition);
    $this->assertEquals($expected, $plugin->getParentPluginId());
  }

  /**
   * Data provider for parent plugin tests.
   */
  public static function parentPluginDataProvider(): array {
    return [
      'with parent' => [
        ['parent' => 'parent_plugin'],
        'parent_plugin',
      ],
      'without parent' => [
        [],
        NULL,
      ],
    ];
  }

  /**
   * @covers ::hasParentPlugin
   */
  public function testHasParentPluginTrue(): void {
    $definition = ['settings' => ['parent' => 'parent_id']];
    $plugin = $this->createPlugin([], 'test', $definition);
    $this->assertTrue($plugin->hasParentPlugin());
  }

  /**
   * @covers ::hasParentPlugin
   */
  public function testHasParentPluginFalse(): void {
    $definition = ['settings' => []];
    $plugin = $this->createPlugin([], 'test', $definition);
    $this->assertFalse($plugin->hasParentPlugin());
  }

  /**
   * @covers ::isPluginParent
   * @dataProvider parentCheckDataProvider
   */
  public function testIsPluginParent(array $settings, string $check_id, bool $expected): void {
    $definition = ['settings' => $settings];
    $plugin = $this->createPlugin([], 'test', $definition);
    $this->assertEquals($expected, $plugin->isPluginParent($check_id));
  }

  /**
   * Data provider for testIsPluginParent.
   */
  public static function parentCheckDataProvider(): array {
    return [
      'matches' => [
        ['parent' => 'parent_plugin'],
        'parent_plugin',
        TRUE,
      ],
      'does not match' => [
        ['parent' => 'parent_plugin'],
        'other_plugin',
        FALSE,
      ],
      'no parent set' => [
        [],
        'any_plugin',
        FALSE,
      ],
    ];
  }

  /**
   * @covers ::isSubtype
   */
  public function testIsSubtype(): void {
    $definition = ['settings' => ['parent' => 'parent_id']];
    $plugin = $this->createPlugin([], 'test', $definition);
    $this->assertTrue($plugin->isSubtype());
  }

  /**
   * @covers ::getWeight
   * @dataProvider weightDataProvider
   */
  public function testGetWeight(array $settings, int $expected): void {
    $definition = ['settings' => $settings];
    $plugin = $this->createPlugin([], 'test', $definition);
    $this->assertEquals($expected, $plugin->getWeight());
  }

  /**
   * Data provider for testGetWeight.
   */
  public static function weightDataProvider(): array {
    return [
      'with weight' => [
        ['weight' => 15],
        15,
      ],
      'without weight' => [
        [],
        0,
      ],
      'negative weight' => [
        ['weight' => -5],
        -5,
      ],
    ];
  }

  /**
   * @covers ::getValidationRulesDefinition
   */
  public function testGetValidationRulesDefinition(): void {
    $plugin = $this->createPlugin();
    $this->assertEquals([], $plugin->getValidationRulesDefinition());
  }

  /**
   * @covers ::getElementList
   * @dataProvider elementListDataProvider
   */
  public function testGetElementList($value, array $expected): void {
    $plugin = $this->createPlugin();
    $reflection = new \ReflectionClass($plugin);
    $method = $reflection->getMethod('getElementList');
    $result = $method->invoke($plugin, $value);
    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testGetElementList.
   */
  public static function elementListDataProvider(): array {
    return [
      'array input' => [
        ['class1', 'class2'],
        ['class1', 'class2'],
      ],
      'string input' => [
        'class1 class2 class3',
        ['class1', 'class2', 'class3'],
      ],
      'null input' => [
        NULL,
        [],
      ],
      'empty string' => [
        '',
        [],
      ],
      'object input' => [
        (object) ['foo' => 'bar'],
        [],
      ],
    ];
  }

  /**
   * @covers ::getListFromString
   * @dataProvider stringListDataProvider
   */
  public function testGetListFromString(?string $value, array $expected): void {
    $plugin = $this->createPlugin();
    $reflection = new \ReflectionClass($plugin);
    $method = $reflection->getMethod('getListFromString');
    $result = $method->invoke($plugin, $value);
    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testGetListFromString.
   */
  public static function stringListDataProvider(): array {
    return [
      'space separated' => [
        'class1 class2 class3',
        ['class1', 'class2', 'class3'],
      ],
      'comma separated' => [
        'class1,class2,class3',
        ['class1', 'class2', 'class3'],
      ],
      'mixed separators' => [
        'class1, class2 class3',
        ['class1', 'class2', 'class3'],
      ],
      'multiple spaces' => [
        'class1    class2',
        ['class1', 'class2'],
      ],
      'null value' => [
        NULL,
        [],
      ],
      'empty string' => [
        '',
        [],
      ],
    ];
  }

  /**
   * @covers ::prepareFormElement
   */
  public function testPrepareFormElementCss(): void {
    $plugin = $this->createPlugin();
    $element = ['#type' => 'textfield', '#subtype' => 'css'];
    $value = 'class1 class2';

    $result = $plugin->prepareFormElement($element, $value);

    $this->assertEquals('class1 class2', $result['#default_value']);
  }

  /**
   * @covers ::prepareFormElement
   */
  public function testPrepareFormElementSelect(): void {
    $plugin = $this->createPlugin();
    $element = ['#type' => 'select', '#default_value' => 'default'];
    $value = 'selected';

    $result = $plugin->prepareFormElement($element, $value);

    $this->assertEquals('selected', $result['#default_value']);
  }

  /**
   * @covers ::prepareFormElement
   */
  public function testPrepareFormElementSelectEmpty(): void {
    $plugin = $this->createPlugin();
    $element = ['#type' => 'select', '#default_value' => 'default'];
    $value = '';

    $result = $plugin->prepareFormElement($element, $value);

    $this->assertEquals('default', $result['#default_value']);
  }

  /**
   * @covers ::prepareFormElement
   */
  public function testPrepareFormElementNumber(): void {
    $plugin = $this->createPlugin();
    $element = ['#type' => 'number', '#default_value' => 10, '#min' => 0, '#max' => 100];
    $value = 50;

    $result = $plugin->prepareFormElement($element, $value);

    $this->assertEquals(50, $result['#default_value']);
    $this->assertEquals(0, $result['#min']);
    $this->assertEquals(100, $result['#max']);
  }

  /**
   * @covers ::prepareFormElement
   */
  public function testPrepareFormElementCheckboxes(): void {
    $plugin = $this->createPlugin();
    $element = ['#type' => 'checkboxes', '#default_value' => []];
    $value = ['option1' => 'option1', 'option2' => 'option2'];

    $result = $plugin->prepareFormElement($element, $value);

    $this->assertEquals($value, $result['#default_value']);
  }

  /**
   * @covers ::prepareFormElement
   */
  public function testPrepareFormElementCheckboxesWithObject(): void {
    $plugin = $this->createPlugin();
    $element = ['#type' => 'checkboxes', '#default_value' => []];
    $value = (object) ['option1' => 'option1'];

    $result = $plugin->prepareFormElement($element, $value);

    $this->assertIsArray($result['#default_value']);
    $this->assertEquals(['option1' => 'option1'], $result['#default_value']);
  }

  /**
   * @covers ::prepareFormElement
   */
  public function testPrepareFormElementDefault(): void {
    $plugin = $this->createPlugin();
    $element = ['#type' => 'textfield'];
    $value = 'test value';

    $result = $plugin->prepareFormElement($element, $value);

    $this->assertEquals('test value', $result['#default_value']);
    $this->assertEquals(32, $result['#size']);
  }

  /**
   * @covers ::setValue
   * @dataProvider setValueDataProvider
   */
  public function testSetValue($value, array $values, $section, array $expected): void {
    $plugin = $this->createPlugin([], 'test_plugin');
    $result = $plugin->setValue($value, $values, $section);
    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testSetValue.
   */
  public static function setValueDataProvider(): array {
    return [
      'without section' => [
        'value1',
        ['existing' => 'data'],
        NULL,
        ['existing' => 'data', 'test_plugin' => 'value1'],
      ],
      'with section' => [
        'value2',
        ['existing' => 'data'],
        'field_name',
        ['existing' => 'data', 'field_name' => ['test_plugin' => 'value2']],
      ],
      'array value' => [
        ['class1', 'class2'],
        [],
        NULL,
        ['test_plugin' => ['class1', 'class2']],
      ],
    ];
  }

}