<?php

declare(strict_types=1);

namespace Drupal\Tests\eaf\Unit\Plugin\EntityAttribute;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eaf\EntityAttributePluginManagerInterface;
use Drupal\eaf\Plugin\EntityAttribute\EntityCssClass;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests EntityCssClass plugin functionality.
 */
#[CoversClass(EntityCssClass::class)]
#[Group('eaf')]
final class EntityCssClassTest extends UnitTestCase {

  /**
   * The plugin manager mock.
   *
   * @var \Drupal\eaf\EntityAttributePluginManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->pluginManager = $this->createMock(EntityAttributePluginManagerInterface::class);

    // Mock the string translation service.
    $string_translation = $this->getStringTranslationStub();
    $container = $this->createMock('Symfony\Component\DependencyInjection\ContainerInterface');
    $container->expects($this->any())
      ->method('get')
      ->with('string_translation')
      ->willReturn($string_translation);
    \Drupal::setContainer($container);
  }

  /**
   * Creates a test plugin instance.
   *
   * @param array $configuration
   *   Plugin configuration.
   * @param array $plugin_definition
   *   Plugin definition.
   *
   * @return \Drupal\eaf\Plugin\EntityAttribute\EntityCssClass
   *   The plugin instance.
   */
  protected function createPlugin(array $configuration = [], array $plugin_definition = []): EntityCssClass {
    $definition = [
      'id' => 'entity_css_class',
      'label' => new TranslatableMarkup('Additional classes for the entity'),
      'settings' => ['weight' => 0],
    ] + $plugin_definition;

    return new EntityCssClass($configuration, 'entity_css_class', $definition, $this->pluginManager);
  }

  /**
   * Tests formElement method.
   */
  public function testFormElement(): void {
    $plugin = $this->createPlugin();
    $element = $plugin->formElement();

    $this->assertEquals('textfield', $element['#type']);
    $this->assertEquals('css', $element['#subtype']);
    $this->assertEquals(32, $element['#size']);
    $this->assertEquals(150, $element['#weight']);
    $this->assertArrayHasKey('#description', $element);
    $this->assertArrayHasKey('#title', $element);
    $this->assertArrayHasKey('#plugin', $element);
  }

  /**
   * Tests getDefaultValue method.
   */
  public function testGetDefaultValue(): void {
    $plugin = $this->createPlugin();
    $this->assertEquals('', $plugin->getDefaultValue());
  }

  /**
   * Tests setValue method.
   */
  #[DataProvider('setValueDataProvider')]
  public function testSetValue(?string $value, array $expected_classes, string $expected_value): void {
    $plugin = $this->createPlugin();
    $attributes = ['classes' => []];

    $result = $plugin->setValue($value, $attributes);

    $this->assertEquals($expected_classes, $result['classes']);
    $this->assertEquals($expected_value, $result['entity_css_class']);
  }

  /**
   * Data provider for testSetValue.
   */
  public static function setValueDataProvider(): array {
    return [
      'single class' => [
        'my-class',
        ['my-class'],
        'my-class',
      ],
      'multiple classes' => [
        'class1 class2 class3',
        ['class1', 'class2', 'class3'],
        'class1 class2 class3',
      ],
      'empty string' => [
        '',
        [''],
        '',
      ],
      'null value' => [
        NULL,
        [''],
        '',
      ],
    ];
  }

  /**
   * Tests setValue method cleans invalid CSS identifiers.
   */
  public function testSetValueCleansInvalidCssIdentifiers(): void {
    $plugin = $this->createPlugin();
    $attributes = ['classes' => []];

    // Test with invalid CSS identifiers that need cleaning.
    $result = $plugin->setValue('My-Class! @test #id', $attributes);
    // Html::cleanCssIdentifier converts these to valid CSS identifiers but not
    // change the cases.
    $this->assertContains('My-Class', $result['classes']);
    $this->assertNotContains('my-class', $result['classes']);
    $this->assertContains('test', $result['classes']);
    $this->assertContains('id', $result['classes']);
  }

  /**
   * Tests setValue method with multiple classes.
   */
  public function testSetValueMultipleClasses(): void {
    $plugin = $this->createPlugin();
    $attributes = ['classes' => ['existing-class']];

    $result = $plugin->setValue('new-class another-class', $attributes);
    $this->assertNotContains('existing-class', $result['classes']);
    $this->assertContains('new-class', $result['classes']);
    $this->assertContains('another-class', $result['classes']);
  }

  /**
   * Tests setValue method with empty string.
   */
  public function testSetValueEmptyString(): void {
    $plugin = $this->createPlugin();
    $attributes = ['classes' => ['existing']];

    $result = $plugin->setValue('', $attributes);

    // Empty string should still add an empty element to classes array.
    $this->assertNotContains('existing', $result['classes']);
    $this->assertEquals('', $result['entity_css_class']);
  }

  /**
   * Tests setValue method with section parameter.
   */
  public function testSetValueWithSection(): void {
    $plugin = $this->createPlugin();
    $attributes = ['classes' => []];

    // setValue calls parent::setValue which handles section parameter.
    $result = $plugin->setValue('test-class', $attributes, 'field_name');

    // Should set value in section.
    $this->assertArrayHasKey('field_name', $result);
    $this->assertArrayHasKey('entity_css_class', $result['field_name']);
    $this->assertEquals('test-class', $result['field_name']['entity_css_class']);
  }

  /**
   * Tests setValue method with existing classes.
   */
  public function testSetValueWithExistingClasses(): void {
    $plugin = $this->createPlugin();
    $attributes = ['classes' => ['class1', 'class2']];

    $result = $plugin->setValue('class3 class4 class5 class4', $attributes);
    // Should merge with existing classes.
    $this->assertCount(3, $result['classes']);
    $this->assertNotContains('class1', $result['classes']);
    $this->assertNotContains('class2', $result['classes']);
    $this->assertContains('class3', $result['classes']);
    $this->assertContains('class4', $result['classes']);
    $this->assertContains('class5', $result['classes']);
  }

}
