<?php

namespace Drupal\eaf;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Field attribute service class.
 *
 * This class handles the field's attributes.
 */
class FieldAttributeService {

  use StringTranslationTrait;

  /**
   * Attribute allowed field types.
   *
   * @var array
   */
  const ALLOWED_FIELD_TYPES = [
    'string',
    'string_long',
    'entity_reference',
    'entity_reference_revisions',
    'link',
  ];

  /**
   * The forbidden field names to attribute.
   *
   * @var array
   */
  const FORBIDDEN_FIELD_NAMES = [
    'default_langcode',
    'revision_default',
    'revision_uid',
    'revision_translation_affected',
    'path',
    'menu_link',
    // Paragraph entity fields.
    'parent_id',
    'parent_type',
    'parent_field_name',
    // @todo Add ability to alter field names.
  ];

  /**
   * The attribute plugin manager.
   *
   * @var \Drupal\eaf\EntityAttributePluginManagerInterface
   */
  protected $attributePluginManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructs a FieldAttributeService object.
   *
   * @param \Drupal\eaf\EntityAttributePluginManagerInterface $attribute_plugin_manager
   *   The attribute plugin manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   */
  public function __construct(
    EntityAttributePluginManagerInterface $attribute_plugin_manager,
    EntityFieldManagerInterface $entity_field_manager,
  ) {
    $this->attributePluginManager = $attribute_plugin_manager;
    $this->entityFieldManager = $entity_field_manager;
  }

  /**
   * Filter the fields if applicable for field attributes.
   *
   * @param object $field_definition
   *   Field definition.
   *
   * @return bool
   *   The field is allowed for attribute or not.
   */
  public function isAllowed($field_definition): bool {
    $type = $field_definition->getType();
    // @todo Move the decision to a separated method?.
    if (in_array($type, self::ALLOWED_FIELD_TYPES, TRUE) &&
      !$field_definition->isReadOnly() &&
      !in_array($field_definition->getName(), self::FORBIDDEN_FIELD_NAMES, TRUE)
    ) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * Provide the attribute section name.
   *
   * @return string
   *   The attribute section name.
   */
  public function getFieldAttributeSectionName(): string {
    return EntityAttributes::FIELD_ATTRIBUTES;
  }

  /**
   * Returns with the form state's entity.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form_state object.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|\Drupal\Core\Entity\EntityInterface
   *   The from entity.
   */
  public function getEntityFromFormState(FormStateInterface $form_state): EntityInterface {
    $form_object = $form_state->getFormObject();
    $entity = NULL;

    if (method_exists($form_object, 'getParagraph')) {
      /** @var \Drupal\paragraphs\ParagraphInterface $entity */
      $entity = $form_object->getParagraph();
    }
    elseif (method_exists($form_object, 'getEntity')) {
      /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
      $entity = $form_object->getEntity();
    }

    return $entity;
  }

  /**
   * Implements hook_field_widget_complete_form_alter().
   *
   * @see hook_field_widget_complete_form_alter()
   */
  public function fieldWidgetCompleteFormAlter(
    array &$element,
    FormStateInterface $form_state,
    array $context,
  ): void {
    /** @var \Drupal\Core\Field\FieldDefinitionInterface $field_definition */
    $field_definition = $context['items']->getFieldDefinition();
    $field_name = $field_definition->getName();

    $parent = $context['items']->getParent();
    $parent_entity = $parent->getEntity();
    $attribute_fields = $this->getAttributeFields($parent_entity);

    if (empty($attribute_fields)) {
      // No attribute field = nothing to do.
      return;
    }

    $attribute_field_name = array_keys($attribute_fields)[0];
    if ($attribute_field_name === $field_name) {
      // No attribute widget alter for attribute an field.
      return;
    }

    $settings = $this->getAttributeFieldSettings($parent_entity, $attribute_field_name);

    if (empty($settings)) {
      return;
    }

    /** @var \Drupal\eaf\EntityAttributesFieldItemList $field_item_list */
    $field_item_list = $parent_entity->get($attribute_field_name);
    $plugin_form_elements = $this->getAttributePluginManager()
      ->getAttributeFormElements();

    // Handling the whole field attribute plugins.
    if (isset($settings['field_attribute_plugins'][$field_name])) {
      $attribute_value = $field_item_list->getFieldAttributeValue($field_name);
      $attribute_elements = [];
      foreach ($settings['field_attribute_plugins'][$field_name] as $id => $status) {
        if (empty($status)) {
          // Filter out the not enabled elements.
          continue;
        }

        /** @var \Drupal\eaf\EntityAttributePluginInterface $plugin */
        $plugin = $plugin_form_elements[$id]['#plugin'];
        $value = $attribute_value[$id] ?? '';

        $attribute_elements[$id] = $plugin->prepareFormElement($plugin_form_elements[$id], $value);
        $attribute_elements[$id]['#tree'] = TRUE;
      }

      if ($attribute_elements) {
        if (method_exists($field_definition, 'getLabel')) {
          $label = $field_definition->getLabel();
        }
        else {
          $label = $this->t('This field');
        }

        $widget = [
          '#type' => 'details',
          '#title' => $this->t('@label attributes',
            ['@label' => $label]),
          '#open' => FALSE,
          '#attributes' => [
            'class' => ['field-attribute--details'],
          ],
          '#tree' => TRUE,
          '#weight' => -100,
        ] + $attribute_elements;

        $section = 'add_more';
        $is_paragraphs_widget = $element['widget']['#paragraphs_widget'] ?? FALSE;
        if ($is_paragraphs_widget) {
          if (!empty($element['widget']['add_more']['#suffix'])) {
            $element['widget']['add_more']['add_more_button_section']['#suffix'] = $element['widget']['add_more']['#suffix'];
            unset($element['widget']['add_more']['#suffix']);
          }
          $element['widget']['add_more']['#attributes']['class'][] = 'field-attribute--details';
          $widget['#weight'] = 100;
        }
        $element['widget'][$section][EntityAttributes::FIELD_ATTRIBUTES] = $widget;

      }
    }

    // Handling field item attribute plugins.
    if (isset($settings['field_item_attribute_plugins'][$field_name])) {
      $attribute_value = $field_item_list->getFieldItemAttributeValue($field_name);
      $children = Element::children($element['widget']);
      $count = count($children);
      foreach ($children as $index => $child) {
        if (!is_int($child)) {
          continue;
        }
        $attribute_elements = [];
        foreach ($settings['field_item_attribute_plugins'][$field_name] as $id => $status) {
          if (empty($status)) {
            // Filter out the not enabled elements.
            continue;
          }

          /** @var \Drupal\eaf\EntityAttributePluginInterface $plugin */
          $plugin = $plugin_form_elements[$id]['#plugin'];
          $value = $attribute_value[$child][$id] ?? '';

          $attribute_elements[$id] = $plugin->prepareFormElement($plugin_form_elements[$id], $value);
          $attribute_elements[$id]['#tree'] = TRUE;
        }

        if ($attribute_elements) {
          $index_label = '';
          if ($count) {
            $current = $index + 1;
            $index_label = '[' . $current . '] ';
          }

          if (method_exists($field_definition, 'getLabel')) {
            $label = $field_definition->getLabel();
          }
          else {
            $label = $this->t('this field');
          }
          $element['widget'][$child][EntityAttributes::FIELD_ITEM_ATTRIBUTES] = [
            '#type' => 'details',
            '#title' => $this->t('@index_labelAttributes for @label',
              [
                '@label' => $label,
                '@index_label' => $index_label,
              ]),
            '#open' => FALSE,
            '#attributes' => [
              'class' => ['field-attribute--details'],
            ],
            '#tree' => TRUE,
          ] + $attribute_elements;
        }
      }
    }

    $element['#attached']['library'][] = 'eaf/eaf.attribute-widget';
  }

  /**
   * Helper to get the attribute plugin manager.
   *
   * @return \Drupal\eaf\EntityAttributePluginManagerInterface
   *   The attribute plugin manager.
   */
  protected function getAttributePluginManager(): EntityAttributePluginManagerInterface {
    return $this->attributePluginManager;
  }

  /**
   * Get the field definitions of an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity.
   * @param string $attribute_field_name
   *   The field name.
   *
   * @return object|null
   *   The field definitions object or null.
   */
  public function getAttributeFieldDefinition($entity, $attribute_field_name): ?object {
    static $definitions = [];
    $type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();
    $section = $type_id . '-' . $bundle . '-' . $attribute_field_name;
    if (!isset($definitions[$section])) {
      $field_definitions = $this->entityFieldManager->getFieldDefinitions($type_id, $bundle);
      $definitions[$section] = $field_definitions[$attribute_field_name] ?? NULL;
    }

    return $definitions[$section] ?? NULL;
  }

  /**
   * Get the field settings of an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity.
   * @param string $attribute_field_name
   *   The field name.
   *
   * @return mixed|null
   *   The field settings array or null.
   */
  public function getAttributeFieldSettings(ContentEntityInterface $entity, string $attribute_field_name): mixed {
    static $field_settings = [];
    $section = $entity->getEntityTypeId() . '-' . $entity->bundle() . '-' . $attribute_field_name;

    if (!isset($field_settings[$section])) {
      // Get the field settings.
      $attribute_field_definition = $this->getAttributeFieldDefinition($entity, $attribute_field_name);
      if ($attribute_field_definition) {
        $settings = $attribute_field_definition->getSettings();
      }
      $field_settings[$section] = $settings ?? NULL;
    }

    return $field_settings[$section] ?? [];
  }

  /**
   * Get the entity's attribute type fields.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity.
   *
   * @return array
   *   The field names - label array of the attribute fields.
   *
   * @todo give back only the first????
   */
  public function getAttributeFields(ContentEntityInterface $entity): array {
    $attribute_type_fields = [];

    if (method_exists($entity, 'getFieldDefinitions')) {
      $field_definitions = $entity->getFieldDefinitions();
      foreach ($field_definitions as $field_name => $def) {
        if ($def->getType() === 'field_attributes_storage') {
          $attribute_type_fields[$field_name] = $def->getLabel();
        }
      }
    }

    return $attribute_type_fields;
  }

}
