<?php

declare(strict_types=1);

namespace Drupal\eaf\Hook;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\eaf\FieldAttributeService;

/**
 * Hook implementations for eaf module.
 */
final class EafHooks {

  /**
   * Constructs an EafHooks object.
   */
  public function __construct(
    private readonly FieldAttributeService $fieldAttributeService,
  ) {}

  /**
   * Implements hook_field_widget_complete_form_alter().
   */
  #[Hook('field_widget_complete_form_alter')]
  public function fieldWidgetCompleteFormAlter(array &$element, FormStateInterface $form_state, array $context): void {
    $this->fieldAttributeService->fieldWidgetCompleteFormAlter($element, $form_state, $context);
  }

  /**
   * Implements template_preprocess_field().
   */
  #[Hook('preprocess_field')]
  public function preprocessField(array &$variables): void {
    $element = $variables['element'];
    $variables['field_name'] = $element['#field_name'];

    // @todo How to handle the title field?
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    $entity = $element['#object'] ?? NULL;
    if (!$entity) {
      return;
    }

    $attribute_fields = $this->fieldAttributeService->getAttributeFields($entity);
    if (empty($attribute_fields)) {
      return;
    }

    $attribute_field_name = array_keys($attribute_fields)[0];
    $attributes = $entity->get($attribute_field_name)
      ->getFieldAttributeValue($variables['field_name']);

    $variables['field_attributes'] = $attributes ?? [];

    $attributes_item = $entity->get($attribute_field_name)
      ->getFieldItemAttributeValue($variables['field_name']);
    $variables['field_item_attributes'] = $attributes_item;

    // Creating variables for the template.
    $variables['entity_type'] = $element['#entity_type'];
    $variables['field_type'] = $element['#field_type'];
    $variables['label_display'] = $element['#label_display'];
  }

  /**
   * Implements hook_preprocess_node().
   *
   * This is an example preprocess how to add attributes to entities.
   */
  #[Hook('preprocess_node')]
  public function preprocessNode(array &$variables): void {
    $entity = $variables['node'];
    $this->preprocessEntityCommon($variables, $entity);
  }

  /**
   * Implements hook_preprocess_paragraph().
   */
  #[Hook('preprocess_paragraph')]
  public function preprocessParagraph(array &$variables): void {
    $entity = $variables['paragraph'];
    $this->preprocessEntityCommon($variables, $entity);
  }

  /**
   * Implements hook_preprocess_form_element__new_storage_type().
   */
  #[Hook('preprocess_form_element__new_storage_type')]
  public function preprocessFormElementNewStorageType(array &$variables): void {
    $variables['#attached']['library'][] = 'eaf/eaf.field-type-icon';
  }

  /**
   * Common preprocess logic for entity types.
   *
   * @param array $variables
   *   Variables array.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   * @param string $section
   *   The section name for attributes.
   */
  private function preprocessEntityCommon(array &$variables, ContentEntityInterface $entity, string $section = 'entity_attributes'): void {
    $fields = $this->fieldAttributeService->getAttributeFields($entity);

    $field_name = array_key_first($fields);
    if ($field_name) {
      $value = $entity->get($field_name)->getValue();
      $variables[$section] = $value[0]['value'] ?? [];
    }
    else {
      $variables[$section] = new \stdClass();
    }
  }

}
