<?php

namespace Drupal\eaf\Plugin\Field\FieldType;

use Drupal\Component\Utility\Random;
use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\eaf\EntityAttributes;
use Drupal\eaf\EntityAttributesFieldItemList;

/**
 * Defines the 'field_attributes_storage' field type.
 */
#[FieldType(
  id: "field_attributes_storage",
  label: new TranslatableMarkup("Entity attributes field"),
  description: new TranslatableMarkup('Stores a collection of entity or field attributes.'),
  category: "general",
  default_widget: "eaf_widget",
  default_formatter: "string",
  list_class: EntityAttributesFieldItemList::class,
  cardinality: 1
)]
class AttributesStorageItem extends FieldItemBase implements AttributesStorageInterface {

  /**
   * {@inheritdoc}
   */
  public static function defaultStorageSettings() {
    $settings = [];
    return $settings + parent::defaultStorageSettings();
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    $settings = [
      'entity_attribute_plugins' => [],
      'field_attribute_plugins' => [],
      'field_item_attribute_plugins' => [],
    ];

    return $settings + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state): array {
    /** @var \Drupal\eaf\EntityAttributePluginManagerInterface $manager */
    $manager = \Drupal::service('eaf.eaf_plugin_manager');
    $plugins = $manager->getAll();
    $settings = $this->getSetting('entity_attribute_plugins');
    $options = [];
    $enabled = [];
    foreach ($plugins as $id => $plugin) {
      $options[$id] = $plugin->label();
      if (!empty($settings[$id])) {
        $enabled[] = $id;
      }
    }

    $element['entity_attribute_plugins'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed attribute plugins for this type'),
      '#options' => $options,
      '#default_value' => $enabled,
    ];

    // Get the entity. @todo DI.
    /** @var \Drupal\eaf\FieldAttributeService $fieldAttributeService */
    $fieldAttributeService = \Drupal::service('eaf.field_attribute.service');

    /** @var \Drupal\Core\Field\FieldConfigInterface $field */
    $field = $form_state->getFormObject()->getEntity();
    $target_entity_bundle = $field->getTargetBundle();
    $target_entity_type_id = $field->getTargetEntityTypeId();

    // @todo DI.
    /** @var \Drupal\Core\Entity\EntityFieldManager $entity_field_manager */
    $entity_field_manager = \Drupal::service('entity_field.manager');
    $target_field_definitions = $entity_field_manager->getFieldDefinitions($target_entity_type_id, $target_entity_bundle);
    $allowed_fields = [];
    foreach ($target_field_definitions as $field_name => $def) {
      if ($fieldAttributeService->isAllowed($def)) {
        $allowed_fields[$field_name] = $def->getLabel();
      }
    }

    if ($allowed_fields) {
      // We have fields for attributes so add the attribute config form
      // for the fields.
      // @todo fix the schema file for field_attribute_plugins section.
      $fields_settings = $this->getSetting('field_attribute_plugins');

      $element['field_attribute_plugins'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Setup allowed attribute plugins for fields'),
      ];

      foreach ($allowed_fields as $field_name => $label) {
        $enabled = [];
        if (!empty($fields_settings[$field_name])) {
          foreach ($fields_settings[$field_name] as $plugin_id => $status) {
            if (!empty($status)) {
              $enabled[] = $plugin_id;
            }
          }
        }
        $element['field_attribute_plugins'][$field_name] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Allowed attributes for @name', ['@name' => $label]),
          '#options' => $options,
          '#default_value' => $enabled,
        ];
      }

      // Handling field items.
      $fields_settings = $this->getSetting('field_item_attribute_plugins');
      $element['field_item_attribute_plugins'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Setup allowed attribute plugins for field items'),
      ];

      foreach ($allowed_fields as $field_name => $label) {
        $enabled = [];
        if (!empty($fields_settings[$field_name])) {
          foreach ($fields_settings[$field_name] as $plugin_id => $status) {
            if (!empty($status)) {
              $enabled[] = $plugin_id;
            }
          }
        }
        $element['field_item_attribute_plugins'][$field_name] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Allowed attributes for @name items', ['@name' => $label]),
          '#options' => $options,
          '#default_value' => $enabled,
        ];
      }
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function fieldSettingsToConfigData(array $settings): array {
    $new_settings = [
      'entity_attribute_plugins' => [],
      'field_attribute_plugins' => [],
      'field_item_attribute_plugins' => [],
    ];

    foreach ($settings['entity_attribute_plugins'] as $id => $value) {
      $new_settings['entity_attribute_plugins'][$id] = (bool) $value;
    }

    foreach ($settings['field_attribute_plugins'] as $field_name => $plugins) {
      foreach ($plugins as $id => $status) {
        $new_settings['field_attribute_plugins'][$field_name][$id] = (bool) $status;
      }
    }

    foreach ($settings['field_item_attribute_plugins'] as $field_name => $plugins) {
      foreach ($plugins as $id => $status) {
        $new_settings['field_item_attribute_plugins'][$field_name][$id] = (bool) $status;
      }
    }

    return $new_settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    return [
      'columns' => [
        'value' => [
          'type' => 'text',
          'size' => 'normal',
          'not null' => FALSE,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getValue() {
    $values = parent::getValue();

    $decoded_value = [];
    if (isset($values['value'])) {
      if (is_string($values['value'])) {
        $decoded_value['value'] = json_decode($values['value'] ?? '', TRUE);
      }
      else {
        $decoded_value = $values;
      }
    }
    else {
      $decoded_value['value'] = $values;
    }

    return $decoded_value;
  }

  /**
   * {@inheritdoc}
   */
  public function setValue($values, $notify = TRUE): void {
    $value = [
      'value' => '',
    ];

    if (is_array($values) && isset($values['value'])) {
      $value['value'] = $values['value'];
    }

    parent::setValue($value, $notify);
  }

  /**
   * Set the field attributes section of the content field.
   *
   * @param array $value
   *   The new value of the field attributes section or just for a section.
   * @param string $attribute_type
   *   The attribute type - field or entity.
   * @param string $target_section
   *   The field's $field_definition->getConfigTarget() value like
   *   node.test_content.field_c_subtitle for a node bundle test_content or
   *   paragraph.p_section.field_p_title p_section bundle for paragraph.
   *
   * @throws \Drupal\Core\TypedData\Exception\ReadOnlyException
   */
  public function setAttributeValue(
    array $value,
    string $attribute_type = EntityAttributes::ENTITY_ATTRIBUTES,
    string $target_section = '',
  ): void {
    $raw = $this->getValue();

    if (!in_array($attribute_type, [
      EntityAttributes::ENTITY_ATTRIBUTES,
      EntityAttributes::FIELD_ATTRIBUTES,
      EntityAttributes::FIELD_ITEM_ATTRIBUTES,
    ])) {
      $attribute_type = EntityAttributes::ENTITY_ATTRIBUTES;
    }

    if ($target_section) {
      $raw[$attribute_type][$target_section] = $value;
    }
    else {
      $raw[$attribute_type] = $value;
    }

    $this->setValue($raw);
  }

  /**
   * Get the field attributes for a field or all stored values.
   *
   * @param string $attribute_type
   *   The type.
   * @param string|null $target_section
   *   The field's $field_definition->getConfigTarget().
   *
   * @return array
   *   The stored field attribute value.
   *
   * @see setFieldAttributeValue()
   */
  public function getAttributeValue(
    string $attribute_type = EntityAttributes::ENTITY_ATTRIBUTES,
    ?string $target_section = NULL,
  ): array {
    if (!in_array($attribute_type, [
      EntityAttributes::ENTITY_ATTRIBUTES,
      EntityAttributes::FIELD_ATTRIBUTES,
      EntityAttributes::FIELD_ITEM_ATTRIBUTES,
    ])) {
      $attribute_type = EntityAttributes::ENTITY_ATTRIBUTES;
    }

    $raw = $this->getValue();
    $value = $raw['value'] ?? [];

    if ($target_section) {
      return $value[$attribute_type][$target_section] ?? [];
    }

    return $value[$attribute_type] ?? [];
  }

  /**
   * Set the field attributes section of the content field.
   *
   * @param array $value
   *   The new value of the field attributes section or just for a section.
   * @param string $target_section
   *   The field's $field_definition->getConfigTarget() value like
   *   node.test_content.field_c_subtitle for a node bundle test_content or
   *   paragraph.p_section.field_p_title p_section bundle for paragraph.
   *
   * @throws \Drupal\Core\TypedData\Exception\ReadOnlyException
   */
  public function setEntityAttributeValue(array $value, ?string $target_section = NULL): void {
    $this->setAttributeValue($value, EntityAttributes::ENTITY_ATTRIBUTES, $target_section);
  }

  /**
   * Set the field attributes section of the content field.
   *
   * @param array $value
   *   The new value of the field attributes section or just for a section.
   * @param string $target_section
   *   The field's $field_definition->getConfigTarget() value like
   *   node.test_content.field_c_subtitle for a node bundle test_content or
   *   paragraph.p_section.field_p_title p_section bundle for paragraph.
   *
   * @throws \Drupal\Core\TypedData\Exception\ReadOnlyException
   */
  public function setFieldAttributeValue(array $value, ?string $target_section = NULL): void {
    $this->setAttributeValue($value, EntityAttributes::FIELD_ATTRIBUTES, $target_section);
  }

  /**
   * Set the field item attributes section of the content field.
   *
   * @param array $value
   *   The new value of the field attributes section or just for a section.
   * @param string $target_section
   *   The field's $field_definition->getConfigTarget() value like
   *   node.test_content.field_c_subtitle for a node bundle test_content or
   *   paragraph.p_section.field_p_title p_section bundle for paragraph.
   *
   * @throws \Drupal\Core\TypedData\Exception\ReadOnlyException
   */
  public function setFieldItemAttributeValue(array $value, ?string $target_section = NULL): void {
    $this->setAttributeValue($value, EntityAttributes::FIELD_ITEM_ATTRIBUTES, $target_section);
  }

  /**
   * Get the field attributes for a field or all stored values.
   *
   * @param string|null $target_section
   *   The field's $field_definition->getConfigTarget().
   *
   * @return array
   *   The stored field attribute value.
   *
   * @see setFieldAttributeValue()
   */
  public function getEntityAttributeValue(?string $target_section = NULL): array {
    return $this->getAttributeValue(EntityAttributes::ENTITY_ATTRIBUTES, $target_section);
  }

  /**
   * Get the field attributes for a field or all stored values.
   *
   * @param string|null $target_section
   *   The field's $field_definition->getConfigTarget().
   *
   * @return array
   *   The stored field attribute value.
   *
   * @see setFieldAttributeValue()
   */
  public function getFieldAttributeValue(?string $target_section = NULL): array {
    return $this->getAttributeValue(EntityAttributes::FIELD_ATTRIBUTES, $target_section);
  }

  /**
   * Get the field item attributes for a field or all stored values.
   *
   * @param string|null $target_section
   *   The field's $field_definition->getConfigTarget().
   *
   * @return array
   *   The stored field attribute value.
   *
   * @see setFieldAttributeValue()
   */
  public function getFieldItemAttributeValue(?string $target_section = NULL): array {
    return $this->getAttributeValue(EntityAttributes::FIELD_ITEM_ATTRIBUTES, $target_section);
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    $value = $this->get('value')->getValue();

    return $value === NULL;
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    // See /core/lib/Drupal/Core/TypedData/Plugin/DataType directory for
    // available data types.
    $properties['value'] = DataDefinition::create('string')
      ->setLabel(t('Attributes settings'));
    // ->setRequired(TRUE);
    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function generateSampleValue(FieldDefinitionInterface $field_definition): array {
    $random = new Random();
    $values['value'] = json_encode($random->word(random_int(1, 50)));
    return $values;
  }

}
