<?php

declare(strict_types=1);

namespace Drupal\Tests\eaf\Traits;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;

/**
 * Provides common test methods for attribute field tests.
 */
trait AttributeFieldTestTrait {

  /**
   * Creates an attribute field on an entity type.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $bundle
   *   The bundle.
   * @param string $field_name
   *   The field name.
   * @param array $settings
   *   Field settings.
   *
   * @return \Drupal\field\Entity\FieldConfig
   *   The created field.
   */
  protected function createAttributeField(string $entity_type, string $bundle, string $field_name = 'field_attributes', array $settings = []): FieldConfig {
    // Create field storage.
    $field_storage = FieldStorageConfig::create([
      'field_name' => $field_name,
      'entity_type' => $entity_type,
      'type' => 'field_attributes_storage',
      'cardinality' => 1,
    ]);
    $field_storage->save();

    // Create field instance.
    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => $bundle,
      'label' => 'Attributes',
      'settings' => $settings,
    ]);
    $field->save();

    return $field;
  }

  /**
   * Creates a text field for attribute testing.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $bundle
   *   The bundle.
   * @param string $field_name
   *   The field name.
   *
   * @return \Drupal\field\Entity\FieldConfig
   *   The created field.
   */
  protected function createTestTextField(string $entity_type, string $bundle, string $field_name): FieldConfig {
    $field_storage = FieldStorageConfig::create([
      'field_name' => $field_name,
      'entity_type' => $entity_type,
      'type' => 'string',
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => $bundle,
      'label' => ucfirst(str_replace('_', ' ', $field_name)),
    ]);
    $field->save();

    return $field;
  }

  /**
   * Enables specific attribute plugins for a field.
   *
   * @param \Drupal\field\Entity\FieldConfig $field
   *   The field to configure.
   * @param array $entity_plugins
   *   Entity attribute plugins to enable.
   * @param array $field_plugins
   *   Field attribute plugins to enable.
   * @param array $item_plugins
   *   Field item attribute plugins to enable.
   */
  protected function enableAttributePlugins(FieldConfig $field, array $entity_plugins, array $field_plugins = [], array $item_plugins = []): void {
    $settings = $field->getSettings();

    // Enable entity plugins.
    foreach ($entity_plugins as $plugin_id) {
      $settings['entity_attribute_plugins'][$plugin_id] = TRUE;
    }

    // Enable field plugins.
    foreach ($field_plugins as $field_name => $plugins) {
      foreach ($plugins as $plugin_id) {
        $settings['field_attribute_plugins'][$field_name][$plugin_id] = TRUE;
      }
    }

    // Enable field item plugins.
    foreach ($item_plugins as $field_name => $plugins) {
      foreach ($plugins as $plugin_id) {
        $settings['field_item_attribute_plugins'][$field_name][$plugin_id] = TRUE;
      }
    }

    $field->setSettings($settings);
    $field->save();
  }

  /**
   * Asserts attribute data structure is valid.
   *
   * @param array $data
   *   The attribute data to validate.
   */
  protected function assertValidAttributeStructure(array $data): void {
    $this->assertIsArray($data, 'Attribute data should be an array');

    // Check for expected top-level keys.
    $expected_keys = ['entity_attributes', 'field_attributes', 'field_item_attributes'];
    foreach ($expected_keys as $key) {
      if (isset($data[$key])) {
        $this->assertIsArray($data[$key], "'{$key}' should be an array");
      }
    }
  }

  /**
   * Gets attribute field value from entity.
   *
   * @param mixed $entity
   *   The entity.
   * @param string $field_name
   *   The attribute field name.
   *
   * @return array
   *   The attribute values.
   */
  protected function getAttributeFieldValue($entity, string $field_name = 'field_attributes'): array {
    if (!$entity->hasField($field_name)) {
      return [];
    }

    $value = $entity->get($field_name)->getValue();
    return $value[0]['value'] ?? [];
  }

}
