<?php

declare(strict_types=1);

namespace Drupal\Tests\eaf\Unit;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\eaf\EntityAttributePluginManagerInterface;
use Drupal\eaf\FieldAttributeService;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests FieldAttributeService functionality.
 */
#[CoversClass(FieldAttributeService::class)]
#[Group('eaf')]
final class FieldAttributeServiceTest extends UnitTestCase {

  /**
   * The field attribute service.
   *
   * @var \Drupal\eaf\FieldAttributeService
   */
  protected $service;

  /**
   * The attribute plugin manager mock.
   *
   * @var \Drupal\eaf\EntityAttributePluginManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $attributePluginManager;

  /**
   * The entity field manager mock.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityFieldManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->attributePluginManager = $this->createMock(EntityAttributePluginManagerInterface::class);
    $this->entityFieldManager = $this->createMock(EntityFieldManagerInterface::class);
    $this->service = new FieldAttributeService(
      $this->attributePluginManager,
      $this->entityFieldManager
    );
  }

  /**
   * Tests isAllowed method.
   */
  #[DataProvider('isAllowedDataProvider')]
  public function testIsAllowed(string $type, bool $is_read_only, string $field_name, bool $expected): void {
    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $field_definition->method('getType')->willReturn($type);
    $field_definition->method('isReadOnly')->willReturn($is_read_only);
    $field_definition->method('getName')->willReturn($field_name);

    $result = $this->service->isAllowed($field_definition);

    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testIsAllowed.
   */
  public static function isAllowedDataProvider(): array {
    return [
      'allowed string field' => [
        'string',
        FALSE,
        'field_test',
        TRUE,
      ],
      'allowed string_long field' => [
        'string_long',
        FALSE,
        'field_body',
        TRUE,
      ],
      'allowed entity_reference field' => [
        'entity_reference',
        FALSE,
        'field_ref',
        TRUE,
      ],
      'allowed entity_reference_revisions field' => [
        'entity_reference_revisions',
        FALSE,
        'field_paragraphs',
        TRUE,
      ],
      'allowed link field' => [
        'link',
        FALSE,
        'field_link',
        TRUE,
      ],
      'not allowed - text_with_summary' => [
        'text_with_summary',
        FALSE,
        'field_test',
        FALSE,
      ],
      'not allowed - read only' => [
        'string',
        TRUE,
        'field_test',
        FALSE,
      ],
      'not allowed - default_langcode' => [
        'string',
        FALSE,
        'default_langcode',
        FALSE,
      ],
      'not allowed - revision_default' => [
        'string',
        FALSE,
        'revision_default',
        FALSE,
      ],
      'not allowed - revision_uid' => [
        'string',
        FALSE,
        'revision_uid',
        FALSE,
      ],
      'not allowed - path' => [
        'string',
        FALSE,
        'path',
        FALSE,
      ],
      'not allowed - menu_link' => [
        'string',
        FALSE,
        'menu_link',
        FALSE,
      ],
      'not allowed - parent_id' => [
        'string',
        FALSE,
        'parent_id',
        FALSE,
      ],
      'not allowed - parent_type' => [
        'string',
        FALSE,
        'parent_type',
        FALSE,
      ],
      'not allowed - parent_field_name' => [
        'string',
        FALSE,
        'parent_field_name',
        FALSE,
      ],
    ];
  }

  /**
   * Tests getFieldAttributeSectionName method.
   */
  public function testGetFieldAttributeSectionName(): void {
    $result = $this->service->getFieldAttributeSectionName();
    $this->assertEquals('_field_attributes', $result);
  }

  /**
   * Tests getAttributeFields method.
   */
  public function testGetAttributeFields(): void {
    $entity = $this->createMock(ContentEntityInterface::class);

    $field_definition1 = $this->createMock(FieldDefinitionInterface::class);
    $field_definition1->method('getType')->willReturn('field_attributes_storage');
    $field_definition1->method('getLabel')->willReturn('Attributes');

    $field_definition2 = $this->createMock(FieldDefinitionInterface::class);
    $field_definition2->method('getType')->willReturn('string');
    $field_definition2->method('getLabel')->willReturn('Title');

    $field_definitions = [
      'field_attributes' => $field_definition1,
      'field_title' => $field_definition2,
    ];

    $entity->method('getFieldDefinitions')->willReturn($field_definitions);

    $result = $this->service->getAttributeFields($entity);

    $this->assertArrayHasKey('field_attributes', $result);
    $this->assertArrayNotHasKey('field_title', $result);
    $this->assertEquals('Attributes', $result['field_attributes']);
  }

  /**
   * Tests getAttributeFields method with empty result.
   */
  public function testGetAttributeFieldsEmpty(): void {
    $entity = $this->createMock(ContentEntityInterface::class);

    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $field_definition->method('getType')->willReturn('string');

    $field_definitions = [
      'field_title' => $field_definition,
    ];

    $entity->method('getFieldDefinitions')->willReturn($field_definitions);

    $result = $this->service->getAttributeFields($entity);

    $this->assertEmpty($result);
  }

  /**
   * Tests allowed field types constant.
   */
  public function testAllowedFieldTypesConstant(): void {
    $expected = [
      'string',
      'string_long',
      'entity_reference',
      'entity_reference_revisions',
      'link',
    ];

    $this->assertEquals($expected, FieldAttributeService::ALLOWED_FIELD_TYPES);
  }

  /**
   * Tests forbidden field names constant.
   */
  public function testForbiddenFieldNamesConstant(): void {
    $expected_forbidden = [
      'default_langcode',
      'revision_default',
      'revision_uid',
      'revision_translation_affected',
      'path',
      'menu_link',
      'parent_id',
      'parent_type',
      'parent_field_name',
    ];

    $this->assertEquals($expected_forbidden, FieldAttributeService::FORBIDDEN_FIELD_NAMES);
  }

  /**
   * Tests isAllowed method with all allowed types.
   */
  public function testIsAllowedWithAllAllowedTypes(): void {
    foreach (FieldAttributeService::ALLOWED_FIELD_TYPES as $type) {
      $field_definition = $this->createMock(FieldDefinitionInterface::class);
      $field_definition->method('getType')->willReturn($type);
      $field_definition->method('isReadOnly')->willReturn(FALSE);
      $field_definition->method('getName')->willReturn('field_test_' . $type);

      $result = $this->service->isAllowed($field_definition);

      $this->assertTrue($result, "Field type {$type} should be allowed");
    }
  }

  /**
   * Tests isAllowed method with all forbidden names.
   */
  public function testIsAllowedWithAllForbiddenNames(): void {
    foreach (FieldAttributeService::FORBIDDEN_FIELD_NAMES as $field_name) {
      $field_definition = $this->createMock(FieldDefinitionInterface::class);
      $field_definition->method('getType')->willReturn('string');
      $field_definition->method('isReadOnly')->willReturn(FALSE);
      $field_definition->method('getName')->willReturn($field_name);

      $result = $this->service->isAllowed($field_definition);

      $this->assertFalse($result, "Field name {$field_name} should be forbidden");
    }
  }

}
