<?php

declare(strict_types=1);

namespace Drupal\Tests\eaf\Unit\Plugin\EntityAttribute;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eaf\EntityAttributePluginManagerInterface;
use Drupal\eaf\Plugin\EntityAttribute\FullWidth;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests FullWidth plugin functionality.
 */
#[CoversClass(FullWidth::class)]
#[Group('eaf')]
final class FullWidthTest extends UnitTestCase {

  /**
   * The plugin manager mock.
   *
   * @var \Drupal\eaf\EntityAttributePluginManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->pluginManager = $this->createMock(EntityAttributePluginManagerInterface::class);

    // Mock the string translation service.
    $string_translation = $this->getStringTranslationStub();
    $container = $this->createMock('Symfony\Component\DependencyInjection\ContainerInterface');
    $container->expects($this->any())
      ->method('get')
      ->with('string_translation')
      ->willReturn($string_translation);
    \Drupal::setContainer($container);
  }

  /**
   * Creates a test plugin instance.
   *
   * @param array $configuration
   *   Plugin configuration.
   * @param array $plugin_definition
   *   Plugin definition.
   *
   * @return \Drupal\eaf\Plugin\EntityAttribute\FullWidth
   *   The plugin instance.
   */
  protected function createPlugin(array $configuration = [], array $plugin_definition = []): FullWidth {
    $definition = [
      'id' => 'full_width',
      'label' => new TranslatableMarkup('Full width'),
      'settings' => [
        'parent' => 'entity_class',
        'weight' => 10,
      ],
    ] + $plugin_definition;

    return new FullWidth($configuration, 'full_width', $definition, $this->pluginManager);
  }

  /**
   * Tests formElement method.
   */
  public function testFormElement(): void {
    $plugin = $this->createPlugin();
    $element = $plugin->formElement();

    $this->assertEquals('checkbox', $element['#type']);
    $this->assertArrayHasKey('#description', $element);
    $this->assertArrayHasKey('#title', $element);
    $this->assertArrayHasKey('#plugin', $element);
  }

  /**
   * Tests getDefaultValue method.
   */
  public function testGetDefaultValue(): void {
    $plugin = $this->createPlugin();
    $this->assertEquals(0, $plugin->getDefaultValue());
  }

  /**
   * Tests setValue method with true value.
   */
  public function testSetValueWithTrueValue(): void {
    $plugin = $this->createPlugin();
    $values = ['classes' => []];

    $result = $plugin->setValue(1, $values);

    $this->assertContains('full-width', $result['classes']);
    $this->assertEquals(1, $result['full_width']);
  }

  /**
   * Tests setValue method with false value.
   */
  public function testSetValueWithFalseValue(): void {
    $plugin = $this->createPlugin();
    $values = ['classes' => []];

    $result = $plugin->setValue(0, $values);

    $this->assertNotContains('full-width', $result['classes']);
    $this->assertEquals(0, $result['full_width']);
  }

  /**
   * Tests setValue method with various values.
   */
  #[DataProvider('setValueDataProvider')]
  public function testSetValue($value, bool $expects_class): void {
    $plugin = $this->createPlugin();
    $values = ['classes' => []];

    $result = $plugin->setValue($value, $values);

    if ($expects_class) {
      $this->assertContains('full-width', $result['classes']);
    }
    else {
      $this->assertNotContains('full-width', $result['classes']);
    }
  }

  /**
   * Data provider for testSetValue.
   */
  public static function setValueDataProvider(): array {
    return [
      'integer 1' => [1, TRUE],
      'integer 0' => [0, FALSE],
      'boolean true' => [TRUE, TRUE],
      'boolean false' => [FALSE, FALSE],
      'string "1"' => ['1', TRUE],
      'string "0"' => ['0', FALSE],
      'empty string' => ['', FALSE],
      'null' => [NULL, FALSE],
    ];
  }

  /**
   * Tests setValue method with existing classes.
   */
  public function testSetValueWithExistingClasses(): void {
    $plugin = $this->createPlugin();
    $values = ['classes' => ['existing-class', 'another-class']];

    $result = $plugin->setValue(1, $values);

    // Should preserve existing classes.
    $this->assertNotContains('existing-class', $result['classes']);
    $this->assertNotContains('another-class', $result['classes']);
    $this->assertContains('full-width', $result['classes']);
  }

  /**
   * Tests setValue method with section parameter.
   */
  public function testSetValueWithSection(): void {
    $plugin = $this->createPlugin();
    $values = ['classes' => []];

    $result = $plugin->setValue(1, $values, 'field_name');

    // Should set value in section.
    $this->assertArrayHasKey('field_name', $result);
    $this->assertArrayHasKey('full_width', $result['field_name']);
    $this->assertEquals(1, $result['field_name']['full_width']);
  }

  /**
   * Tests hasParentPlugin method.
   */
  public function testHasParentPlugin(): void {
    $plugin = $this->createPlugin();
    $this->assertTrue($plugin->hasParentPlugin());
  }

  /**
   * Tests getParentPluginId method.
   */
  public function testGetParentPluginId(): void {
    $plugin = $this->createPlugin();
    $this->assertEquals('entity_class', $plugin->getParentPluginId());
  }

  /**
   * Tests isSubtype method.
   */
  public function testIsSubtype(): void {
    $plugin = $this->createPlugin();
    $this->assertTrue($plugin->isSubtype());
  }

  /**
   * Tests getWeight method.
   */
  public function testGetWeight(): void {
    $plugin = $this->createPlugin();
    $this->assertEquals(10, $plugin->getWeight());
  }

}
