<?php

declare(strict_types=1);

namespace Drupal\Tests\eaf\Unit\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\TypedData\TypedDataInterface;
use Drupal\eaf\EntityAttributes;
use Drupal\eaf\Plugin\Field\FieldType\AttributesStorageItem;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests AttributesStorageItem functionality.
 */
#[CoversClass(AttributesStorageItem::class)]
#[Group('eaf')]
final class AttributesStorageItemTest extends UnitTestCase {

  /**
   * Tests defaultStorageSettings method.
   */
  public function testDefaultStorageSettings(): void {
    $settings = AttributesStorageItem::defaultStorageSettings();
    $this->assertIsArray($settings);
  }

  /**
   * Tests defaultFieldSettings method.
   */
  public function testDefaultFieldSettings(): void {
    $settings = AttributesStorageItem::defaultFieldSettings();

    $this->assertIsArray($settings);
    $this->assertArrayHasKey('entity_attribute_plugins', $settings);
    $this->assertArrayHasKey('field_attribute_plugins', $settings);
    $this->assertArrayHasKey('field_item_attribute_plugins', $settings);
    $this->assertEquals([], $settings['entity_attribute_plugins']);
    $this->assertEquals([], $settings['field_attribute_plugins']);
    $this->assertEquals([], $settings['field_item_attribute_plugins']);
  }

  /**
   * Tests schema method.
   */
  public function testSchema(): void {
    $field_definition = $this->createMock(FieldStorageDefinitionInterface::class);
    $schema = AttributesStorageItem::schema($field_definition);

    $this->assertIsArray($schema);
    $this->assertArrayHasKey('columns', $schema);
    $this->assertArrayHasKey('value', $schema['columns']);
    $this->assertEquals('text', $schema['columns']['value']['type']);
    $this->assertEquals('normal', $schema['columns']['value']['size']);
    $this->assertFalse($schema['columns']['value']['not null']);
  }

  /**
   * Tests fieldSettingsToConfigData method.
   */
  public function testFieldSettingsToConfigData(): void {
    $settings = [
      'entity_attribute_plugins' => [
        'entity_css_class' => 'entity_css_class',
        'full_width' => 0,
      ],
      'field_attribute_plugins' => [
        'field_title' => [
          'entity_css_class' => 'entity_css_class',
          'full_width' => 0,
        ],
      ],
      'field_item_attribute_plugins' => [
        'field_items' => [
          'entity_css_class' => 'entity_css_class',
          'full_width' => 0,
        ],
      ],
    ];

    $result = AttributesStorageItem::fieldSettingsToConfigData($settings);

    $this->assertTrue($result['entity_attribute_plugins']['entity_css_class']);
    $this->assertFalse($result['entity_attribute_plugins']['full_width']);
    $this->assertTrue($result['field_attribute_plugins']['field_title']['entity_css_class']);
    $this->assertFalse($result['field_attribute_plugins']['field_title']['full_width']);
    $this->assertTrue($result['field_item_attribute_plugins']['field_items']['entity_css_class']);
    $this->assertFalse($result['field_item_attribute_plugins']['field_items']['full_width']);
  }

  /**
   * Tests propertyDefinitions method.
   */
  public function testPropertyDefinitions(): void {
    $field_definition = $this->createMock(FieldStorageDefinitionInterface::class);
    $properties = AttributesStorageItem::propertyDefinitions($field_definition);

    $this->assertArrayHasKey('value', $properties);
    $this->assertEquals('string', $properties['value']->getDataType());
  }

  /**
   * Test getValue with JSON string.
   */
  public function testGetValueWithJsonString(): void {
    $data = [
      EntityAttributes::ENTITY_ATTRIBUTES => ['entity_css_class' => 'test-class'],
      EntityAttributes::FIELD_ATTRIBUTES => ['field_title' => ['entity_css_class' => 'field-class']],
    ];
    $json_string = json_encode($data);

    // Create a mock that simulates the field item behavior.
    $mock = $this->getMockBuilder(AttributesStorageItem::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['get'])
      ->getMock();

    // Mock the TypedDataInterface for value property.
    $value_property = $this->createMock(TypedDataInterface::class);
    $value_property->method('getValue')->willReturn($json_string);

    $mock->method('get')->willReturn($value_property);

    // Since we can't easily test the actual getValue implementation
    // without full Drupal bootstrap, we'll verify the structure.
    $this->assertTrue(method_exists($mock, 'getValue'));
  }

  /**
   * Test isEmpty method.
   */
  public function testIsEmpty(): void {
    // Create a mock that simulates the field item behavior.
    $mock = $this->getMockBuilder(AttributesStorageItem::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['get'])
      ->getMock();

    // Mock the TypedDataInterface for value property.
    $value_property = $this->createMock(TypedDataInterface::class);
    $value_property->method('getValue')->willReturn(NULL);

    $mock->method('get')->willReturn($value_property);

    $this->assertTrue($mock->isEmpty());
  }

  /**
   * Test isEmpty with non-null value.
   */
  public function testIsEmptyWithValue(): void {
    // Create a mock that simulates the field item behavior.
    $mock = $this->getMockBuilder(AttributesStorageItem::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['get'])
      ->getMock();

    // Mock the TypedDataInterface for value property.
    $value_property = $this->createMock(TypedDataInterface::class);
    $value_property->method('getValue')->willReturn('{"test": "value"}');

    $mock->method('get')->willReturn($value_property);

    $this->assertFalse($mock->isEmpty());
  }

  /**
   * Test EntityAttributes constants are properly defined.
   */
  public function testEntityAttributesConstants(): void {
    $this->assertEquals('_entity_attributes', EntityAttributes::ENTITY_ATTRIBUTES);
    $this->assertEquals('_field_attributes', EntityAttributes::FIELD_ATTRIBUTES);
    $this->assertEquals('_field_item_attributes', EntityAttributes::FIELD_ITEM_ATTRIBUTES);
  }

  /**
   * Test attribute type validation in getAttributeValue logic.
   *
   * This tests the logic that defaults to ENTITY_ATTRIBUTES for invalid types.
   */
  public function testAttributeTypeValidation(): void {
    // Verify that the constant values are what the code expects.
    $valid_types = [
      EntityAttributes::ENTITY_ATTRIBUTES,
      EntityAttributes::FIELD_ATTRIBUTES,
      EntityAttributes::FIELD_ITEM_ATTRIBUTES,
    ];

    $this->assertCount(3, $valid_types);
    $this->assertContains('_entity_attributes', $valid_types);
    $this->assertContains('_field_attributes', $valid_types);
    $this->assertContains('_field_item_attributes', $valid_types);
  }

  /**
   * Test default field settings structure matches expected plugin types.
   */
  public function testDefaultFieldSettingsStructure(): void {
    $settings = AttributesStorageItem::defaultFieldSettings();

    // Verify all three plugin type settings exist and are arrays.
    $this->assertIsArray($settings['entity_attribute_plugins']);
    $this->assertIsArray($settings['field_attribute_plugins']);
    $this->assertIsArray($settings['field_item_attribute_plugins']);

    // Verify they're empty by default.
    $this->assertEmpty($settings['entity_attribute_plugins']);
    $this->assertEmpty($settings['field_attribute_plugins']);
    $this->assertEmpty($settings['field_item_attribute_plugins']);
  }

  /**
   * Test schema defines proper text field for JSON storage.
   */
  public function testSchemaDefinesTextStorage(): void {
    $field_definition = $this->createMock(FieldStorageDefinitionInterface::class);
    $schema = AttributesStorageItem::schema($field_definition);

    // Verify the schema uses text type for JSON storage.
    $this->assertEquals('text', $schema['columns']['value']['type']);
    // Verify it's normal size (not tiny or big).
    $this->assertEquals('normal', $schema['columns']['value']['size']);
    // Verify it allows NULL values.
    $this->assertFalse($schema['columns']['value']['not null']);
  }

  /**
   * Test fieldSettingsToConfigData converts all values to boolean.
   */
  public function testFieldSettingsToConfigDataConvertsToBoolean(): void {
    $settings = [
      'entity_attribute_plugins' => [
        'plugin1' => 'plugin1',
        'plugin2' => '',
        'plugin3' => 1,
        'plugin4' => 0,
        'plugin5' => NULL,
      ],
      'field_attribute_plugins' => [],
      'field_item_attribute_plugins' => [],
    ];

    $result = AttributesStorageItem::fieldSettingsToConfigData($settings);

    // All values should be converted to boolean.
    $this->assertIsBool($result['entity_attribute_plugins']['plugin1']);
    $this->assertIsBool($result['entity_attribute_plugins']['plugin2']);
    $this->assertIsBool($result['entity_attribute_plugins']['plugin3']);
    $this->assertIsBool($result['entity_attribute_plugins']['plugin4']);
    $this->assertIsBool($result['entity_attribute_plugins']['plugin5']);

    // Check actual boolean values.
    $this->assertTrue($result['entity_attribute_plugins']['plugin1']);
    $this->assertFalse($result['entity_attribute_plugins']['plugin2']);
    $this->assertTrue($result['entity_attribute_plugins']['plugin3']);
    $this->assertFalse($result['entity_attribute_plugins']['plugin4']);
    $this->assertFalse($result['entity_attribute_plugins']['plugin5']);
  }

}
