<?php

namespace Drupal\easy_entity_field\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Field\FieldFilteredMarkup;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;

/**
 * The form for editing EasyEntityField.
 *
 * @package Drupal\easy_entity_field\Form
 */
class EasyEntityFieldForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): ?array {
    $form = parent::form($form, $form_state);

    $form_title = $this->t('%field settings', [
      '%field' => $this->entity->getLabel(),
    ]);
    $form['#title'] = $form_title;

    /**
     * @var \Drupal\easy_entity_field\Entity\EasyEntityFieldInterface $easy_entity_field
     */
    $easy_entity_field = $this->entity;
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $easy_entity_field->label(),
      '#weight' => -20,
      '#required' => TRUE,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Help text'),
      '#default_value' => $easy_entity_field->getDescription(),
      '#rows' => 5,
      '#description' => $this->t('Instructions to present to the user below this field on the editing form.<br />Allowed HTML tags: @tags', ['@tags' => FieldFilteredMarkup::displayAllowedTags()]) . '<br />' . $this->t('This field supports tokens.'),
      '#weight' => -10,
    ];

    $form['required'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Required field'),
      '#default_value' => $easy_entity_field->isRequired(),
      '#weight' => -5,
    ];

    $form['locked'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Lock: Cannot delete'),
      '#default_value' => $easy_entity_field->isLocked(),
    ];

    // Create an arbitrary entity object (used by the 'default value' widget).
    $ids = (object) [
      'entity_type' => $easy_entity_field->getTargetEntityTypeId(),
      'bundle' => $easy_entity_field->getTargetEntityTypeId(),
      'entity_id' => NULL,
    ];

    $form['#entity'] = _field_create_entity_from_ids($ids);

    if (!$items = $form['#entity']->get($easy_entity_field->getName())) {
      return NULL;
    }
    $item = $items->first() ?: $items->appendItem();

    /**
     * @var \Drupal\easy_entity_field\Plugin\EasyEntityFieldBase $field_plugin
     */
    $field_plugin = $easy_entity_field->hasFieldPlugin() ? $easy_entity_field->getFieldPlugin() : '';

    $field_settings_element = '';
    if (!empty($field_plugin) && !empty($field_plugin->fieldSettingsForm($form, $form_state))) {
      $field_settings_element = $field_plugin->fieldSettingsForm($form, $form_state);
    }
    elseif (!empty($item->fieldSettingsForm($form, $form_state))) {
      $field_settings_element = $item->fieldSettingsForm($form, $form_state);
    }

    if ($field_settings_element) {
      $field_settings_element = array_merge($field_settings_element, [
        '#type' => 'details',
        '#title' => $this->t('settings'),
        '#open' => TRUE,
        '#tree' => TRUE,
        '#description' => $this->t('Enable to add a path field and allow to define alias patterns for the given type. Disabled types already define a path field themselves or currently have a pattern.'),
        '#weight' => 12,
        '#attributes' => ['data-drupal-selector' => 'edit-settings'],
      ]);

      $form['settings'] = $field_settings_element;
    }

    if ($default_values_element = $items->defaultValuesForm($form, $form_state)) {
      $has_required = $this->hasAnyRequired($default_values_element);
      $default_values_element = array_merge($default_values_element, [
        '#type' => 'details',
        '#title' => $this->t('Default value'),
        '#open' => TRUE,
        '#tree' => TRUE,
        '#description' => $this->t('The default value for this field, used when creating new content.'),
        '#weight' => 20,
      ]);

      if (!$has_required) {
        $has_default_value = count($this->entity->getDefaultValue($form['#entity'])) > 0;
        $default_values_element['#states'] = [
          'invisible' => [
            ':input[name="set_default_value"]' => ['checked' => FALSE],
          ],
        ];
        $form['set_default_value'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Set default value'),
          '#default_value' => $has_default_value,
          '#description' => $this->t('Provide a pre-filled value for the editing form.'),
          '#weight' => $element['#weight'] ?? 15,
        ];
      }
      $form['default_value'] = $default_values_element;
    }

    $form['#prefix'] = '<div id="field-combined">';
    $form['#suffix'] = '</div>';
    $form['#attached']['library'][] = 'field_ui/drupal.field_ui';

    return $form;
  }

  /**
   * A function to check if element contains any required elements.
   *
   * @param array $element
   *   An element to check.
   *
   * @return bool
   *   TRUE if element contains any required elements, FALSE otherwise.
   */
  private function hasAnyRequired(array $element) {
    $has_required = FALSE;
    foreach (Element::children($element) as $child) {
      if (isset($element[$child]['#required']) && $element[$child]['#required']) {
        $has_required = TRUE;
        break;
      }
      if (Element::children($element[$child])) {
        return $this->hasAnyRequired($element[$child]);
      }
    }

    return $has_required;
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state): array {
    $actions = parent::actions($form, $form_state);
    $actions['submit']['#value'] = $this->t('Save settings');

    if (!$this->entity->isNew()) {
      $actions['delete'] = [
        '#type' => 'link',
        '#title' => $this->t('Delete'),
        '#url' => $this->entity->toUrl('delete-form'),
        '#access' => $this->entity->access('delete'),
        '#attributes' => [
          'class' => ['button', 'button--danger'],
        ],
      ];
    }

    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
    if (isset($form['default_value'])) {
      $item = $form['#entity']->get($this->entity->getName());
      $item->defaultValuesFormValidate($form['default_value'], $form, $form_state);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);
    if (isset($form['settings']) && $form_state->getValue('settings')) {
      $settings = $form_state->getValue('settings');

      // Process the field settings to ensure they are properly formatted
      // This is especially important for dynamic_entity_reference fields
      $easy_entity_field_plugin = $this->entity->getFieldPlugin();
      if ($easy_entity_field_plugin && method_exists($easy_entity_field_plugin, 'processFieldSettings')) {
        $settings = $easy_entity_field_plugin->processFieldSettings($settings);
      }

      $this->entity->setFieldSettings($settings);
    }

    // Handle the default value.
    $default_value = [];
    if (isset($form['default_value']) && (!isset($form['set_default_value']) || $form_state->getValue('set_default_value'))) {
      $items = $form['#entity']->get($this->entity->getName());
      $default_value = $items->defaultValuesFormSubmit($form['default_value'], $form, $form_state);
    }

    $this->entity->setDefaultValue($default_value);
  }

  /**
   * AJAX callback for field settings changes.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  public function settingsAjaxCallback(array $form, FormStateInterface $form_state): AjaxResponse {
    // Create a new AJAX response.
    $response = new AjaxResponse();

    // Get the triggering element to determine which wrapper to update.
    $triggering_element = $form_state->getTriggeringElement();

    // For regular EntityReference fields.
    if (isset($form['settings']['handler']['handler_settings'])) {
      $response->addCommand(new ReplaceCommand('[data-drupal-selector="edit-settings-handler-settings"]', $form['settings']['handler']['handler_settings']));
    }
    // For DynamicEntityReference fields - need to determine which entity type.
    elseif ($triggering_element && isset($triggering_element['#name'])) {
      // Extract entity type from element name like "settings[quiz][handler]".
      if (preg_match('/^settings\[([^\]]+)\]\[handler\]$/', $triggering_element['#name'], $matches)) {
        $entity_type = $matches[1];
        $wrapper_selector = '[data-drupal-selector="edit-settings-' . $entity_type . '-handler-settings"]';

        if (isset($form['settings'][$entity_type]['handler']['handler_settings'])) {
          $response->addCommand(new ReplaceCommand($wrapper_selector, $form['settings'][$entity_type]['handler']['handler_settings']));
        }
        else {
          // Show debug message for dynamic entity reference.
          $response->addCommand(new HtmlCommand($wrapper_selector, '<div style="color: red;">AJAX triggered for ' . $entity_type . ' but no handler_settings found</div>'));
        }
      }
    }
    else {
      // Fallback debug message.
      $response->addCommand(new HtmlCommand('body', '<div style="position: fixed; top: 0; left: 0; background: red; color: white; padding: 10px; z-index: 9999;">AJAX triggered but could not determine target</div>'));
    }

    return $response;
  }

  /**
   * Submit handler for field settings AJAX changes.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function settingsAjaxSubmit(array $form, FormStateInterface $form_state): void {
    // Get the triggering element to find the new handler value.
    $triggering_element = $form_state->getTriggeringElement();
    $form_object = $form_state->getFormObject();
    $entity = $form_object->getEntity();

    if ($triggering_element && isset($triggering_element['#name'])) {
      $current_settings = $entity->getSettings();

      // Handle regular EntityReference: settings[handler]
      if ($triggering_element['#name'] === 'settings[handler]') {
        $current_settings['handler'] = $triggering_element['#value'];
        $current_settings['handler_settings'] = [];
        $entity->setFieldSettings($current_settings);
      }
      // Handle DynamicEntityReference: settings[entity_type][handler]
      elseif (preg_match('/^settings\[([^\]]+)\]\[handler\]$/', $triggering_element['#name'], $matches)) {
        $entity_type = $matches[1];
        if (!isset($current_settings[$entity_type])) {
          $current_settings[$entity_type] = [];
        }
        $current_settings[$entity_type]['handler'] = $triggering_element['#value'];
        $current_settings[$entity_type]['handler_settings'] = [];
        $entity->setFieldSettings($current_settings);
      }
    }

    // Also check user input as fallback.
    $user_input = $form_state->getUserInput();
    if (isset($user_input['settings'])) {
      $current_settings = $entity->getSettings();

      // Handle regular EntityReference.
      if (isset($user_input['settings']['handler']) && is_string($user_input['settings']['handler'])) {
        $current_settings['handler'] = $user_input['settings']['handler'];
        $current_settings['handler_settings'] = [];
        $entity->setFieldSettings($current_settings);
      }
      // Handle DynamicEntityReference.
      else {
        foreach ($user_input['settings'] as $entity_type => $settings) {
          if (isset($settings['handler']) && is_string($settings['handler'])) {
            if (!isset($current_settings[$entity_type])) {
              $current_settings[$entity_type] = [];
            }
            $current_settings[$entity_type]['handler'] = $settings['handler'];
            $current_settings[$entity_type]['handler_settings'] = [];
          }
        }
        $entity->setFieldSettings($current_settings);
      }
    }

    // Clear any cached form storage data to ensure proper rebuild.
    $form_storage = &$form_state->getStorage();
    unset($form_storage['default_value_widget']);

    // Mark the form for rebuild.
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): void {
    $easy_entity_field = $this->entity;
    $easy_entity_field->save();

    // Trigger generic storage settings save hook for all modules to respond.
    \Drupal::moduleHandler()->invokeAll('easy_entity_field_storage_save', [$easy_entity_field]);

    // Also trigger field type specific hook.
    \Drupal::moduleHandler()->invokeAll('easy_entity_field_storage_save_' . $easy_entity_field->getFieldType(), [$easy_entity_field]);

    $this->messenger()->addStatus($this->t('Saved %label configuration.', ['%label' => $this->entity->getLabel()]));
    $form_state->setRedirectUrl($easy_entity_field->toUrl('collection'));
  }

}
