<?php

namespace Drupal\easy_entity_field\Hook;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\easy_entity_field\Entity\EasyEntityField;

/**
 * Hook implementations for Easy Entity Field module.
 */
class EasyEntityFieldHook {

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help($route_name, RouteMatchInterface $route_match) {
    switch ($route_name) {
      case 'help.page.easy_entity_field':
        $output = '';
        $output .= '<h3>' . t('About') . '</h3>';
        $output .= '<p>' . t('The Easy entity base field module can add basic fields to the data table data_table of all content entities, such as: Node, User, block_content, etc. The added fields can be managed using form display and display which are build-in drupal core by default.');
        return $output;
    }
  }

    /**
   * Implements hook_easy_entity_field_remove_entity_type().
   */
  #[Hook('easy_entity_field_remove_entity_type')]
  public function easyEntityFieldRemoveEntityType(array $entity_types): array {
    // Add entity types that should be excluded (mark as true to exclude)
    $entity_types['group_relationship'] = true;
    $entity_types['group_config_wrapper'] = true;

    return $entity_types;
  }

  /**
   * Implements hook_entity_base_field_info().
   */
  #[Hook('entity_base_field_info')]
  public function entityBaseFieldInfo(EntityTypeInterface $entity_type): array {
    $fields = [];

    foreach (EasyEntityField::loadMultiple() as $easy_entity_field) {
      if (!empty($easy_entity_field->isTargetEntityType()) && $easy_entity_field->getTargetEntityTypeId() === $entity_type->id()) {
        $fields[$easy_entity_field->getName()] = $easy_entity_field->buildBaseFieldDefinitions();
      }
    }

    return $fields;
  }

  /**
   * Implements hook_ENTITY_TYPE_access().
   */
  #[Hook('easy_entity_field_access')]
  public function easyEntityFieldAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    if ($operation === 'delete' && !empty($entity->isLocked())) {
      return AccessResult::forbidden()->addCacheableDependency($entity);
    }

    return AccessResult::neutral();
  }

  /**
   * Implements hook_ENTITY_TYPE_presave().
   */
  #[Hook('easy_entity_field_presave')]
  public function easyEntityFieldPresave(EntityInterface $entity) {
    /** @var \Drupal\easy_entity_field\Entity\EasyEntityFieldInterface $entity */
    if ($entity->getFieldType() === 'dynamic_entity_reference') {
      $field_settings = $entity->getFieldSettings();
      $storage_settings = $entity->getStorageSettings();
      $changed = FALSE;

      // Get target types from storage settings
      if (isset($storage_settings['entity_type_ids']) && is_array($storage_settings['entity_type_ids'])) {
        foreach ($storage_settings['entity_type_ids'] as $target_type) {
          // Ensure each target type has proper field settings
          if (!isset($field_settings[$target_type])) {
            $field_settings[$target_type] = [
              'handler' => 'default:' . $target_type,
              'handler_settings' => [],
            ];
            $changed = TRUE;
          } else {
            // Ensure handler_settings is always an array
            if (!isset($field_settings[$target_type]['handler_settings']) || !is_array($field_settings[$target_type]['handler_settings'])) {
              $field_settings[$target_type]['handler_settings'] = [];
              $changed = TRUE;
            }
            // Ensure handler is set
            if (!isset($field_settings[$target_type]['handler'])) {
              $field_settings[$target_type]['handler'] = 'default:' . $target_type;
              $changed = TRUE;
            }
          }
        }
      }

      if ($changed) {
        $entity->setFieldSettings($field_settings);
      }
    }
  }

}
