<?php

namespace Drupal\easy_entity_field\Plugin\EasyEntityField;

use Drupal\Component\Utility\Html;
use Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItem;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\easy_entity_field\Attribute\EasyEntityField;
use Drupal\easy_entity_field\Plugin\EasyEntityFieldBase;

/**
 * Defines the 'entity_reference' entity field type.
 */

#[EasyEntityField(
  id: 'entity_reference',
  label: new TranslatableMarkup('Entity reference'),
  description: new TranslatableMarkup('An entity field containing an entity reference.'),
)]

class EntityReference extends EasyEntityFieldBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state): array {
    /**
     * @var \Drupal\easy_entity_field\Entity\EasyEntityFieldInterface $easy_entity_field
     */
    $easy_entity_field = $form_state->getFormObject()->getEntity();
    // Get the entity reference selection handler options.
    $manager = \Drupal::service('plugin.manager.entity_reference_selection');

    // Get all selection plugins for this entity type.
    $selection_plugins = $manager->getSelectionGroups($easy_entity_field->getSetting('target_type'));
    $handlers_options = [];
    foreach (array_keys($selection_plugins) as $selection_group_id) {
      // We only display base plugins (e.g. 'default', 'views', ...) and not
      // entity type specific plugins (e.g. 'default:node', 'default:user',
      // ...).
      if (array_key_exists($selection_group_id, $selection_plugins[$selection_group_id])) {
        $handlers_options[$selection_group_id] = Html::escape($selection_plugins[$selection_group_id][$selection_group_id]['label']);
      }
      elseif (array_key_exists($selection_group_id . ':' . $easy_entity_field->getSetting('target_type'), $selection_plugins[$selection_group_id])) {
        $selection_group_plugin = $selection_group_id . ':' . $easy_entity_field->getSetting('target_type');
        $handlers_options[$selection_group_plugin] = Html::escape($selection_plugins[$selection_group_id][$selection_group_plugin]['base_plugin_label']);
      }
    }

    // Calculate the correct handler value BEFORE building form elements
    $handler_value = $easy_entity_field->getSetting('handler');

    // If no handler is set (new field), use the first available option
    if (empty($handler_value) && !empty($handlers_options)) {
      $handler_value = array_key_first($handlers_options);
    }

    // Check if we have a triggering element (AJAX request).
    $triggering_element = $form_state->getTriggeringElement();
    if ($triggering_element && isset($triggering_element['#name']) && $triggering_element['#name'] === 'settings[handler][handler]') {
      $handler_value = $triggering_element['#value'];
    }

    // Also check the user input directly for AJAX updates.
    $user_input = $form_state->getUserInput();
    if (isset($user_input['settings']['handler']['handler'])) {
      $handler_value = $user_input['settings']['handler']['handler'];
    }

    // Also check form values as a fallback.
    $values = $form_state->getValues();
    if (isset($values['settings']['handler']['handler'])) {
      $handler_value = $values['settings']['handler']['handler'];
    }

    // Ensure the handler_value is valid (exists in options)
    if (!isset($handlers_options[$handler_value])) {
      $handler_value = array_key_first($handlers_options);
    }

    $form = [
      '#type' => 'container',
      '#process' => [[EntityReferenceItem::class, 'fieldSettingsAjaxProcess']],
      '#element_validate' => [[static::class, 'fieldSettingsFormValidate']],
    ];

    $form['handler'] = [
      '#type' => 'details',
      '#title' => $this->t('Reference type'),
      '#open' => TRUE,
      '#tree' => TRUE,
      '#process' => [[EntityReferenceItem::class, 'formProcessMergeParent']],
    ];

    $form['handler']['handler'] = [
      '#type' => 'select',
      '#title' => $this->t('Reference method'),
      '#options' => $handlers_options,
      '#default_value' => $handler_value,
      '#required' => TRUE,
      '#ajax' => [
        'callback' => ['Drupal\easy_entity_field\Form\EasyEntityFieldForm', 'settingsAjaxCallback'],
        'wrapper' => 'edit-settings-handler-settings',
        'trigger_as' => ['name' => 'handler_settings_submit'],
      ],
      '#limit_validation_errors' => [],
    ];

    $form['handler']['handler_submit'] = [
      '#type' => 'submit',
      '#name' => 'handler_settings_submit',
      '#value' => $this->t('Change handler'),
      '#limit_validation_errors' => [],
      '#attributes' => [
        'class' => ['js-hide'],
      ],
      '#submit' => [['Drupal\easy_entity_field\Form\EasyEntityFieldForm', 'settingsAjaxSubmit']],
      '#ajax' => [
        'callback' => ['Drupal\easy_entity_field\Form\EasyEntityFieldForm', 'settingsAjaxCallback'],
        'wrapper' => 'edit-settings-handler-settings',
      ],
    ];

    $form['handler']['handler_settings'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['entity_reference-settings']],
    ];



    // Ensure handler_settings is an array and not null
    $handler_settings = $easy_entity_field->getSetting('handler_settings') ?: [];
    if (!is_array($handler_settings)) {
      $handler_settings = [];
    }

    $options = $handler_settings + [
      'target_type' => $easy_entity_field->getSetting('target_type'),
      'handler' => $handler_value,
      'entity' => $easy_entity_field,
    ];

    $selection_handler = $manager->getInstance($options);
    if(!empty($selection_handler->buildConfigurationForm([], $form_state))){
      $form['handler']['handler_settings'] += $selection_handler->buildConfigurationForm([], $form_state);
    }

    return $form;
  }

  /**
   * Form element validation handler; Invokes selection plugin's validation.
   *
   * @param array $form
   *   The form where the settings form is being included in.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state of the (entire) configuration form.
   */
  public static function fieldSettingsFormValidate(array $form, FormStateInterface $form_state): void {
    /**
     * @var \Drupal\easy_entity_field\Entity\EasyEntityFieldInterface $easy_entity_field
     */
    $easy_entity_field = $form_state->getFormObject()->getEntity();
    $manager = \Drupal::service('plugin.manager.entity_reference_selection');

    // Ensure handler_settings is an array and not null
    $handler_settings = $easy_entity_field->getSetting('handler_settings') ?: [];
    if (!is_array($handler_settings)) {
      $handler_settings = [];
    }

    $options = $handler_settings + [
      'target_type' => $easy_entity_field->getSetting('target_type'),
      'handler' => $easy_entity_field->getSetting('handler'),
      'entity' => NULL,
    ];

    $selection_handler = $manager->getInstance($options);
    $selection_handler->validateConfigurationForm($form, $form_state);
  }

}
