# Drush Commands Reference

Complete reference for Entity Builder's Drush commands.

## Import Commands

### eb:import

Import a YAML definition file.

```bash
# Import as definition only (default)
drush eb:import path/to/definition.yml

# Import and execute immediately
drush eb:import path/to/definition.yml --execute

# Overwrite existing definition
drush eb:import path/to/definition.yml --force

# Import with full preview
drush eb:import path/to/definition.yml --execute --full
```

**Options:**

| Option | Description |
|--------|-------------|
| `--execute` | Execute operations immediately after import |
| `--force` | Overwrite existing definition with same ID |
| `--full` | Show detailed preview output |

**Output:**
```
Importing: blog-architecture.yml
Format: Flat YAML v5.0
Mode: sync

Definition ID: blog_architecture
Label: Blog Architecture
Bundles: 2
Fields: 4

Definition "blog_architecture" saved successfully.
To apply this definition, run:
  drush eb:apply blog_architecture
```

### eb:apply

Apply a saved definition.

```bash
drush eb:apply <definition-id>

# Example
drush eb:apply blog_architecture
```

**Output:**
```
Applying definition: blog_architecture

Preview of 6 operations:
1. create_bundle taxonomy_term:blog_category
2. create_bundle node:blog_post
3. create_field node:blog_post:field_category
4. create_field node:blog_post:field_image
5. configure_form_mode node:blog_post:field_category
6. configure_form_mode node:blog_post:field_image

Proceed? (y/n) y

Executing...
✓ Created bundle: Blog Categories
✓ Created bundle: Blog Post
✓ Created field: Category
✓ Created field: Featured Image
✓ Configured widget: Category
✓ Configured widget: Featured Image

All 6 operations completed successfully.
Rollback ID: 42
```

## Validation Commands

### eb:validate

Validate a YAML definition file.

```bash
drush eb:validate path/to/definition.yml
```

**Output (success):**
```
Validating: blog-architecture.yml
Format: Flat YAML v5.0
Mode: sync

Validation successful!
  Bundles: 2
  Fields: 4
  Field Groups: 3
  Operations: 12
```

**Output (errors):**
```
Validating: blog-architecture.yml

Validation failed with 2 errors:

1. [field_type_not_found] Field type "invalid_type" does not exist.
   Field: field_type
   Context: field_definitions[2]

2. [bundle_not_found] Bundle "missing_bundle" does not exist.
   Field: bundle
   Context: field_definitions[3]
```

### eb:preview

Preview what operations would be executed.

```bash
drush eb:preview path/to/definition.yml
```

**Output:**
```
Preview of 8 operations:

#   Operation            Target                              Description
--------------------------------------------------------------------------------
1   create_bundle        taxonomy_term:blog_category         Create Blog Categories
2   create_bundle        node:blog_post                      Create Blog Post
3   create_field         node:blog_post:field_category       Create Category field
4   create_field         node:blog_post:field_image          Create Featured Image
5   configure_form_mode  node:blog_post:field_category       Configure widget
6   configure_form_mode  node:blog_post:field_image          Configure widget
7   configure_view_mode  node:blog_post:field_category       Configure formatter
8   configure_view_mode  node:blog_post:field_image          Configure formatter

Dependency Analysis:
  field_category depends on:
    - bundle:taxonomy_term:blog_category
    - bundle:node:blog_post
```

## Export Commands

### eb:export

Export a definition to YAML file.

```bash
drush eb:export <definition-id> <output-file>

# Example
drush eb:export blog_architecture blog-export.yml
```

### eb:generate

Generate a definition from existing Drupal entities.

```bash
# Generate from single bundle
drush eb:generate my_definition --entity-type=node --bundle=article

# Generate with output file
drush eb:generate my_definition --entity-type=node --bundle=article --output=article.yml

# Generate from multiple bundles
drush eb:generate my_definition --entity-type=node --bundle=article,page
```

**Options:**

| Option | Description |
|--------|-------------|
| `--entity-type` | Entity type to export (node, taxonomy_term, etc.) |
| `--bundle` | Bundle(s) to export (comma-separated) |
| `--output` | Output file path |

## Rollback Commands

### eb:rollback

Execute a rollback by ID.

```bash
drush eb:rollback <rollback-id>

# Example
drush eb:rollback 42
```

### eb:rollback-definition

Rollback all pending rollbacks for a definition.

```bash
drush eb:rollback-definition <definition-id>

# Example
drush eb:rollback-definition blog_architecture
```

### eb:list-rollbacks

List available rollbacks.

```bash
# List all rollbacks
drush eb:list-rollbacks

# Filter by definition
drush eb:list-rollbacks --definition=blog_architecture

# Filter by status
drush eb:list-rollbacks --status=pending
```

**Output:**
```
ID    Definition           Status     Operations  Created
----  -------------------  ---------  ----------  -------------------
45    blog_architecture    pending    6           2024-01-15 10:30:00
44    ecommerce_setup      completed  12          2024-01-14 15:45:00
43    blog_architecture    completed  4           2024-01-13 09:00:00
```

### eb:purge-rollbacks

Purge old rollback data.

```bash
# Purge rollbacks older than 30 days (default)
drush eb:purge-rollbacks

# Purge rollbacks older than 7 days
drush eb:purge-rollbacks 7
```

## Discovery Commands

### eb:discovery

Show available field types, widgets, and formatters.

```bash
drush eb:discovery
```

**Output:**
```
Field Types:
  string          - Text (plain)
  text_long       - Text (formatted, long)
  integer         - Number (integer)
  decimal         - Number (decimal)
  boolean         - Boolean
  ...

Widgets:
  string_textfield        - Textfield (for string)
  text_textarea           - Text area (for text_long)
  number                  - Number field (for integer, decimal)
  ...

Formatters:
  string                  - Plain text (for string)
  text_default            - Default (for text_long)
  number_integer          - Default (for integer)
  ...
```

### eb:discovery:schema

Output full schema for AI assistants.

```bash
# Full schema
drush eb:discovery:schema

# Specific section
drush eb:discovery:schema --section=field_types
drush eb:discovery:schema --section=widgets
drush eb:discovery:schema --section=formatters
```

## List Commands

### eb:list

List all definitions.

```bash
drush eb:list
```

**Output:**
```
ID                    Label                  Status   Bundles  Fields
--------------------  ---------------------  -------  -------  ------
blog_architecture     Blog Architecture      applied  2        4
ecommerce_setup       E-commerce Setup       draft    5        15
event_management      Event Management       applied  3        8
```

## Common Workflows

### Development Workflow

```bash
# 1. Create and validate your YAML
drush eb:validate my-architecture.yml

# 2. Preview changes
drush eb:preview my-architecture.yml

# 3. Import and execute
drush eb:import my-architecture.yml --execute

# 4. If something goes wrong, rollback
drush eb:list-rollbacks --definition=my_architecture
drush eb:rollback 123
```

### CI/CD Workflow

```bash
# In your deployment script
drush eb:import config/entity-builder/site-architecture.yml --execute --force
```

### Export for Sharing

```bash
# Generate from existing site
drush eb:generate my_site --entity-type=node --bundle=article,page,event

# Export to file
drush eb:export my_site my-site-architecture.yml
```

### Cleanup Old Rollbacks

```bash
# Weekly cleanup cron job
drush eb:purge-rollbacks 14
```
