# Troubleshooting Guide

Common issues and solutions when working with Entity Builder.

## Installation Issues

### Module Dependencies Not Found

**Error:** `Unable to install Entity Builder due to missing dependencies`

**Solution:**

```bash
# Install all dependencies via Composer
composer require drupal/eb drupal/field_group drupal/pathauto drupal/auto_entitylabel

# For AG-Grid (requires Asset Packagist configuration)
composer require npm-asset/ag-grid-community:33.0.0

# Enable the modules
drush en eb eb_ui eb_field_group eb_pathauto eb_auto_entitylabel -y
drush cr
```

### AG-Grid Library Not Loading

**Symptoms:**

- Grid interface doesn't appear
- JavaScript errors in browser console
- 404 errors for AG-Grid files

**Solutions:**

1. **Verify Asset Packagist configuration:**

```json
{
  "repositories": [
    {
      "type": "composer",
      "url": "https://asset-packagist.org"
    }
  ],
  "extra": {
    "installer-types": ["npm-asset", "bower-asset"],
    "installer-paths": {
      "web/libraries/{$name}": [
        "type:drupal-library",
        "type:npm-asset",
        "type:bower-asset"
      ]
    }
  }
}
```

2. **Install the composer-installers-extender:**

```bash
composer require oomphinc/composer-installers-extender
```

3. **Install AG-Grid library:**

```bash
composer require npm-asset/ag-grid-community:33.0.0
```

4. **Verify installation:**

```bash
ls web/libraries/ag-grid-community
```

5. **Clear caches:**

```bash
drush cr
```

### Service Container Errors

**Error:** `Service "eb.operation_processor" not found`

**Solution:**

```bash
# Rebuild the service container
drush cr

# Verify module status
drush pm:list --filter=eb

# If needed, reinstall the module
drush pmu eb -y
drush en eb -y
drush cr
```

## Validation Errors

### Field Type Not Found

**Error:** `Field type "invalid_type" does not exist.`

**Cause:** Misspelled field type or missing module.

**Solution:**

```bash
# List available field types
drush eb:discovery

# Common field types:
# string, text_long, text_with_summary, integer, decimal, boolean,
# datetime, entity_reference, image, file, link, email,
# list_string, list_integer
```

### Widget/Formatter Incompatible

**Error:** `Widget "string_textfield" is not compatible with field type "text_long".`

**Solution:** Use compatible widgets:

| Field Type | Compatible Widgets |
|------------|-------------------|
| `string` | `string_textfield` |
| `text_long` | `text_textarea` |
| `text_with_summary` | `text_textarea_with_summary` |
| `entity_reference` | `entity_reference_autocomplete`, `options_select` |
| `boolean` | `boolean_checkbox`, `options_buttons` |

### Bundle Not Found

**Error:** `Bundle "article" not found for entity type "node".`

**Solutions:**

1. **Create the bundle in the same definition:**

```yaml
bundle_definitions:
  - entity_type: node
    bundle_id: article
    label: Article

field_definitions:
  - entity_type: node
    bundle: article  # Now valid
    field_name: field_body
```

2. **Or check that the bundle exists:**

```bash
drush pm:list --filter=node
# Verify content types exist in UI
```

### Circular Dependency Detected

**Error:** `Circular dependency detected: A > B > A`

**Cause:** Operations depend on each other in a cycle.

**Solution:** Review entity reference fields and ensure they don't create cycles.

### Naming Convention Errors

**Errors:**

- `Field name must start with "field_".`
- `Group name must start with "group_".`

**Solution:**

```yaml
# Correct naming
field_name: field_body      # NOT "body"
group_name: group_content   # NOT "content"
bundle_id: my_bundle        # lowercase, underscores
```

## Execution Errors

### Permission Denied

**Error:** `You do not have permission to apply entity definitions.`

**Solution:**

```bash
drush role:perm:add administrator "administer entity builder"
drush role:perm:add administrator "apply entity definitions"
```

### Bundle Already Exists

**Error in `create_only` mode:** `Bundle "article" already exists.`

**Solutions:**

1. **Use sync mode (default):**

```yaml
mode: sync  # Updates existing entities
```

2. **Or delete the existing bundle first:**

```bash
drush entity:delete node_type article
```

### Field Storage Conflict

**Error:** `Field storage for "field_body" already exists with different settings.`

**Cause:** Field storage with same name exists but has different configuration.

**Solutions:**

1. **Use a different field name:**

```yaml
field_name: field_body_v2
```

2. **Or match the existing storage settings:**

```yaml
field_storage_settings:
  # Must match existing storage
```

## UI Issues

### Grid Not Displaying Data

**Symptoms:**

- Grid appears empty
- Data not loading

**Solutions:**

1. Check browser console for JavaScript errors
2. Verify CSRF token is valid (logout and login)
3. Clear Drupal cache: `drush cr`
4. Check for AJAX errors in network tab

### Validation Indicators Not Showing

**Symptoms:**

- Red highlighting not appearing
- Error messages not visible

**Solutions:**

1. Ensure JavaScript is enabled
2. Check browser console for errors
3. Verify the correct theme is loaded
4. Clear browser cache

### Settings Modal Not Opening

**Solutions:**

1. Ensure field type is selected first
2. For entity reference, select target type
3. Check browser console for errors
4. Verify AJAX endpoints are accessible

## Rollback Issues

### Rollback Not Found

**Error:** `Rollback with ID 42 not found.`

**Causes:**

- Rollback was already executed
- Rollback was purged due to retention policy
- Apply failed before creating rollback

**Solutions:**

```bash
# List available rollbacks
drush eb:list-rollbacks --definition=my_definition

# Check rollback status
drush eb:list-rollbacks --status=pending
```

### Cannot Rollback - Content Exists

**Error:** `Cannot delete bundle - content exists.`

**Solution:** Delete content first, then rollback:

```bash
# Delete all content of a type
drush entity:delete node --bundle=article

# Then rollback
drush eb:rollback 42
```

### Partial Rollback Failure

**Symptoms:**

- Some operations rolled back successfully
- Others failed

**Solution:** Manually fix failed items and mark rollback complete.

## Performance Issues

### Slow Definition Apply

**Symptoms:**

- Apply takes a long time
- Timeouts on large definitions

**Solutions:**

1. **Split large definitions:**

```yaml
# Create separate definitions for each bundle group
# e.g., blog_definitions.yml, commerce_definitions.yml
```

2. **Use CLI instead of UI:**

```bash
drush eb:import large-definition.yml --execute
```

3. **Increase PHP limits:**

```php
# settings.php
ini_set('max_execution_time', 300);
ini_set('memory_limit', '512M');
```

### High Memory Usage

**Solutions:**

1. Process definitions in smaller batches
2. Increase PHP memory limit
3. Use Drush with increased memory:

```bash
php -d memory_limit=512M vendor/bin/drush eb:apply my_definition
```

## Debug Mode

Enable debug mode for detailed logging:

**Via Settings:**

Navigate to `/admin/config/development/eb/settings` and enable Debug Mode.

**Via Config:**

```yaml
# In settings.local.php
$config['eb.settings']['debug_mode'] = TRUE;
```

**Debug output shows:**

- Dependency resolution steps
- Change detection results
- Validation details
- Operation data before/after

## Getting Help

### Check Logs

```bash
# View recent Entity Builder logs
drush watchdog:show --filter=eb --count=50

# View with full details
drush watchdog:show --filter=eb --extended
```

### Validate Definition

```bash
# Validate before applying
drush eb:validate path/to/definition.yml

# Preview operations
drush eb:preview path/to/definition.yml
```

### Discover Available Plugins

```bash
# List field types, widgets, formatters
drush eb:discovery

# Full schema for reference
drush eb:discovery:schema
```

### Report Issues

1. Check the [issue queue](https://www.drupal.org/project/issues/eb)
2. Include:
   - Drupal and PHP versions
   - Entity Builder version
   - Steps to reproduce
   - Error messages
   - Relevant YAML definitions
