/**
 * @file
 * EB admin listing behaviors.
 */

(function (Drupal, once) {

  'use strict';

  /**
   * Filter tables by text input.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.ebTableFilter = {
    attach(context, settings) {
      const inputs = once(
        'eb-table-filter',
        'input.eb-definition-filter-text, input.eb-discovery-filter-text',
        context
      );

      inputs.forEach((input) => {
        const tableSelector = input.getAttribute('data-table');
        const table = document.querySelector(tableSelector);

        if (!table) {
          return;
        }

        const rows = table.querySelectorAll('tbody tr');
        let debounceTimer;

        input.addEventListener('input', (e) => {
          clearTimeout(debounceTimer);
          debounceTimer = setTimeout(() => {
            filterTable(e.target.value, rows);
          }, 200);
        });
      });

      /**
       * Filter table rows based on query.
       *
       * @param {string} query - The search query.
       * @param {NodeList} rows - The table rows to filter.
       */
      function filterTable(query, rows) {
        // Minimum 2 characters to filter.
        if (query.length < 2) {
          rows.forEach((row) => {
            row.style.display = '';
          });
          Drupal.announce(
            Drupal.t('Showing all @count items', { '@count': rows.length })
          );
          return;
        }

        // Escape regex special characters.
        const escapedQuery = query.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        const re = new RegExp(escapedQuery, 'i');

        let visibleCount = 0;
        rows.forEach((row) => {
          const sources = row.querySelectorAll('[data-drupal-selector*="filter-text-source"]');
          let text = '';
          sources.forEach((el) => {
            text += ' ' + el.textContent;
          });

          if (re.test(text)) {
            row.style.display = '';
            visibleCount++;
          } else {
            row.style.display = 'none';
          }
        });

        // Announce results for screen readers.
        Drupal.announce(
          Drupal.t('@count items shown', { '@count': visibleCount })
        );
      }
    }
  };

})(Drupal, once);
