<?php

namespace Drupal\eb\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\eb\Entity\EbDefinition;
use Drupal\eb\Service\ExportSecurityService;
use Drupal\eb\Service\YamlParser;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for exporting EbDefinition entities to YAML.
 *
 * Provides a download endpoint that serializes definition data to YAML format.
 * The exported YAML is compatible with import, enabling round-trip capability.
 * Internal tracking fields are sanitized for security.
 */
class EbDefinitionExportController extends ControllerBase {

  /**
   * Constructor.
   *
   * @param \Drupal\eb\Service\YamlParser $yamlParser
   *   The YAML parser service.
   * @param \Drupal\eb\Service\ExportSecurityService $exportSecurity
   *   The export security service.
   */
  public function __construct(
    protected YamlParser $yamlParser,
    protected ExportSecurityService $exportSecurity,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    // @phpstan-ignore new.static
    return new static(
      $container->get('eb.yaml_parser'),
      $container->get('eb.export_security'),
    );
  }

  /**
   * Exports a definition entity to YAML file download.
   *
   * Sanitizes the export by removing internal tracking fields and filtering
   * sensitive permissions from role definitions.
   *
   * @param \Drupal\eb\Entity\EbDefinition $eb_definition
   *   The definition entity to export.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response with YAML content and appropriate headers for download.
   */
  public function export(EbDefinition $eb_definition): Response {
    // Get exportable array from definition.
    $data = $eb_definition->toExportArray();

    // Sanitize export data (remove internal fields, filter sensitive perms).
    $data = $this->exportSecurity->sanitizeForExport($data);

    // Convert to YAML.
    $content = $this->yamlParser->export($data);

    // Optionally sign the export if signing is configured.
    $content = $this->exportSecurity->signExport($content);

    // Build filename from definition ID.
    $filename = $eb_definition->id() . '.yml';

    // Create response with YAML content.
    $response = new Response($content);
    $response->headers->set('Content-Type', 'application/x-yaml');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');
    $response->headers->set('Cache-Control', 'private, no-cache, no-store, must-revalidate');
    $response->headers->set('Pragma', 'no-cache');
    $response->headers->set('Expires', '0');

    return $response;
  }

  /**
   * Title callback for export page.
   *
   * @param \Drupal\eb\Entity\EbDefinition $eb_definition
   *   The definition entity.
   *
   * @return string
   *   The page title.
   */
  public function title(EbDefinition $eb_definition): string {
    return $this->t('Export @label', ['@label' => $eb_definition->label()])->render();
  }

}
