<?php

namespace Drupal\eb;

use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Url;
use Drupal\eb\Entity\EbDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a class to build a listing of entity builder definitions.
 */
class EbDefinitionListBuilder extends ConfigEntityListBuilder {

  /**
   * Disable pagination for client-side filtering.
   *
   * @var int|false
   */
  protected $limit = FALSE;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * Constructs an EbDefinitionListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    DateFormatterInterface $date_formatter,
  ) {
    parent::__construct($entity_type, $storage);
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    // @phpstan-ignore new.static
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build = [];

    // Attach filtering library.
    $build['#attached']['library'][] = 'eb/admin.listing';

    // Filter input following Views UI pattern.
    $build['filters'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['table-filter', 'js-show']],
    ];
    $build['filters']['text'] = [
      '#type' => 'search',
      '#title' => $this->t('Filter'),
      '#title_display' => 'invisible',
      '#size' => 60,
      '#placeholder' => $this->t('Filter by name or entity type'),
      '#attributes' => [
        'class' => ['eb-definition-filter-text'],
        'data-table' => '.eb-definition-listing-table',
        'autocomplete' => 'off',
      ],
    ];

    // Table with status grouping.
    $build['table'] = [
      '#type' => 'table',
      '#header' => $this->buildHeader(),
      '#rows' => [],
      '#empty' => $this->t('No definitions available. Import a YAML definition to get started.'),
      '#attributes' => ['class' => ['eb-definition-listing-table']],
    ];

    foreach ($this->load() as $entity) {
      if ($row = $this->buildRow($entity)) {
        $build['table']['#rows'][$entity->id()] = $row;
      }
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    return [
      'label' => [
        'data' => $this->t('Name'),
        'class' => ['eb-definition-name'],
      ],
      'id' => [
        'data' => $this->t('Machine name'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      'entity_type' => [
        'data' => $this->t('Entity types'),
        'class' => [RESPONSIVE_PRIORITY_MEDIUM],
      ],
      'status' => $this->t('Status'),
      'applied' => [
        'data' => $this->t('Applied'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      'operations' => $this->t('Operations'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\eb\Entity\EbDefinition $entity */
    $status = $entity->getApplicationStatus();

    $row = [
      'label' => [
        'data' => [
          '#type' => 'link',
          '#title' => $entity->label(),
          '#url' => $entity->toUrl('canonical'),
        ],
        'data-drupal-selector' => 'eb-definition-filter-text-source',
      ],
      'id' => [
        'data' => ['#plain_text' => $entity->id()],
        'data-drupal-selector' => 'eb-definition-filter-text-source',
      ],
      'entity_type' => [
        'data' => ['#plain_text' => $this->formatEntityTypes($entity)],
        'data-drupal-selector' => 'eb-definition-filter-text-source',
      ],
      'status' => [
        'data' => $this->buildStatusBadge($status),
      ],
      'applied' => $this->formatAppliedDate($entity->getAppliedDate()),
      'operations' => [
        'data' => $this->buildOperations($entity),
      ],
    ];

    return [
      'data' => $row,
      'class' => ['eb-definition-' . $status],
    ];
  }

  /**
   * Builds status badge markup.
   *
   * @param string $status
   *   The application status.
   *
   * @return array
   *   Render array for status badge.
   */
  protected function buildStatusBadge(string $status): array {
    $labels = [
      'draft' => $this->t('Draft'),
      'applied' => $this->t('Applied'),
      'outdated' => $this->t('Outdated'),
    ];
    $classes = [
      'draft' => 'admin-missing',
      'applied' => 'admin-enabled',
      'outdated' => 'admin-required',
    ];

    return [
      '#markup' => '<span class="eb-status-badge ' . ($classes[$status] ?? '') . '">' . ($labels[$status] ?? $status) . '</span>',
    ];
  }

  /**
   * Formats the applied date.
   *
   * @param int $timestamp
   *   The applied timestamp.
   *
   * @return string
   *   Formatted date or N/A.
   */
  protected function formatAppliedDate(int $timestamp): string {
    if ($timestamp === 0) {
      return (string) $this->t('Never');
    }
    return $this->dateFormatter->format($timestamp, 'short');
  }

  /**
   * Formats entity types for display.
   *
   * @param \Drupal\eb\Entity\EbDefinition $entity
   *   The definition entity.
   *
   * @return string
   *   Comma-separated list of entity types or 'Not set'.
   */
  protected function formatEntityTypes(EbDefinition $entity): string {
    $entityTypes = $entity->getEntityTypes();
    if (empty($entityTypes)) {
      return (string) $this->t('Not set');
    }
    return implode(', ', $entityTypes);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);

    // Add view operation.
    if ($entity->hasLinkTemplate('canonical')) {
      $operations['view'] = [
        'title' => $this->t('View'),
        'weight' => -10,
        'url' => $entity->toUrl('canonical'),
      ];
    }

    // Add edit operation (provided by eb_ui module).
    if ($this->moduleHandler()->moduleExists('eb_ui')) {
      $operations['edit'] = [
        'title' => $this->t('Edit'),
        'weight' => 0,
        'url' => Url::fromRoute('eb_ui.edit', [
          'definition_id' => $entity->id(),
        ]),
      ];
    }

    // Add apply operation.
    if ($entity->hasLinkTemplate('apply-form')) {
      $operations['apply'] = [
        'title' => $this->t('Apply'),
        'weight' => 5,
        'url' => $entity->toUrl('apply-form'),
      ];
    }

    // Add export operation.
    $operations['export'] = [
      'title' => $this->t('Export'),
      'weight' => 10,
      'url' => Url::fromRoute('entity.eb_definition.export', [
        'eb_definition' => $entity->id(),
      ]),
    ];

    return $operations;
  }

}
