<?php

declare(strict_types=1);

namespace Drupal\eb\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eb\ListBuilder\EbLogListBuilder;
use Drupal\user\EntityOwnerTrait;
use Drupal\views\EntityViewsData;

/**
 * Defines the EbLog content entity.
 *
 * Tracks apply/rollback/import sessions with timestamps for watchdog queries.
 */
#[ContentEntityType(
  id: 'eb_log',
  label: new TranslatableMarkup('Entity Builder Log'),
  label_collection: new TranslatableMarkup('Entity Builder Logs'),
  label_singular: new TranslatableMarkup('log'),
  label_plural: new TranslatableMarkup('logs'),
  entity_keys: [
    'id' => 'id',
    'uuid' => 'uuid',
    'label' => 'label',
    'owner' => 'uid',
  ],
  handlers: [
    'storage' => SqlContentEntityStorage::class,
    'list_builder' => EbLogListBuilder::class,
    'views_data' => EntityViewsData::class,
  ],
  links: [
    'collection' => '/admin/config/development/eb/log',
    'canonical' => '/admin/config/development/eb/log/{eb_log}',
  ],
  admin_permission: 'administer entity builder',
  base_table: 'eb_log',
)]
final class EbLog extends ContentEntityBase implements EbLogInterface {

  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Label'))
      ->setDescription(new TranslatableMarkup('Human-readable label for the log entry.'))
      ->setSetting('max_length', 255)
      ->setRequired(TRUE);

    $fields['definition_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Definition'))
      ->setDescription(new TranslatableMarkup('The entity builder definition this log relates to.'))
      ->setSetting('target_type', 'eb_definition')
      ->setRequired(FALSE);

    $fields['action'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Action'))
      ->setDescription(new TranslatableMarkup('The action type: apply, rollback, import.'))
      ->setSetting('max_length', 32)
      ->setDefaultValue('apply');

    $fields['operation_count'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Operation Count'))
      ->setDescription(new TranslatableMarkup('Total number of operations attempted.'))
      ->setDefaultValue(0);

    $fields['success_count'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Success Count'))
      ->setDescription(new TranslatableMarkup('Number of successful operations.'))
      ->setDefaultValue(0);

    $fields['failure_count'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Failure Count'))
      ->setDescription(new TranslatableMarkup('Number of failed operations.'))
      ->setDefaultValue(0);

    $fields['status'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Status'))
      ->setDescription(new TranslatableMarkup('The status: pending, success, partial, failed.'))
      ->setSetting('max_length', 32)
      ->setDefaultValue('pending');

    $fields['rollback_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Rollback'))
      ->setDescription(new TranslatableMarkup('Reference to the rollback entity if applicable.'))
      ->setSetting('target_type', 'eb_rollback')
      ->setRequired(FALSE);

    $fields['started'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(new TranslatableMarkup('Started'))
      ->setDescription(new TranslatableMarkup('Unix timestamp when action started.'));

    $fields['completed'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(new TranslatableMarkup('Completed'))
      ->setDescription(new TranslatableMarkup('Unix timestamp when action completed.'));

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(new TranslatableMarkup('Created'))
      ->setDescription(new TranslatableMarkup('The time the log entry was created.'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefinitionId(): ?string {
    $value = $this->get('definition_id')->target_id;
    return $value !== NULL ? (string) $value : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setDefinitionId(string $definitionId): self {
    $this->set('definition_id', $definitionId);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getAction(): string {
    return $this->get('action')->value ?? 'apply';
  }

  /**
   * {@inheritdoc}
   */
  public function setAction(string $action): self {
    $this->set('action', $action);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperationCount(): int {
    return (int) $this->get('operation_count')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setOperationCount(int $count): self {
    $this->set('operation_count', $count);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getSuccessCount(): int {
    return (int) $this->get('success_count')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setSuccessCount(int $count): self {
    $this->set('success_count', $count);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getFailureCount(): int {
    return (int) $this->get('failure_count')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setFailureCount(int $count): self {
    $this->set('failure_count', $count);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(): string {
    return $this->get('status')->value ?? 'pending';
  }

  /**
   * {@inheritdoc}
   */
  public function setStatus(string $status): self {
    $this->set('status', $status);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getRollbackId(): ?int {
    $value = $this->get('rollback_id')->target_id;
    return $value !== NULL ? (int) $value : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setRollbackId(int $rollbackId): self {
    $this->set('rollback_id', $rollbackId);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStarted(): ?int {
    $value = $this->get('started')->value;
    return $value !== NULL ? (int) $value : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setStarted(int $timestamp): self {
    $this->set('started', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCompleted(): ?int {
    $value = $this->get('completed')->value;
    return $value !== NULL ? (int) $value : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setCompleted(int $timestamp): self {
    $this->set('completed', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isPending(): bool {
    return $this->getStatus() === 'pending';
  }

  /**
   * {@inheritdoc}
   */
  public function isSuccess(): bool {
    return $this->getStatus() === 'success';
  }

}
