<?php

declare(strict_types=1);

namespace Drupal\eb\Entity;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\user\EntityOwnerInterface;

/**
 * Interface for EbLog content entity.
 *
 * Tracks apply/rollback/import sessions with timestamps for watchdog queries.
 */
interface EbLogInterface extends ContentEntityInterface, EntityOwnerInterface {

  /**
   * Gets the definition ID.
   *
   * @return string|null
   *   The definition ID or NULL if not set.
   */
  public function getDefinitionId(): ?string;

  /**
   * Sets the definition ID.
   *
   * @param string $definitionId
   *   The definition ID.
   *
   * @return $this
   */
  public function setDefinitionId(string $definitionId): self;

  /**
   * Gets the action type.
   *
   * @return string
   *   The action type (apply, rollback, import).
   */
  public function getAction(): string;

  /**
   * Sets the action type.
   *
   * @param string $action
   *   The action type (apply, rollback, import).
   *
   * @return $this
   */
  public function setAction(string $action): self;

  /**
   * Gets the operation count.
   *
   * @return int
   *   The total number of operations attempted.
   */
  public function getOperationCount(): int;

  /**
   * Sets the operation count.
   *
   * @param int $count
   *   The total number of operations.
   *
   * @return $this
   */
  public function setOperationCount(int $count): self;

  /**
   * Gets the success count.
   *
   * @return int
   *   The number of successful operations.
   */
  public function getSuccessCount(): int;

  /**
   * Sets the success count.
   *
   * @param int $count
   *   The number of successful operations.
   *
   * @return $this
   */
  public function setSuccessCount(int $count): self;

  /**
   * Gets the failure count.
   *
   * @return int
   *   The number of failed operations.
   */
  public function getFailureCount(): int;

  /**
   * Sets the failure count.
   *
   * @param int $count
   *   The number of failed operations.
   *
   * @return $this
   */
  public function setFailureCount(int $count): self;

  /**
   * Gets the status.
   *
   * @return string
   *   The status (pending, success, partial, failed).
   */
  public function getStatus(): string;

  /**
   * Sets the status.
   *
   * @param string $status
   *   The status (pending, success, partial, failed).
   *
   * @return $this
   */
  public function setStatus(string $status): self;

  /**
   * Gets the rollback entity ID.
   *
   * @return int|null
   *   The rollback entity ID or NULL if not set.
   */
  public function getRollbackId(): ?int;

  /**
   * Sets the rollback entity ID.
   *
   * @param int $rollbackId
   *   The rollback entity ID.
   *
   * @return $this
   */
  public function setRollbackId(int $rollbackId): self;

  /**
   * Gets the started timestamp.
   *
   * @return int|null
   *   The Unix timestamp when the action started.
   */
  public function getStarted(): ?int;

  /**
   * Sets the started timestamp.
   *
   * @param int $timestamp
   *   The Unix timestamp when the action started.
   *
   * @return $this
   */
  public function setStarted(int $timestamp): self;

  /**
   * Gets the completed timestamp.
   *
   * @return int|null
   *   The Unix timestamp when the action completed, or NULL if not completed.
   */
  public function getCompleted(): ?int;

  /**
   * Sets the completed timestamp.
   *
   * @param int $timestamp
   *   The Unix timestamp when the action completed.
   *
   * @return $this
   */
  public function setCompleted(int $timestamp): self;

  /**
   * Checks if this log entry is pending.
   *
   * @return bool
   *   TRUE if status is pending.
   */
  public function isPending(): bool;

  /**
   * Checks if this log entry completed successfully.
   *
   * @return bool
   *   TRUE if status is success.
   */
  public function isSuccess(): bool;

}
