<?php

declare(strict_types=1);

namespace Drupal\eb\Entity;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\user\EntityOwnerInterface;

/**
 * Interface for EbRollback content entity.
 *
 * Represents a single definition application that can be rolled back.
 * Contains multiple EbRollbackOperation entities.
 */
interface EbRollbackInterface extends ContentEntityInterface, EntityChangedInterface, EntityOwnerInterface {

  /**
   * Gets the definition ID.
   *
   * @return string|null
   *   The definition ID or NULL if not set.
   */
  public function getDefinitionId(): ?string;

  /**
   * Sets the definition ID.
   *
   * @param string $definitionId
   *   The definition ID.
   *
   * @return $this
   */
  public function setDefinitionId(string $definitionId): self;

  /**
   * Gets the rollback status.
   *
   * @return string
   *   The status (pending, completed, failed).
   */
  public function getStatus(): string;

  /**
   * Sets the rollback status.
   *
   * @param string $status
   *   The status (pending, completed, failed).
   *
   * @return $this
   */
  public function setStatus(string $status): self;

  /**
   * Gets the operation count.
   *
   * @return int
   *   The number of operations in this rollback.
   */
  public function getOperationCount(): int;

  /**
   * Sets the operation count.
   *
   * @param int $count
   *   The number of operations.
   *
   * @return $this
   */
  public function setOperationCount(int $count): self;

  /**
   * Gets the completed timestamp.
   *
   * @return int|null
   *   The timestamp when rollback was executed, or NULL if not completed.
   */
  public function getCompletedTime(): ?int;

  /**
   * Sets the completed timestamp.
   *
   * @param int $timestamp
   *   The completion timestamp.
   *
   * @return $this
   */
  public function setCompletedTime(int $timestamp): self;

  /**
   * Checks if this rollback is pending.
   *
   * @return bool
   *   TRUE if status is pending.
   */
  public function isPending(): bool;

  /**
   * Checks if this rollback has been completed.
   *
   * @return bool
   *   TRUE if status is completed.
   */
  public function isCompleted(): bool;

}
