<?php

declare(strict_types=1);

namespace Drupal\eb\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eb\Access\EbRollbackOperationAccessControlHandler;
use Drupal\views\EntityViewsData;

/**
 * Defines the EbRollbackOperation content entity.
 *
 * Represents a single operation that was executed during a definition apply.
 * Always rolled back as part of its parent EbRollback entity.
 */
#[ContentEntityType(
  id: 'eb_rollback_operation',
  label: new TranslatableMarkup('Rollback Operation'),
  label_collection: new TranslatableMarkup('Rollback Operations'),
  label_singular: new TranslatableMarkup('rollback operation'),
  label_plural: new TranslatableMarkup('rollback operations'),
  entity_keys: [
    'id' => 'id',
    'uuid' => 'uuid',
    'label' => 'description',
  ],
  handlers: [
    'storage' => SqlContentEntityStorage::class,
    'access' => EbRollbackOperationAccessControlHandler::class,
    'views_data' => EntityViewsData::class,
  ],
  admin_permission: 'administer entity builder',
  base_table: 'eb_rollback_operation',
)]
class EbRollbackOperation extends ContentEntityBase implements EbRollbackOperationInterface {

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['rollback_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Rollback'))
      ->setDescription(new TranslatableMarkup('The parent rollback entity.'))
      ->setSetting('target_type', 'eb_rollback')
      ->setRequired(TRUE);

    $fields['definition_id'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Definition ID'))
      ->setDescription(new TranslatableMarkup('Denormalized from rollback for query optimization.'))
      ->setSetting('max_length', 255)
      ->setRequired(TRUE);

    $fields['operation_type'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Operation Type'))
      ->setDescription(new TranslatableMarkup('The operation plugin ID.'))
      ->setSetting('max_length', 64)
      ->setRequired(TRUE);

    $fields['description'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Description'))
      ->setDescription(new TranslatableMarkup('Human-readable operation description.'))
      ->setSetting('max_length', 255)
      ->setRequired(TRUE);

    $fields['original_data'] = BaseFieldDefinition::create('map')
      ->setLabel(new TranslatableMarkup('Original Data'))
      ->setDescription(new TranslatableMarkup('Serialized rollback data.'));

    $fields['sequence'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Sequence'))
      ->setDescription(new TranslatableMarkup('Execution order within rollback.'))
      ->setDefaultValue(0);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    // Validate definition_id matches rollback.definition_id.
    $rollback = $this->getRollback();
    if ($rollback !== NULL) {
      $rollbackDefinitionId = $rollback->getDefinitionId();
      if ($rollbackDefinitionId !== NULL && $this->getDefinitionId() !== $rollbackDefinitionId) {
        throw new \InvalidArgumentException(
          'Operation definition_id must match rollback.definition_id'
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getRollback(): ?EbRollbackInterface {
    $entity = $this->get('rollback_id')->entity;
    return $entity instanceof EbRollbackInterface ? $entity : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getRollbackId(): ?int {
    /** @var string|int|null $value */
    $value = $this->get('rollback_id')->target_id;
    return $value !== NULL ? (int) $value : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setRollbackId(int $rollbackId): self {
    $this->set('rollback_id', $rollbackId);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefinitionId(): ?string {
    return $this->get('definition_id')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setDefinitionId(string $definitionId): self {
    $this->set('definition_id', $definitionId);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperationType(): string {
    return $this->get('operation_type')->value ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setOperationType(string $type): self {
    $this->set('operation_type', $type);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string {
    return $this->get('description')->value ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription(string $description): self {
    $this->set('description', $description);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOriginalData(): array {
    $first = $this->get('original_data')->first();
    return $first !== NULL ? ($first->getValue() ?? []) : [];
  }

  /**
   * {@inheritdoc}
   */
  public function setOriginalData(array $data): self {
    $this->set('original_data', $data);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getSequence(): int {
    return (int) $this->get('sequence')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setSequence(int $sequence): self {
    $this->set('sequence', $sequence);
    return $this;
  }

}
