<?php

declare(strict_types=1);

namespace Drupal\eb\PluginInterfaces;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\eb\Result\ExecutionResult;
use Drupal\eb\Result\ValidationResult;

/**
 * Base interface for Entity Builder operation plugins.
 *
 * This interface defines the core methods that all operations must implement:
 * validation and execution. Optional capabilities like preview and rollback
 * are defined in separate interfaces that extend this one.
 *
 * @see \Drupal\eb\PluginInterfaces\PreviewableOperationInterface
 * @see \Drupal\eb\PluginInterfaces\ReversibleOperationInterface
 */
interface OperationInterface extends PluginInspectionInterface {

  /**
   * Validates the operation configuration.
   *
   * @return \Drupal\eb\Result\ValidationResult
   *   Validation result with errors, warnings, and messages.
   */
  public function validate(): ValidationResult;

  /**
   * Executes the operation.
   *
   * @return \Drupal\eb\Result\ExecutionResult
   *   Execution result with affected entities and rollback data.
   *
   * @throws \Drupal\eb\Exception\ExecutionException
   *   If execution fails.
   */
  public function execute(): ExecutionResult;

  /**
   * Checks access for executing this operation.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account to check.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   Access result.
   */
  public function checkAccess(AccountInterface $account): AccessResult;

  /**
   * Gets all operation data.
   *
   * @return array<string, mixed>
   *   Operation data array.
   */
  public function getData(): array;

  /**
   * Gets the operation type from the plugin definition.
   *
   * Returns the operation type (create, update, delete) as defined
   * in the plugin's #[EbOperation] attribute.
   *
   * @return string
   *   The operation type.
   */
  public function getOperationType(): string;

}
