<?php

namespace Drupal\eb\PluginManager;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\eb\Attribute\EbExtension;
use Drupal\eb\PluginInterfaces\EbExtensionInterface;

/**
 * Plugin manager for Entity Builder Extension plugins.
 *
 * Extension plugins allow modules to extend YAML parsing capabilities
 * through a single plugin class instead of multiple hooks.
 *
 * @see \Drupal\eb\Attribute\EbExtension
 * @see \Drupal\eb\PluginInterfaces\EbExtensionInterface
 * @see \Drupal\eb\PluginBase\EbExtensionBase
 */
class EbExtensionPluginManager extends DefaultPluginManager {

  /**
   * Constructor.
   *
   * @param \Traversable<string, mixed> $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cacheBackend,
    ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct(
      'Plugin/EbExtension',
      $namespaces,
      $moduleHandler,
      EbExtensionInterface::class,
      EbExtension::class
    );

    $this->alterInfo('eb_extension_info');
    $this->setCacheBackend($cacheBackend, 'eb_extension_plugins');
  }

  /**
   * Gets all extension plugin instances.
   *
   * @return \Drupal\eb\PluginInterfaces\EbExtensionInterface[]
   *   Array of extension plugin instances keyed by plugin ID.
   */
  public function getExtensions(): array {
    $extensions = [];
    foreach ($this->getDefinitions() as $id => $definition) {
      $extensions[$id] = $this->createInstance($id);
    }
    return $extensions;
  }

  /**
   * Gets extensions that handle a specific operation type.
   *
   * @param string $operationType
   *   Operation type ID (e.g., 'create_field_group').
   *
   * @return \Drupal\eb\PluginInterfaces\EbExtensionInterface[]
   *   Extensions that handle this operation, keyed by plugin ID.
   */
  public function getExtensionsForOperation(string $operationType): array {
    $matching = [];
    foreach ($this->getExtensions() as $id => $extension) {
      if (in_array($operationType, $extension->getOperations(), TRUE)) {
        $matching[$id] = $extension;
      }
    }
    return $matching;
  }

  /**
   * Gets extensions that handle a specific YAML key.
   *
   * @param string $yamlKey
   *   YAML key (e.g., 'field_groups').
   *
   * @return \Drupal\eb\PluginInterfaces\EbExtensionInterface[]
   *   Extensions that handle this YAML key, keyed by plugin ID.
   */
  public function getExtensionsForYamlKey(string $yamlKey): array {
    $matching = [];
    foreach ($this->getExtensions() as $id => $extension) {
      if (in_array($yamlKey, $extension->getYamlKeys(), TRUE)) {
        $matching[$id] = $extension;
      }
    }
    return $matching;
  }

  /**
   * Gets all definition keys from all extensions.
   *
   * Collects the yaml_keys from all registered extension plugins.
   * These are the keys that extensions handle in definition data
   * (e.g., 'field_group_definitions' from eb_field_group).
   *
   * @return array<string>
   *   Array of definition keys from all extensions.
   */
  public function getAllDefinitionKeys(): array {
    $keys = [];
    foreach ($this->getDefinitions() as $definition) {
      if (!empty($definition['yaml_keys']) && is_array($definition['yaml_keys'])) {
        $keys = array_merge($keys, $definition['yaml_keys']);
      }
    }
    return array_unique($keys);
  }

}
