<?php

namespace Drupal\eb\Service;

/**
 * Interface for definition generator service.
 *
 * The DefinitionGenerator reverse-engineers Drupal entities into definition
 * format data suitable for EbDefinition entities. This enables
 * the export workflow to be definition-centric rather than operation-centric.
 */
interface DefinitionGeneratorInterface {

  /**
   * Generates definition data from Drupal entities.
   *
   * Extracts bundle, field, and display configuration from the specified
   * bundles and converts them to definition format. Extensions contribute
   * via their extractConfig() method.
   *
   * @param array<string, array<string>> $bundleSelection
   *   Bundles to include, keyed by entity type.
   *   E.g., ['node' => ['article', 'page'], 'taxonomy_term' => ['tags']].
   * @param array<string, bool> $options
   *   Generation options:
   *   - include_fields: Include field definitions (default: true).
   *   - include_displays: Include display configurations (default: true).
   *   - include_extensions: Include extension data (default: true).
   *   - normalize_settings: Strip default values (default: true).
   *
   * @return array<string, mixed>
   *   Definition format data suitable for DefinitionFactory::createFromYaml().
   *   Contains keys like bundle_definitions, field_definitions,
   *   display_field_definitions, and extension-specific keys.
   */
  public function generate(array $bundleSelection, array $options = []): array;

  /**
   * Generates definition data for a single bundle.
   *
   * @param string $entityTypeId
   *   The entity type ID (e.g., 'node', 'taxonomy_term').
   * @param string $bundleId
   *   The bundle machine name (e.g., 'article', 'tags').
   * @param array<string, bool> $options
   *   Generation options (same as generate()).
   *
   * @return array<string, mixed>
   *   Single bundle's definition data in definition format.
   */
  public function generateBundle(string $entityTypeId, string $bundleId, array $options = []): array;

}
