<?php

declare(strict_types=1);

namespace Drupal\eb\Service;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\FieldConfigInterface;
use Drupal\field\FieldStorageConfigInterface;

/**
 * Interface for field management operations.
 *
 * Provides an abstraction layer over FieldStorageConfig and FieldConfig
 * to improve testability and follow the Dependency Inversion Principle (DIP).
 */
interface FieldManagementServiceInterface {

  /**
   * Loads a field storage configuration by entity type and field name.
   *
   * @param string $entityType
   *   The entity type ID.
   * @param string $fieldName
   *   The field name.
   *
   * @return \Drupal\field\FieldStorageConfigInterface|null
   *   The field storage config, or NULL if not found.
   */
  public function loadFieldStorage(string $entityType, string $fieldName): ?FieldStorageConfigInterface;

  /**
   * Creates a new field storage configuration.
   *
   * @param array<string, mixed> $config
   *   The field storage configuration array containing:
   *   - field_name: The field name.
   *   - entity_type: The entity type ID.
   *   - type: The field type.
   *   - cardinality: (optional) The cardinality.
   *   - translatable: (optional) Whether translatable.
   *   - settings: (optional) Field storage settings.
   *
   * @return \Drupal\field\FieldStorageConfigInterface
   *   The created field storage config.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   If the save operation fails.
   */
  public function createFieldStorage(array $config): FieldStorageConfigInterface;

  /**
   * Deletes a field storage configuration.
   *
   * @param \Drupal\field\FieldStorageConfigInterface $fieldStorage
   *   The field storage config to delete.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   If the delete operation fails.
   */
  public function deleteFieldStorage(FieldStorageConfigInterface $fieldStorage): void;

  /**
   * Loads a field configuration by entity type, bundle, and field name.
   *
   * @param string $entityType
   *   The entity type ID.
   * @param string $bundle
   *   The bundle ID.
   * @param string $fieldName
   *   The field name.
   *
   * @return \Drupal\field\Entity\FieldConfig|null
   *   The field config, or NULL if not found.
   */
  public function loadFieldConfig(string $entityType, string $bundle, string $fieldName): ?FieldConfig;

  /**
   * Loads a field configuration by its ID.
   *
   * @param string $fieldConfigId
   *   The field config ID (e.g., "node.article.field_body").
   *
   * @return \Drupal\field\Entity\FieldConfig|null
   *   The field config, or NULL if not found.
   */
  public function loadFieldConfigById(string $fieldConfigId): ?FieldConfig;

  /**
   * Creates a new field configuration.
   *
   * @param array<string, mixed> $config
   *   The field configuration array containing:
   *   - field_name: The field name.
   *   - entity_type: The entity type ID.
   *   - bundle: The bundle ID.
   *   - label: The field label.
   *   - description: (optional) The field description.
   *   - required: (optional) Whether required.
   *   - default_value: (optional) Default value.
   *   - settings: (optional) Field config settings.
   *
   * @return \Drupal\field\Entity\FieldConfig
   *   The created field config.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   If the save operation fails.
   */
  public function createFieldConfig(array $config): FieldConfig;

  /**
   * Deletes a field configuration.
   *
   * @param \Drupal\field\FieldConfigInterface $fieldConfig
   *   The field config to delete.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   If the delete operation fails.
   */
  public function deleteFieldConfig(FieldConfigInterface $fieldConfig): void;

  /**
   * Loads a field storage configuration by its ID.
   *
   * @param string $storageConfigId
   *   The storage config ID (e.g., "node.field_body").
   *
   * @return \Drupal\field\FieldStorageConfigInterface|null
   *   The field storage config, or NULL if not found.
   */
  public function loadFieldStorageById(string $storageConfigId): ?FieldStorageConfigInterface;

  /**
   * Checks if a field storage exists.
   *
   * @param string $entityType
   *   The entity type ID.
   * @param string $fieldName
   *   The field name.
   *
   * @return bool
   *   TRUE if the field storage exists, FALSE otherwise.
   */
  public function fieldStorageExists(string $entityType, string $fieldName): bool;

  /**
   * Checks if a field config exists.
   *
   * @param string $entityType
   *   The entity type ID.
   * @param string $bundle
   *   The bundle ID.
   * @param string $fieldName
   *   The field name.
   *
   * @return bool
   *   TRUE if the field config exists, FALSE otherwise.
   */
  public function fieldConfigExists(string $entityType, string $bundle, string $fieldName): bool;

  /**
   * Gets the bundles using a specific field.
   *
   * @param string $entityType
   *   The entity type ID.
   * @param string $fieldName
   *   The field name.
   *
   * @return array<string>
   *   Array of bundle IDs using this field.
   */
  public function getBundlesUsingField(string $entityType, string $fieldName): array;

  /**
   * Clears cached field definitions.
   */
  public function clearCachedFieldDefinitions(): void;

}
