<?php

namespace Drupal\eb\Service\Traits;

/**
 * Provides methods for generating entity keys from operations.
 *
 * This trait consolidates entity key generation logic used across
 * DependencyResolver, ChangeDetector, and other services.
 */
trait EntityKeyTrait {

  /**
   * Generate canonical entity key for dependency tracking.
   *
   * Keys follow the format: "type:entity_type:bundle:field_name" etc.
   *
   * @param array<string, mixed> $operation
   *   Operation data.
   * @param int|null $index
   *   Optional operation index for uniqueness.
   *
   * @return string
   *   Canonical key like "bundle:node:article" or "field:node:article:body".
   */
  protected function makeEntityKey(array $operation, ?int $index = NULL): string {
    $operationType = $operation['operation'] ?? '';

    if (in_array($operationType, ['create_bundle', 'update_bundle'])) {
      $entityType = $operation['entity_type'] ?? '';
      $bundleId = $operation['bundle_id'] ?? '';
      return "bundle:{$entityType}:{$bundleId}";
    }

    if (in_array($operationType, ['create_field', 'update_field'])) {
      $entityType = $operation['entity_type'] ?? '';
      $bundle = $operation['bundle'] ?? '';
      $fieldName = $operation['field_name'] ?? '';
      return "field:{$entityType}:{$bundle}:{$fieldName}";
    }

    if (in_array($operationType, ['configure_form_mode', 'configure_view_mode'])) {
      $entityType = $operation['entity_type'] ?? '';
      $bundle = $operation['bundle'] ?? '';
      $displayMode = $operation['display_mode'] ?? 'default';
      $fieldName = $operation['field_name'] ?? '';
      $displayType = $operationType === 'configure_form_mode' ? 'form' : 'view';
      return "display:{$displayType}:{$entityType}:{$bundle}:{$displayMode}:{$fieldName}";
    }

    if ($operationType === 'create_menu') {
      $menuId = $operation['menu_id'] ?? '';
      return "menu:{$menuId}";
    }

    if ($operationType === 'create_menu_link') {
      $menuId = $operation['menu_id'] ?? '';
      $title = $operation['title'] ?? '';
      return "menu_link:{$menuId}:{$title}";
    }

    if (in_array($operationType, ['create_field_group', 'update_field_group'])) {
      $entityType = $operation['entity_type'] ?? '';
      $bundle = $operation['bundle'] ?? '';
      $groupName = $operation['group_name'] ?? '';
      $displayMode = $operation['display_mode'] ?? 'default';
      $context = $operation['context'] ?? 'form';
      return "field_group:{$context}:{$entityType}:{$bundle}:{$displayMode}:{$groupName}";
    }

    // Fallback for unknown operation types.
    $suffix = $index !== NULL ? ":{$index}" : ':0';
    return "operation:{$operationType}{$suffix}";
  }

  /**
   * Get human-readable entity key for error messages.
   *
   * @param array<string, mixed> $operation
   *   Operation data.
   *
   * @return string
   *   Human-readable key like "Bundle node.article".
   */
  protected function getHumanReadableEntityKey(array $operation): string {
    $operationType = $operation['operation'] ?? '';

    if (str_contains($operationType, 'bundle')) {
      $entityType = $operation['entity_type'] ?? '';
      $bundleId = $operation['bundle_id'] ?? '';
      return "Bundle {$entityType}.{$bundleId}";
    }

    if (str_contains($operationType, 'field') && !str_contains($operationType, 'field_group')) {
      $entityType = $operation['entity_type'] ?? '';
      $bundle = $operation['bundle'] ?? '';
      $fieldName = $operation['field_name'] ?? '';
      return "Field {$entityType}.{$bundle}.{$fieldName}";
    }

    if (str_contains($operationType, 'field_group')) {
      $entityType = $operation['entity_type'] ?? '';
      $bundle = $operation['bundle'] ?? '';
      $groupName = $operation['group_name'] ?? '';
      return "Field Group {$entityType}.{$bundle}.{$groupName}";
    }

    if (str_contains($operationType, 'menu')) {
      $menuId = $operation['menu_id'] ?? '';
      return "Menu {$menuId}";
    }

    return "Entity";
  }

}
