# Entity Builder Tests

This directory contains comprehensive tests for the Entity Builder module.

## Directory Structure

```
tests/
├── src/
│   ├── Unit/                      # Pure unit tests (no Drupal bootstrap)
│   │   ├── Service/
│   │   │   ├── Yaml/             # YAML parsing and structure detection
│   │   │   ├── Dependency/       # Dependency resolution
│   │   │   └── ChangeDetection/  # Change detection and sync logic
│   │   └── Plugin/               # Plugin unit tests
│   │
│   ├── Kernel/                    # Kernel tests (minimal Drupal bootstrap)
│   │   ├── Service/              # Service integration with Drupal
│   │   ├── Operation/            # Operation execution tests
│   │   └── Integration/          # Multi-service integration tests
│   │
│   └── Functional/                # Browser-based functional tests
│       ├── Drush/                # Drush command tests
│       └── Workflow/             # End-to-end workflow tests
│
├── fixtures/                      # Test data and YAML files
│   ├── yaml/                     # Sample YAML files
│   ├── expected/                 # Expected outputs
│   └── invalid/                  # Invalid data for error testing
│
└── Traits/                        # Reusable test traits
    ├── BundleCreationTrait.php
    ├── FieldCreationTrait.php
    ├── OperationTestTrait.php
    ├── ValidationAssertionsTrait.php
    └── YamlFixtureTrait.php
```

## Test Categories

### Unit Tests (tests/src/Unit/)
- **Pure unit tests** with no Drupal dependencies
- Fast execution (< 1 second for all unit tests)
- Mock all external dependencies
- Focus on business logic in isolation

**Current Coverage:**
- StructureDetectorTest (17 tests)
- DependencyResolverTest (7 tests)
- HierarchicalValidatorTest (16 tests)
- HierarchicalYamlParserTest (13 tests)
- ChangeDetectorTest (3 tests)

### Kernel Tests (tests/src/Kernel/)
- **Minimal Drupal bootstrap** (database, entity system)
- Test services with real Drupal entities
- Verify integration between components
- Test change detection against real config

**Planned Coverage:**
- Parser with EntityTypeManager
- Dependency resolution with real bundles
- Change detection against Drupal config
- All mode combinations

### Functional Tests (tests/src/Functional/)
- **Full Drupal bootstrap** with browser testing
- End-to-end workflow testing
- Drush command execution
- User interface testing (if applicable)

**Planned Coverage:**
- Complete YAML import workflows
- Drush command functionality
- Preview and validation commands

## Naming Conventions

### File Naming
- **Unit Tests**: `ServiceNameTest.php`
- **Kernel Tests**: `ServiceNameKernelTest.php`
- **Functional Tests**: `FeatureFunctionalTest.php`

### Test Method Naming
- Use descriptive names: `testParseEntityReferenceField()`
- Follow pattern: `test{Action}{Condition}{ExpectedResult}()`
- Examples:
  - `testParseBundleWithFields()`
  - `testValidateFailsMissingFieldType()`
  - `testDetermineSyncOperationSkip()`

### Fixtures Naming
- Use descriptive names matching test scenarios
- Example: `hierarchical-with-entity-reference.yml`
- Group by feature: `invalid-missing-bundle-id.yml`

## Running Tests

### Configuration Files

**Project Root phpunit.xml** (Recommended for DDEV)
- Located at: `/var/www/html/phpunit.xml`
- Pre-configured with DDEV database connection
- Includes custom eb test suite
- Use this for running all tests with database access

**Note:** Always use `/var/www/html/phpunit.xml` for running tests.
The core `phpunit.xml.dist` is not configured for this project.

### Run All Tests (with Database)
```bash
# Using project phpunit.xml (DDEV configured)
vendor/bin/phpunit -c phpunit.xml web/modules/custom/eb/tests/

# Or using custom test suite
vendor/bin/phpunit -c phpunit.xml --testsuite eb
```

### Run Specific Test Suite
```bash
# Unit tests only (no database needed)
vendor/bin/phpunit -c phpunit.xml web/modules/custom/eb/tests/src/Unit/

# Kernel tests only (requires database)
vendor/bin/phpunit -c phpunit.xml web/modules/custom/eb/tests/src/Kernel/

# Functional tests only (requires database and browser)
vendor/bin/phpunit -c phpunit.xml web/modules/custom/eb/tests/src/Functional/
```

### Run Specific Test File
```bash
vendor/bin/phpunit -c phpunit.xml web/modules/custom/eb/tests/src/Unit/Service/Yaml/StructureDetectorTest.php
```

### Run with Test Documentation
```bash
vendor/bin/phpunit -c phpunit.xml web/modules/custom/eb/tests/ --testdox
```

### Quick Test Commands

**Unit Tests Only (Fast)**
```bash
vendor/bin/phpunit -c phpunit.xml web/modules/custom/eb/tests/src/Unit/ --testdox
```

**Kernel Tests Only**
```bash
vendor/bin/phpunit -c phpunit.xml web/modules/custom/eb/tests/src/Kernel/ --testdox
```

**All Entity Builder Tests**
```bash
vendor/bin/phpunit -c phpunit.xml --testsuite eb --testdox
```

## Best Practices

### 1. Test Independence
- Each test should be independent
- Use setUp() and tearDown() properly
- Don't rely on test execution order

### 2. Clear Assertions
- One logical assertion per test (may use multiple assert statements)
- Use descriptive failure messages
- Use assertEquals() over assertTrue() when possible

### 3. Test Data
- Store complex test data in fixtures/
- Use realistic data that matches production scenarios
- Include edge cases and invalid data

### 4. Coverage Goals
- **Unit Tests**: 90%+ coverage for business logic
- **Kernel Tests**: Cover all service integrations
- **Functional Tests**: Cover critical user workflows

### 5. Performance
- Keep unit tests fast (< 1 second total)
- Kernel tests should be reasonably fast (< 30 seconds)
- Functional tests can be slower but should be optimized

## Test Fixtures

Store test YAML files in `tests/fixtures/`:
- `simple_bundle.yml` - Basic bundle creation
- `bundle_with_fields.yml` - Bundle with multiple field types
- `bundle_with_displays.yml` - Form and view display configurations
- `roles_and_menus.yml` - Role and menu definitions
- `invalid_*.yml` - Invalid structures for error testing

## Debugging Tests

### Enable Verbose Output
```bash
vendor/bin/phpunit -c /var/www/html/phpunit.xml web/modules/custom/eb/tests/ -vvv
```

### Run Single Test Method
```bash
vendor/bin/phpunit -c /var/www/html/phpunit.xml web/modules/custom/eb/tests/ --filter testParseEntityReferenceField
```

### Debug with Print Statements
Use `fwrite(STDERR, print_r($variable, TRUE));` instead of `var_dump()` in tests.

## Continuous Integration

Tests should be run automatically on:
- Every commit (unit tests)
- Pull requests (all tests)
- Before release (all tests + coverage report)

## Coverage Reports

Generate coverage report:
```bash
XDEBUG_MODE=coverage vendor/bin/phpunit -c /var/www/html/phpunit.xml web/modules/custom/eb/tests/ --coverage-html coverage/
```

View coverage:
```bash
open coverage/index.html
```
