# YAML Format Reference

Complete specification for Entity Builder YAML definition files (v1.0).

## Document Structure

```yaml
version: "1.0"                    # Required: Format version
id: my_definition                 # Required: Machine name
label: My Definition              # Required: Human-readable label
mode: sync                        # Optional: sync (default) | create_only

bundle_definitions: []            # Bundle/content type definitions
field_definitions: []             # Field definitions
field_group_definitions: []       # Field groups (requires eb_field_group)
display_field_definitions: []     # Non-default display mode overrides
menu_definitions: []              # Custom menus
```

## Modes

| Mode | Creates New? | Updates Existing? | Skips Unchanged? |
|------|--------------|-------------------|------------------|
| `sync` (default) | Yes | Yes | Yes |
| `create_only` | Yes | Fails | N/A |

## Bundle Definitions

### Node (Content Type)

```yaml
bundle_definitions:
  - entity_type: node
    bundle_id: article
    label: Article
    description: Blog articles and news posts.
    settings:
      published_by_default: true
      promoted_by_default: false
      sticky_by_default: false
```

### Taxonomy Vocabulary

```yaml
bundle_definitions:
  - entity_type: taxonomy_term
    bundle_id: tags
    label: Tags
    description: Content classification tags.
```

### Media Type

```yaml
bundle_definitions:
  - entity_type: media
    bundle_id: image
    label: Image
    description: Image media assets.
    settings:
      source: image
```

### Bundle Properties

| Property | Type | Required | Description |
|----------|------|----------|-------------|
| `entity_type` | string | Yes | `node`, `taxonomy_term`, `media`, etc. |
| `bundle_id` | string | Yes | Machine name (lowercase, underscores) |
| `label` | string | Yes | Human-readable name |
| `description` | string | No | Bundle description |
| `settings.published_by_default` | boolean | No | Auto-publish new content |
| `settings.promoted_by_default` | boolean | No | Promote to front page |
| `settings.sticky_by_default` | boolean | No | Make sticky |
| `settings.source` | string | No | Media source plugin ID |

### Extension Properties

With `eb_pathauto`:
```yaml
bundle_definitions:
  - entity_type: node
    bundle_id: article
    label: Article
    pathauto_pattern: /blog/[node:title]
```

With `eb_auto_entitylabel`:
```yaml
bundle_definitions:
  - entity_type: node
    bundle_id: article
    label: Article
    auto_entitylabel_status: enabled
    auto_entitylabel_pattern: "[node:field_date] - [node:title]"
```

## Field Definitions

### Basic Structure

```yaml
field_definitions:
  - entity_type: node
    bundle: article
    field_name: field_summary
    field_type: text_long
    label: Summary
    required: true
    widget: text_textarea
    formatter: text_default
    weight: 0
```

### Required Properties

| Property | Type | Description |
|----------|------|-------------|
| `entity_type` | string | Target entity type |
| `bundle` | string | Target bundle |
| `field_name` | string | Machine name (must start with `field_`) |
| `field_type` | string | Field type plugin ID |
| `label` | string | Human-readable label |

### Display Properties

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `widget` | string | auto | Form widget plugin ID |
| `formatter` | string | auto | View formatter plugin ID |
| `widget_settings` | object | `{}` | Widget configuration |
| `formatter_settings` | object | `{}` | Formatter configuration |
| `label_display` | string | `above` | `above`, `inline`, `hidden`, `visually_hidden` |
| `weight` | integer | `0` | Display order (lower = higher) |

### Field Configuration Properties

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `required` | boolean | `false` | Field is required |
| `translatable` | boolean | `false` | Field supports translation |
| `cardinality` | integer | `1` | Number of values (`-1` for unlimited) |
| `description` | string | `""` | Help text for editors |
| `default_value` | array | `[]` | Default field value |

### Storage and Config Settings

```yaml
field_definitions:
  - entity_type: node
    bundle: article
    field_name: field_price
    field_type: decimal
    label: Price
    field_storage_settings:
      precision: 10
      scale: 2
    field_config_settings:
      min: 0
      prefix: "$"
```

### Field Group Assignment

With `eb_field_group` enabled:

```yaml
field_definitions:
  - entity_type: node
    bundle: product
    field_name: field_price
    field_type: decimal
    label: Price
    form_group: group_pricing    # Form display group
    view_group: group_details    # View display group
```

## Common Field Types

### Text Fields

```yaml
# Plain text (single line)
- field_name: field_title
  field_type: string
  widget: string_textfield
  formatter: string

# Long text (multi-line)
- field_name: field_body
  field_type: text_long
  widget: text_textarea
  formatter: text_default

# Text with summary
- field_name: field_description
  field_type: text_with_summary
  widget: text_textarea_with_summary
  formatter: text_default
```

### Number Fields

```yaml
# Integer
- field_name: field_quantity
  field_type: integer
  widget: number
  formatter: number_integer
  field_config_settings:
    min: 0
    max: 100

# Decimal
- field_name: field_price
  field_type: decimal
  widget: number
  formatter: number_decimal
  field_storage_settings:
    precision: 10
    scale: 2
```

### Boolean

```yaml
- field_name: field_published
  field_type: boolean
  widget: boolean_checkbox
  formatter: boolean
  field_config_settings:
    on_label: "Yes"
    off_label: "No"
```

### List/Select Fields

```yaml
# String list
- field_name: field_status
  field_type: list_string
  widget: options_select
  formatter: list_default
  field_storage_settings:
    allowed_values:
      draft: Draft
      review: In Review
      published: Published

# As radio buttons
- field_name: field_priority
  field_type: list_integer
  widget: options_buttons
  formatter: list_default
  field_storage_settings:
    allowed_values:
      1: "Low"
      2: "Medium"
      3: "High"
```

### Entity Reference

```yaml
# Reference to content
- field_name: field_related
  field_type: entity_reference
  widget: entity_reference_autocomplete
  formatter: entity_reference_label
  field_storage_settings:
    target_type: node
  field_config_settings:
    handler_settings:
      target_bundles:
        article: article
        page: page

# Reference to taxonomy with auto-create
- field_name: field_tags
  field_type: entity_reference
  widget: entity_reference_autocomplete_tags
  formatter: entity_reference_label
  cardinality: -1
  field_storage_settings:
    target_type: taxonomy_term
  field_config_settings:
    handler_settings:
      target_bundles:
        tags: tags
      auto_create: true
```

### Media/File Fields

```yaml
# Image
- field_name: field_image
  field_type: image
  widget: image_image
  formatter: image
  field_config_settings:
    file_extensions: "png jpg jpeg webp"
    max_filesize: "5 MB"
    alt_field: true
    title_field: false
  formatter_settings:
    image_style: large
    image_link: ""

# File
- field_name: field_document
  field_type: file
  widget: file_generic
  formatter: file_default
  field_config_settings:
    file_extensions: "pdf doc docx"
    max_filesize: "10 MB"
```

### Date/Time

```yaml
# Date only
- field_name: field_date
  field_type: datetime
  widget: datetime_default
  formatter: datetime_default
  field_storage_settings:
    datetime_type: date

# Date and time
- field_name: field_event_date
  field_type: datetime
  widget: datetime_default
  formatter: datetime_default
  field_storage_settings:
    datetime_type: datetime
```

### Link

```yaml
- field_name: field_website
  field_type: link
  widget: link_default
  formatter: link
  field_config_settings:
    link_type: 16        # 16=External, 1=Internal, 17=Both
    title: 1             # 0=disabled, 1=optional, 2=required
```

### Email

```yaml
- field_name: field_email
  field_type: email
  widget: email_default
  formatter: email_mailto
```

## Field Group Definitions

Requires `eb_field_group` module.

### Format Types

```yaml
# Fieldset (bordered container)
- group_name: group_basic
  format_type: fieldset

# Collapsible details
- group_name: group_advanced
  format_type: details
  format_settings:
    open: false

# Tabs container
- group_name: group_tabs
  format_type: tabs
  format_settings:
    direction: horizontal

# Tab item (must have tabs parent)
- group_name: group_tab_general
  format_type: tab
  parent: group_tabs

# HTML element
- group_name: group_header
  format_type: html_element
  format_settings:
    element: header
    classes: "product-header"
```

### Complete Tabs Example

```yaml
field_definitions:
  - field_name: field_sku
    form_group: group_tab_basic
    # ...

  - field_name: field_price
    form_group: group_tab_pricing
    # ...

field_group_definitions:
  # Tabs container
  - entity_type: node
    bundle: product
    display_type: form
    mode: default
    group_name: group_tabs
    label: Product
    format_type: tabs
    weight: 0

  # Basic tab
  - entity_type: node
    bundle: product
    display_type: form
    mode: default
    group_name: group_tab_basic
    label: Basic Info
    format_type: tab
    parent: group_tabs
    weight: 0

  # Pricing tab
  - entity_type: node
    bundle: product
    display_type: form
    mode: default
    group_name: group_tab_pricing
    label: Pricing
    format_type: tab
    parent: group_tabs
    weight: 1
```

## Display Field Definitions

Override display settings for non-default modes:

```yaml
display_field_definitions:
  - entity_type: node
    bundle: article
    display_type: view
    mode: teaser
    field_name: field_body
    formatter: text_trimmed
    formatter_settings:
      trim_length: 200
    label_display: hidden
    weight: 1
```

## Menu Definitions

```yaml
menu_definitions:
  - menu_id: main_navigation
    label: Main Navigation
    description: Primary site navigation.
```

## Naming Conventions

| Element | Convention | Example |
|---------|------------|---------|
| Definition ID | lowercase, underscores | `blog_platform` |
| Bundle ID | lowercase, underscores | `blog_post` |
| Field name | `field_` prefix | `field_summary` |
| Group name | `group_` prefix | `group_details` |
| Menu ID | lowercase, underscores | `main_menu` |

## Validation Rules

1. Fields must reference existing bundles (or bundles defined in same YAML)
2. Field groups must reference existing bundles
3. Entity references must target existing entity types
4. Child groups must reference existing parent groups
5. `display_type: form` uses `widget` and `widget_settings`
6. `display_type: view` uses `formatter` and `formatter_settings`
