<?php

namespace Drupal\eb_ui\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eb_ui\Service\GridProviderManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the settings form for Entity Builder UI.
 */
class EbUiSettingsForm extends ConfigFormBase {

  /**
   * The grid provider manager.
   *
   * @var \Drupal\eb_ui\Service\GridProviderManagerInterface
   */
  protected GridProviderManagerInterface $providerManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->providerManager = $container->get('eb_ui.grid_provider_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string>
   *   The editable config names.
   */
  protected function getEditableConfigNames(): array {
    return ['eb_ui.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'eb_ui_settings_form';
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('eb_ui.settings');

    // Editor mode section.
    $form['editor_mode_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Editor Mode'),
      '#open' => TRUE,
    ];

    // Build editor mode options.
    $options = [
      'yaml' => $this->t('YAML Editor'),
      'auto' => $this->t('Auto (use first available grid provider)'),
    ];

    // Add available providers.
    $providers = $this->providerManager->getAvailableProviders();
    foreach ($providers as $id => $provider) {
      $options[$id] = $provider['label'];
    }

    $form['editor_mode_section']['editor_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Default editor mode'),
      '#description' => $this->t('Select the default editor to use when creating or editing definitions.'),
      '#options' => $options,
      '#default_value' => $config->get('editor_mode') ?? 'auto',
    ];

    // YAML editor section.
    $form['yaml_editor_section'] = [
      '#type' => 'details',
      '#title' => $this->t('YAML Editor Settings'),
      '#open' => TRUE,
    ];

    $form['yaml_editor_section']['yaml_editor'] = [
      '#type' => 'radios',
      '#title' => $this->t('YAML editor type'),
      '#description' => $this->t('Select the YAML editor implementation to use.'),
      '#options' => [
        'codemirror_cdn' => $this->t('CodeMirror 5 (CDN) - Syntax highlighting and line numbers'),
        'plain_textarea' => $this->t('Plain textarea - No external dependencies'),
      ],
      '#default_value' => $config->get('yaml_editor') ?? 'codemirror_cdn',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @param-out array<string, mixed> $form
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('eb_ui.settings')
      ->set('editor_mode', $form_state->getValue('editor_mode'))
      ->set('yaml_editor', $form_state->getValue('yaml_editor'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
