<?php

declare(strict_types=1);

namespace Drupal\eb\Entity;

use Drupal\Core\Entity\ContentEntityInterface;

/**
 * Interface for EbRollbackOperation content entity.
 *
 * Represents a single operation that was executed during a definition apply.
 * Always rolled back as part of its parent EbRollback entity.
 */
interface EbRollbackOperationInterface extends ContentEntityInterface {

  /**
   * Gets the parent rollback entity.
   *
   * @return \Drupal\eb\Entity\EbRollbackInterface|null
   *   The parent rollback entity or NULL if not found.
   */
  public function getRollback(): ?EbRollbackInterface;

  /**
   * Gets the parent rollback ID.
   *
   * @return int|null
   *   The rollback ID or NULL if not set.
   */
  public function getRollbackId(): ?int;

  /**
   * Sets the parent rollback ID.
   *
   * @param int $rollbackId
   *   The rollback ID.
   *
   * @return $this
   */
  public function setRollbackId(int $rollbackId): self;

  /**
   * Gets the definition ID.
   *
   * This is denormalized from the parent rollback for query optimization.
   *
   * @return string|null
   *   The definition ID or NULL if not set.
   */
  public function getDefinitionId(): ?string;

  /**
   * Sets the definition ID.
   *
   * @param string $definitionId
   *   The definition ID.
   *
   * @return $this
   */
  public function setDefinitionId(string $definitionId): self;

  /**
   * Gets the operation type (plugin ID).
   *
   * @return string
   *   The operation plugin ID.
   */
  public function getOperationType(): string;

  /**
   * Sets the operation type.
   *
   * @param string $type
   *   The operation plugin ID.
   *
   * @return $this
   */
  public function setOperationType(string $type): self;

  /**
   * Gets the operation description.
   *
   * @return string
   *   Human-readable operation description.
   */
  public function getDescription(): string;

  /**
   * Sets the operation description.
   *
   * @param string $description
   *   The description.
   *
   * @return $this
   */
  public function setDescription(string $description): self;

  /**
   * Gets the original data for rollback.
   *
   * @return array<string, mixed>
   *   The serialized rollback data.
   */
  public function getOriginalData(): array;

  /**
   * Sets the original data for rollback.
   *
   * @param array<string, mixed> $data
   *   The rollback data.
   *
   * @return $this
   */
  public function setOriginalData(array $data): self;

  /**
   * Gets the sequence number.
   *
   * @return int
   *   The execution order within the rollback.
   */
  public function getSequence(): int;

  /**
   * Sets the sequence number.
   *
   * @param int $sequence
   *   The sequence number.
   *
   * @return $this
   */
  public function setSequence(int $sequence): self;

}
