<?php

declare(strict_types=1);

namespace Drupal\eb\Enum;

/**
 * Defines change detection modes for entity builder operations.
 */
enum ChangeDetectionMode: string {

  // Sync mode: create if not exists, update if changed.
  case Sync = 'sync';

  // Create only: fail if entity already exists.
  case CreateOnly = 'create_only';

  // Update only: fail if entity doesn't exist.
  case UpdateOnly = 'update_only';

  // Replace mode: delete and recreate entity.
  case Replace = 'replace';

  /**
   * Get human-readable label for the mode.
   *
   * @return string
   *   Human-readable label.
   */
  public function getLabel(): string {
    return match ($this) {
      self::Sync => 'Sync (create or update)',
      self::CreateOnly => 'Create only',
      self::UpdateOnly => 'Update only',
      self::Replace => 'Replace (delete and recreate)',
    };
  }

  /**
   * Create enum from string value.
   *
   * @param string $value
   *   String value (sync, create_only, update_only, replace).
   *
   * @return self
   *   The enum case.
   *
   * @throws \ValueError
   *   If value is not a valid mode.
   */
  public static function fromString(string $value): self {
    return match ($value) {
      'sync' => self::Sync,
      'create_only' => self::CreateOnly,
      'update_only' => self::UpdateOnly,
      'replace' => self::Replace,
      default => throw new \ValueError("Invalid change detection mode: $value"),
    };
  }

}
