<?php

namespace Drupal\eb\PluginBase;

use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eb\PluginInterfaces\ValidatorInterface;
use Drupal\eb\Result\ValidationResult;

/**
 * Base class for Entity Builder validator plugins.
 */
abstract class ValidatorBase extends PluginBase implements ValidatorInterface {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  abstract public function validate(array $data): ValidationResult;

  /**
   * Extract operation type from data array.
   *
   * Checks both 'operation_type' and 'operation' keys for consistency,
   * as different parts of the system use different keys.
   *
   * @param array<string, mixed> $data
   *   The operation data.
   *
   * @return string
   *   The operation type, or empty string if not found.
   */
  protected function getOperationType(array $data): string {
    return $data['operation_type'] ?? $data['operation'] ?? '';
  }

  /**
   * Check if the current operation type matches any of the given types.
   *
   * @param array<string, mixed> $data
   *   The operation data.
   * @param array<string> $types
   *   List of operation types to check against.
   *
   * @return bool
   *   TRUE if operation type matches any of the given types.
   */
  protected function isOperationType(array $data, array $types): bool {
    $operation_type = $this->getOperationType($data);
    return in_array($operation_type, $types, TRUE);
  }

}
