<?php

declare(strict_types=1);

namespace Drupal\eb\Result;

/**
 * Value object representing rollback results.
 */
final class RollbackResult {

  /**
   * Constructor.
   *
   * @param bool $successful
   *   Whether rollback was successful.
   * @param array<int, array<string, mixed>> $restoredEntities
   *   Entities that were restored.
   * @param array<int, string> $messages
   *   Informational messages about rollback.
   */
  public function __construct(
    protected bool $successful = FALSE,
    protected array $restoredEntities = [],
    protected array $messages = [],
  ) {}

  /**
   * Checks if rollback was successful.
   */
  public function isSuccessful(): bool {
    return $this->successful;
  }

  /**
   * Gets restored entities.
   *
   * @return array<int, array<string, mixed>>
   *   Array of entity information.
   */
  public function getRestoredEntities(): array {
    return $this->restoredEntities;
  }

  /**
   * Gets rollback messages.
   *
   * @return array<int, string>
   *   Array of messages.
   */
  public function getMessages(): array {
    return $this->messages;
  }

  /**
   * Adds a restored entity.
   *
   * @param array<string, mixed> $entityInfo
   *   Entity information.
   */
  public function addRestoredEntity(array $entityInfo): void {
    $this->restoredEntities[] = $entityInfo;
  }

  /**
   * Adds a message.
   *
   * @param string $message
   *   Informational message.
   */
  public function addMessage(string $message): void {
    $this->messages[] = $message;
  }

  /**
   * Checks if rollback was successful (alias for isSuccessful).
   *
   * @return bool
   *   TRUE if successful.
   */
  public function isSuccess(): bool {
    return $this->isSuccessful();
  }

  /**
   * Gets error messages (alias for getMessages).
   *
   * @return array<int, string>
   *   Error messages.
   */
  public function getErrors(): array {
    return $this->getMessages();
  }

}
