<?php

declare(strict_types=1);

namespace Drupal\eb\Result;

/**
 * Value object representing validation results.
 */
final class ValidationResult {

  /**
   * Constructor.
   *
   * @param array<int, array<string, mixed>> $errors
   *   Array of validation errors. Each error has 'message', 'field', 'code',
   *   and 'severity'.
   * @param array<int, array<string, mixed>> $warnings
   *   Array of validation warnings.
   * @param array<int, string> $messages
   *   Array of informational messages.
   */
  public function __construct(
    protected array $errors = [],
    protected array $warnings = [],
    protected array $messages = [],
  ) {}

  /**
   * Checks if validation passed.
   */
  public function isValid(): bool {
    return empty($this->errors);
  }

  /**
   * Gets validation errors.
   *
   * @return array<int, array<string, mixed>>
   *   Array of error details.
   */
  public function getErrors(): array {
    return $this->errors;
  }

  /**
   * Gets validation warnings.
   *
   * @return array<int, array<string, mixed>>
   *   Array of warning details.
   */
  public function getWarnings(): array {
    return $this->warnings;
  }

  /**
   * Gets informational messages.
   *
   * @return array<int, string>
   *   Array of messages.
   */
  public function getMessages(): array {
    return $this->messages;
  }

  /**
   * Adds an error to the result.
   *
   * @param string $message
   *   Error message.
   * @param string $field
   *   Field that caused the error.
   * @param string $code
   *   Error code.
   * @param string $severity
   *   Severity level (error, warning).
   */
  public function addError(string $message, string $field = '', string $code = '', string $severity = 'error'): void {
    $this->errors[] = [
      'message' => $message,
      'field' => $field,
      'code' => $code,
      'severity' => $severity,
    ];
  }

  /**
   * Adds a warning to the result.
   *
   * @param string $message
   *   Warning message.
   * @param string $field
   *   Field that caused the warning.
   * @param string $code
   *   Warning code.
   */
  public function addWarning(string $message, string $field = '', string $code = ''): void {
    $this->warnings[] = [
      'message' => $message,
      'field' => $field,
      'code' => $code,
      'severity' => 'warning',
    ];
  }

  /**
   * Adds an informational message.
   *
   * @param string $message
   *   Informational message.
   */
  public function addMessage(string $message): void {
    $this->messages[] = $message;
  }

}
