<?php

namespace Drupal\eb\Service;

use Drupal\Core\Session\AccountInterface;
use Drupal\eb\Entity\EbLogInterface;

/**
 * Interface for EbLog manager service.
 */
interface EbLogManagerInterface {

  /**
   * Creates a new log entry.
   *
   * @param array<string, mixed> $fields
   *   Array of field values. Supported fields:
   *   - label: (string) Human-readable label.
   *   - definition_id: (string) Definition entity ID.
   *   - action: (string) apply, rollback, import.
   *   - operation_count: (int) Total operations.
   *   - status: (string) pending, success, partial, failed.
   *   - started: (int) Unix timestamp when action started.
   *
   * @return \Drupal\eb\Entity\EbLogInterface
   *   The created log entity.
   */
  public function log(array $fields): EbLogInterface;

  /**
   * Completes a log entry.
   *
   * @param \Drupal\eb\Entity\EbLogInterface $log
   *   The log entity to complete.
   * @param string $status
   *   The final status (success, partial, failed).
   * @param int $successCount
   *   The number of successful operations.
   * @param int $failureCount
   *   The number of failed operations.
   */
  public function complete(EbLogInterface $log, string $status, int $successCount, int $failureCount): void;

  /**
   * Gets log entries for a definition.
   *
   * @param string $definitionId
   *   The definition entity ID.
   *
   * @return array<int, \Drupal\eb\Entity\EbLogInterface>
   *   Array of log entities.
   */
  public function getByDefinitionId(string $definitionId): array;

  /**
   * Gets a single log entry by ID.
   *
   * @param int $id
   *   The log entity ID.
   *
   * @return \Drupal\eb\Entity\EbLogInterface|null
   *   The log entity or NULL if not found.
   */
  public function getById(int $id): ?EbLogInterface;

  /**
   * Loads log entries with optional filtering.
   *
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   Optional account to filter by user.
   * @param array<string, mixed> $options
   *   Options including:
   *   - limit: (int) Max records to return.
   *   - status: (string) Filter by status.
   *
   * @return array<int, \Drupal\eb\Entity\EbLogInterface>
   *   Array of log entities.
   */
  public function loadLogs(?AccountInterface $account = NULL, array $options = []): array;

  /**
   * Purges old log entries.
   *
   * @param int|null $daysOld
   *   Delete records older than this many days. If NULL, uses config default.
   *
   * @return int
   *   Number of deleted records.
   */
  public function purge(?int $daysOld = NULL): int;

}
