<?php

namespace Drupal\eb\Service;

use Drupal\eb\PluginInterfaces\OperationInterface;
use Drupal\eb\Result\ExecutionResult;

/**
 * Interface for the operation processor service.
 *
 * The operation processor handles execution of operations, including
 * pre/post-execute event dispatching, rollback data storage, error handling,
 * and batch execution.
 */
interface OperationProcessorInterface {

  /**
   * Execute a single operation.
   *
   * This is the main execution method that:
   * 1. Dispatches pre-execute event (allows cancellation)
   * 2. Executes the operation (creates/updates/deletes entities)
   * 3. Stores rollback data if successful (for undo capability)
   * 4. Dispatches post-execute event
   * 5. Handles errors gracefully.
   *
   * @param \Drupal\eb\PluginInterfaces\OperationInterface $operation
   *   The operation to execute.
   * @param array<string, mixed> $context
   *   Optional execution context with keys:
   *   - definition_id: Source definition ID for rollback grouping.
   *
   * @return \Drupal\eb\Result\ExecutionResult
   *   The execution result with success status, messages, and rollback data.
   */
  public function executeOperation(
    OperationInterface $operation,
    array $context = [],
  ): ExecutionResult;

  /**
   * Execute multiple operations in sequence.
   *
   * Processes a batch of operations one by one. Useful for importing
   * multiple entity definitions from a single file.
   *
   * @param array<\Drupal\eb\PluginInterfaces\OperationInterface> $operations
   *   Array of operations to execute.
   * @param bool $stop_on_failure
   *   Whether to stop on first failure. Set to TRUE for atomic operations
   *   where all must succeed or none should be applied. Set to FALSE to
   *   continue processing even if some operations fail.
   * @param array<string, mixed> $context
   *   Optional execution context passed to each operation.
   *   - definition_id: Source definition ID for rollback grouping.
   *
   * @return array<\Drupal\eb\Result\ExecutionResult>
   *   Array of execution results, keyed by operation index.
   */
  public function executeBatch(
    array $operations,
    bool $stop_on_failure = TRUE,
    array $context = [],
  ): array;

}
