<?php

namespace Drupal\eb\Service;

use Drupal\eb\PluginInterfaces\OperationInterface;
use Drupal\eb\Result\ValidationResult;

/**
 * Interface for the validation manager service.
 *
 * The validation manager coordinates validation of operations, performing
 * two-stage validation:
 * 1. Operation's own validation (specific to operation type)
 * 2. All applicable validator plugins (cross-cutting concerns)
 */
interface ValidationManagerInterface {

  /**
   * Validate a single operation.
   *
   * Performs two-stage validation:
   * 1. Operation's own validation (specific to operation type)
   * 2. All applicable validator plugins (cross-cutting concerns)
   *
   * @param \Drupal\eb\PluginInterfaces\OperationInterface $operation
   *   The operation to validate.
   *
   * @return \Drupal\eb\Result\ValidationResult
   *   The validation result containing all errors and warnings.
   */
  public function validateOperation(OperationInterface $operation): ValidationResult;

  /**
   * Validate multiple operations in a batch.
   *
   * Validates operations with awareness of the batch context, allowing
   * validators to check if dependencies will be satisfied by earlier
   * operations in the same batch.
   *
   * @param array<\Drupal\eb\PluginInterfaces\OperationInterface> $operations
   *   Array of operations to validate.
   *
   * @return \Drupal\eb\Result\ValidationResult
   *   Aggregated validation result.
   */
  public function validateBatch(array $operations): ValidationResult;

}
