/**
 * @file
 * Change tracking for EB UI module.
 *
 * Handles tracking of unsaved changes, comparison with initial state,
 * and updating visual indicators for unsaved modifications.
 */

(function (Drupal) {
  'use strict';

  /**
   * Initialize the ebUi namespace if not exists.
   */
  Drupal.ebAggrid = Drupal.ebAggrid || {};

  /**
   * Storage for initial data snapshot for change tracking.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.initialData = {};

  /**
   * Cache for change summary to avoid redundant calculations.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.changeCache = {
    summary: null,
    dirty: true
  };

  /**
   * Mark change cache as dirty (needs recalculation).
   */
  Drupal.ebAggrid.markChangeCacheDirty = () => {
    Drupal.ebAggrid.changeCache.dirty = true;
    Drupal.ebAggrid.changeCache.summary = null;
  };

  /**
   * Store initial row data for change tracking.
   *
   * @param {Object} gridApi
   *   The AG-Grid API instance.
   * @param {string} gridType
   *   The type of grid.
   */
  Drupal.ebAggrid.storeInitialData = (gridApi, gridType) => {
    Drupal.ebAggrid.initialData[gridType] = {};

    gridApi.forEachNode((node) => {
      if (node.data) {
        // Deep clone the data to avoid reference issues.
        const clonedData = JSON.parse(JSON.stringify(node.data));
        // Remove internal fields from initial snapshot.
        Drupal.ebAggrid.INTERNAL_FIELDS.forEach((field) => {
          delete clonedData[field];
        });
        Drupal.ebAggrid.initialData[gridType][node.id] = clonedData;
      }
    });
  };

  /**
   * Compare current row data with initial data (excluding internal fields).
   *
   * @param {Object} currentData
   *   Current row data.
   * @param {Object} initialData
   *   Initial row data.
   *
   * @return {boolean}
   *   True if data has changed.
   */
  Drupal.ebAggrid.hasRowChanged = (currentData, initialData) => {
    // Clone and remove internal fields for comparison.
    const currentCopy = { ...currentData };
    const initialCopy = { ...initialData };

    Drupal.ebAggrid.INTERNAL_FIELDS.forEach((field) => {
      delete currentCopy[field];
      delete initialCopy[field];
    });

    return JSON.stringify(initialCopy) !== JSON.stringify(currentCopy);
  };

  /**
   * Check if there are any unsaved changes across all grids.
   *
   * Uses cached result if available and not dirty.
   *
   * @return {boolean}
   *   True if there are unsaved changes.
   */
  Drupal.ebAggrid.hasUnsavedChanges = () => {
    // Use cached summary if available.
    if (!Drupal.ebAggrid.changeCache.dirty && Drupal.ebAggrid.changeCache.summary !== null) {
      return Object.keys(Drupal.ebAggrid.changeCache.summary).length > 0;
    }

    let hasChanges = false;

    Object.keys(Drupal.ebAggrid.grids).forEach((gridType) => {
      const gridApi = Drupal.ebAggrid.grids[gridType];
      if (!gridApi) {
        return;
      }

      const initialData = Drupal.ebAggrid.initialData[gridType] || {};

      gridApi.forEachNode((node) => {
        if (!node.data) {
          return;
        }

        const rowId = node.id;
        const initial = initialData[rowId];

        // New row (no initial data) = unsaved change.
        if (!initial) {
          hasChanges = true;
          return;
        }

        // Compare current data with initial.
        if (Drupal.ebAggrid.hasRowChanged(node.data, initial)) {
          hasChanges = true;
        }
      });

      // Also check if rows were deleted.
      let currentRowCount = 0;
      gridApi.forEachNode(() => {
        currentRowCount++;
      });
      const initialRowCount = Object.keys(initialData).length;
      if (currentRowCount !== initialRowCount) {
        hasChanges = true;
      }
    });

    return hasChanges;
  };

  /**
   * Get unsaved changes summary by grid type.
   *
   * Uses cached result if available and not dirty.
   *
   * @return {Object}
   *   Object with gridType keys and change counts.
   */
  Drupal.ebAggrid.getUnsavedChangesSummary = () => {
    // Return cached summary if not dirty.
    if (!Drupal.ebAggrid.changeCache.dirty && Drupal.ebAggrid.changeCache.summary !== null) {
      return Drupal.ebAggrid.changeCache.summary;
    }

    const summary = {};

    Object.keys(Drupal.ebAggrid.grids).forEach((gridType) => {
      const gridApi = Drupal.ebAggrid.grids[gridType];
      if (!gridApi) {
        return;
      }

      const initialData = Drupal.ebAggrid.initialData[gridType] || {};
      const changes = {
        added: 0,
        modified: 0,
        deleted: 0
      };

      const currentRowIds = [];

      gridApi.forEachNode((node) => {
        if (!node.data) {
          return;
        }

        const rowId = node.id;
        currentRowIds.push(rowId);
        const initial = initialData[rowId];

        if (!initial) {
          changes.added++;
          return;
        }

        if (Drupal.ebAggrid.hasRowChanged(node.data, initial)) {
          changes.modified++;
        }
      });

      // Count deleted rows.
      Object.keys(initialData).forEach((rowId) => {
        if (!currentRowIds.includes(rowId)) {
          changes.deleted++;
        }
      });

      const total = changes.added + changes.modified + changes.deleted;
      if (total > 0) {
        summary[gridType] = changes;
      }
    });

    // Cache the result.
    Drupal.ebAggrid.changeCache.summary = summary;
    Drupal.ebAggrid.changeCache.dirty = false;

    return summary;
  };

  /**
   * Update tab modified indicators (shows unsaved count under each tab).
   */
  Drupal.ebAggrid.updateTabModifiedIndicators = () => {
    const tabGridMap = Drupal.ebAggrid.TAB_GRID_MAP;
    const summary = Drupal.ebAggrid.getUnsavedChangesSummary();

    Object.keys(tabGridMap).forEach((tabId) => {
      const gridType = tabGridMap[tabId];
      const tab = document.querySelector(`.eb-sheet-tab[data-tab="${tabId}"]`);
      if (!tab) {
        return;
      }

      // Find or create the unsaved text element.
      let unsavedText = tab.querySelector('.eb-sheet-tab__unsaved');

      if (summary[gridType]) {
        tab.classList.add('eb-sheet-tab--has-changes');
        const changes = summary[gridType];
        const total = changes.added + changes.modified + changes.deleted;

        if (!unsavedText) {
          unsavedText = document.createElement('span');
          unsavedText.className = 'eb-sheet-tab__unsaved';
          tab.appendChild(unsavedText);
        }

        // Show "X unsaved" text.
        unsavedText.textContent = Drupal.t('@count unsaved', { '@count': total });
        unsavedText.title = Drupal.t('@added added, @modified modified, @deleted deleted', {
          '@added': changes.added,
          '@modified': changes.modified,
          '@deleted': changes.deleted
        });
      }
      else {
        tab.classList.remove('eb-sheet-tab--has-changes');
        if (unsavedText) {
          unsavedText.remove();
        }
      }
    });
  };

  /**
   * Update unsaved changes indicator near save button.
   */
  Drupal.ebAggrid.updateSaveButtonIndicator = () => {
    const saveButton = document.querySelector('#edit-save, input[name="save"]');
    const applyButton = document.querySelector('#edit-apply, input[name="apply"]');
    let indicator = document.getElementById('eb-unsaved-indicator');

    const hasChanges = Drupal.ebAggrid.hasUnsavedChanges();

    if (hasChanges) {
      // Add visual emphasis to save buttons.
      if (saveButton) {
        saveButton.classList.add('eb-btn--has-changes');
      }
      if (applyButton) {
        applyButton.classList.add('eb-btn--has-changes');
      }

      // Show or create indicator.
      if (!indicator) {
        const actionsWrapper = document.querySelector('.form-actions, #edit-actions');
        if (actionsWrapper) {
          indicator = document.createElement('span');
          indicator.id = 'eb-unsaved-indicator';
          indicator.className = 'eb-unsaved-indicator';
          indicator.innerHTML = `<span class="eb-unsaved-indicator__dot"></span><span class="eb-unsaved-indicator__text">${Drupal.t('Unsaved changes')}</span>`;
          actionsWrapper.insertBefore(indicator, actionsWrapper.firstChild);
        }
      }
      if (indicator) {
        indicator.style.display = '';
      }
    }
    else {
      // Remove visual emphasis.
      if (saveButton) {
        saveButton.classList.remove('eb-btn--has-changes');
      }
      if (applyButton) {
        applyButton.classList.remove('eb-btn--has-changes');
      }

      // Hide indicator.
      if (indicator) {
        indicator.style.display = 'none';
      }
    }
  };

  /**
   * Update all unsaved change indicators.
   */
  Drupal.ebAggrid.updateUnsavedIndicators = () => {
    Drupal.ebAggrid.updateTabModifiedIndicators();
    Drupal.ebAggrid.updateSaveButtonIndicator();
  };

  /**
   * Update row status indicator based on validation and change state.
   *
   * @param {Object} gridApi
   *   The AG-Grid API.
   * @param {Object} node
   *   The row node.
   * @param {string} gridType
   *   The grid type.
   */
  Drupal.ebAggrid.updateRowStatus = (gridApi, node, gridType) => {
    if (!node?.data) {
      return;
    }

    // Get gridType from parameter, or try to extract from gridApi context.
    if (!gridType) {
      gridType = gridApi.gridOptions?.context?.gridType;
    }
    if (!gridType) {
      return;
    }

    let status = Drupal.ebAggrid.ROW_STATUS.VALID;
    const rowId = node.id;

    // Check for validation errors first (highest priority).
    const clientErrors = Drupal.ebAggrid.validationErrors[gridType];
    if (clientErrors?.[rowId] && Object.keys(clientErrors[rowId]).length > 0) {
      status = Drupal.ebAggrid.ROW_STATUS.ERROR;
    }

    // If no errors, check if row is modified or new.
    if (status === Drupal.ebAggrid.ROW_STATUS.VALID) {
      const initialData = Drupal.ebAggrid.initialData[gridType];
      const initial = initialData?.[rowId];

      if (!initial) {
        // New row (no initial data) - show as modified/new.
        status = Drupal.ebAggrid.ROW_STATUS.MODIFIED;
      }
      else if (Drupal.ebAggrid.hasRowChanged(node.data, initial)) {
        // Existing row modified from initial state.
        status = Drupal.ebAggrid.ROW_STATUS.MODIFIED;
      }
    }

    // Update status in row data.
    node.data._status = status;
    gridApi.refreshCells({
      rowNodes: [node],
      columns: ['_status'],
      force: true
    });
  };

})(Drupal);
