/**
 * @file
 * AG-Grid column definitions for EB UI.
 *
 * This file defines column configurations for all five definition types:
 * - Bundle definitions
 * - Field definitions
 * - Field group definitions
 * - Display field definitions
 * - Menu definitions
 *
 * Depends on eb_ui.columns.renderers.js for shared cell renderers,
 * value handlers, and label formatters.
 */

(function (Drupal) {
  'use strict';

  /**
   * Initialize the ebUi namespace if not exists.
   */
  Drupal.ebAggrid = Drupal.ebAggrid || {};

  /**
   * Initialize bundle labels from bundleDefinitions into discovery data.
   *
   * This ensures bundle labels from the Bundles tab are available
   * for the Target column formatter on initial page load.
   */
  Drupal.ebAggrid.initBundleLabelsFromDefinitions = () => {
    const settings = drupalSettings.ebAggrid || {};
    const discovery = settings.discovery || {};
    const bundleDefinitions = settings.bundleDefinitions || [];

    if (!discovery.bundleLabels) {
      discovery.bundleLabels = {};
    }

    bundleDefinitions.forEach((def) => {
      const { entity_type: entityType, bundle_id: bundle, label } = def;
      if (entityType && bundle && label) {
        if (!discovery.bundleLabels[entityType]) {
          discovery.bundleLabels[entityType] = {};
        }
        // Only set if not already present (Drupal bundles take precedence).
        if (!discovery.bundleLabels[entityType][bundle]) {
          discovery.bundleLabels[entityType][bundle] = label;
        }
      }
    });
  };

  // Initialize bundle labels immediately when this file loads.
  if (typeof drupalSettings !== 'undefined' && drupalSettings.ebAggrid) {
    Drupal.ebAggrid.initBundleLabelsFromDefinitions();
  }

  // ============================================
  // BUNDLE ID AUTOGENERATION
  // ============================================

  /**
   * Generate a valid Drupal machine name from a label.
   *
   * Converts label to valid machine name:
   * - Lowercase
   * - Replace non-alphanumeric with underscores
   * - Remove consecutive underscores
   * - Prepend 'type_' if starts with number
   * - Truncate to 32 characters
   *
   * @param {string} label
   *   The human-readable label.
   *
   * @return {string}
   *   A valid Drupal machine name.
   */
  Drupal.ebAggrid.generateBundleId = (label) => {
    if (!label) {
      return '';
    }

    return label
      .toLowerCase()
      .replace(/[^a-z0-9]+/g, '_')
      .replace(/^_+|_+$/g, '')
      .replace(/_+/g, '_')
      .replace(/^(\d)/, 'type_$1')
      .substring(0, 32);
  };

  /**
   * Autogenerate Bundle ID from Label column value.
   *
   * @param {Object} params
   *   AG-Grid onCellValueChanged params.
   */
  Drupal.ebAggrid.autogenerateBundleId = (params) => {
    if (params.column.colId !== 'label') {
      return;
    }

    const { data } = params;
    const newLabel = params.newValue || '';

    // Skip if user has manually edited bundle_id.
    if (data._bundleIdManuallyEdited) {
      return;
    }

    // Generate new bundle_id from label.
    const generatedId = Drupal.ebAggrid.generateBundleId(newLabel);

    // Track the previous auto-generated value.
    const previousAutoId = data._previousAutoBundleId || '';

    // Only update if bundle_id is empty or matches the previous auto-generated value.
    if (!data.bundle_id || data.bundle_id === previousAutoId) {
      params.node.setDataValue('bundle_id', generatedId);
      data._previousAutoBundleId = generatedId;

      // Refresh the cell to update styling.
      params.api.refreshCells({
        rowNodes: [params.node],
        columns: ['bundle_id'],
        force: true
      });
    }
  };

  /**
   * Handle Bundle ID manual edit to stop autogeneration.
   *
   * @param {Object} params
   *   AG-Grid onCellValueChanged params.
   */
  Drupal.ebAggrid.handleBundleIdManualEdit = (params) => {
    if (params.column.colId !== 'bundle_id') {
      return;
    }

    // If user manually edited, mark as manually edited.
    // Unless the value matches what would be auto-generated from current label.
    const currentLabel = params.data.label || '';
    const expectedAutoId = Drupal.ebAggrid.generateBundleId(currentLabel);

    if (params.newValue !== expectedAutoId) {
      params.data._bundleIdManuallyEdited = true;
    }
  };

  // ============================================
  // FIELD NAME AUTOGENERATION
  // ============================================

  /**
   * Generate a valid Drupal field machine name from a label.
   *
   * Converts label to valid field name:
   * - Lowercase
   * - Replace non-alphanumeric with underscores
   * - Remove consecutive underscores
   * - Prepend 'f' if starts with number (after field_ prefix)
   * - Always prefix with 'field_'
   * - Truncate to 32 characters total
   *
   * @param {string} label
   *   The human-readable label.
   *
   * @return {string}
   *   A valid Drupal field machine name with field_ prefix.
   */
  Drupal.ebAggrid.generateFieldName = (label) => {
    if (!label) {
      return 'field_';
    }

    const machineId = label
      .toLowerCase()
      .replace(/[^a-z0-9]+/g, '_')
      .replace(/^_+|_+$/g, '')
      .replace(/_+/g, '_')
      .replace(/^(\d)/, 'f$1');

    // Ensure max length including 'field_' prefix (32 - 6 = 26).
    return `field_${machineId.substring(0, 26)}`;
  };

  /**
   * Handle Label column value change for Field Name autogeneration.
   *
   * @param {Object} params
   *   AG-Grid onCellValueChanged params.
   */
  Drupal.ebAggrid.handleLabelChangeForFieldName = (params) => {
    if (params.column.colId !== 'label') {
      return;
    }

    const { data } = params;
    const newLabel = params.newValue || '';

    // Skip if user has manually edited field_name.
    if (data._fieldNameManuallyEdited) {
      return;
    }

    // Generate new field_name from label.
    const generatedName = Drupal.ebAggrid.generateFieldName(newLabel);

    // Track the previous auto-generated value.
    const previousAutoName = data._previousAutoFieldName || 'field_';

    // Only update if field_name is empty, 'field_', or matches previous auto-value.
    if (!data.field_name || data.field_name === 'field_' || data.field_name === previousAutoName) {
      params.node.setDataValue('field_name', generatedName);
      data._previousAutoFieldName = generatedName;

      // Refresh the cell to update styling.
      params.api.refreshCells({
        rowNodes: [params.node],
        columns: ['field_name'],
        force: true
      });
    }
  };

  /**
   * Handle Field Name manual edit to stop autogeneration.
   *
   * @param {Object} params
   *   AG-Grid onCellValueChanged params.
   */
  Drupal.ebAggrid.handleFieldNameManualEdit = (params) => {
    if (params.column.colId !== 'field_name') {
      return;
    }

    // If user manually edited, mark as manually edited.
    // Unless the value matches what would be auto-generated from current label.
    const currentLabel = params.data.label || '';
    const expectedAutoName = Drupal.ebAggrid.generateFieldName(currentLabel);

    if (params.newValue !== expectedAutoName) {
      params.data._fieldNameManuallyEdited = true;
    }
  };

  // ============================================
  // COMBINED TARGET COLUMN (Entity Type + Bundle)
  // ============================================

  /**
   * Create a combined Entity Type + Bundle column (Target).
   *
   * Uses flat list dropdown showing "entity_type:bundle" values
   * with formatter displaying "Bundle Label (Entity Type Label)".
   * Helper functions are in eb_ui.columns.renderers.js.
   *
   * @return {Object}
   *   Column definition.
   */
  Drupal.ebAggrid.createCombinedTargetColumn = () => Drupal.ebAggrid.markRequired({
    field: '_target',
    headerName: 'Target',
    editable: true,
    cellEditor: 'agSelectCellEditor',
    cellEditorParams: () => ({ values: Drupal.ebAggrid.getTargetValuesFormatted() }),
    valueFormatter: Drupal.ebAggrid.targetLabelFormatter,
    valueGetter: (params) => {
      // Return formatted label for display and dropdown matching.
      const et = params.data.entity_type || '';
      const b = params.data.bundle || '';
      if (!et || !b) {
        return '';
      }
      const discovery = drupalSettings.ebAggrid?.discovery || {};
      const entityLabel = (discovery.entityTypeLabels || {})[et] || et;
      const bundleLabel = ((discovery.bundleLabels || {})[et] || {})[b] || b;
      return `${bundleLabel} (${entityLabel})`;
    },
    valueSetter: (params) => {
      // Parse "Bundle Label (Entity Type Label)" back to entity_type and bundle.
      const selected = params.newValue || '';
      if (!selected) {
        params.data.entity_type = '';
        params.data.bundle = '';
        return true;
      }
      // Look up the raw value from our mapping.
      const mapping = Drupal.ebAggrid.getTargetLabelToValueMap();
      const rawValue = mapping[selected];
      if (rawValue) {
        const parts = rawValue.split(':');
        params.data.entity_type = parts[0] || '';
        params.data.bundle = parts[1] || '';
      }
      return true;
    },
    width: Drupal.ebAggrid.COLUMN_WIDTHS.XLARGE || 280
  });

  // ============================================
  // FIELD TYPE CHANGE HANDLERS
  // ============================================

  /**
   * Handle field type change to populate default settings.
   *
   * @param {Object} params
   *   AG-Grid onCellValueChanged params.
   */
  Drupal.ebAggrid.handleFieldTypeChange = (params) => {
    if (params.column.colId !== 'field_type') {
      return;
    }

    const fieldType = params.newValue;
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    const defaultSettings = discovery.fieldTypeDefaultSettings || {};
    const typeSettings = defaultSettings[fieldType] || {};

    // Only set if storage settings is empty or null.
    if (!params.data.field_storage_settings || Object.keys(params.data.field_storage_settings).length === 0) {
      const storageDefaults = typeSettings.storage || {};
      if (Object.keys(storageDefaults).length > 0) {
        params.node.setDataValue('field_storage_settings', storageDefaults);
      }
    }

    // Only set if config settings is empty or null.
    if (!params.data.field_config_settings || Object.keys(params.data.field_config_settings).length === 0) {
      const configDefaults = typeSettings.config || {};
      if (Object.keys(configDefaults).length > 0) {
        params.node.setDataValue('field_config_settings', configDefaults);
      }
    }

    // Populate default widget and formatter for the new field type.
    Drupal.ebAggrid.populateDefaultWidgetFormatter(params, fieldType);
  };

  /**
   * Handle widget change to populate default widget settings.
   *
   * @param {Object} params
   *   AG-Grid onCellValueChanged params.
   */
  Drupal.ebAggrid.handleWidgetChange = (params) => {
    if (params.column.colId !== 'widget') {
      return;
    }

    const widget = params.newValue;
    // Populate widget settings with defaults from the new widget.
    Drupal.ebAggrid.populateWidgetSettings(params, widget);
  };

  /**
   * Handle formatter change to populate default formatter settings.
   *
   * @param {Object} params
   *   AG-Grid onCellValueChanged params.
   */
  Drupal.ebAggrid.handleFormatterChange = (params) => {
    if (params.column.colId !== 'formatter') {
      return;
    }

    const formatter = params.newValue;
    // Populate formatter settings with defaults from the new formatter.
    Drupal.ebAggrid.populateFormatterSettings(params, formatter);
  };

  // ============================================
  // COLUMN FACTORY HELPERS
  // ============================================

  /**
   * Mark a column as required by adding an asterisk to the header.
   *
   * @param {Object} colDef
   *   The column definition.
   *
   * @return {Object}
   *   The modified column definition.
   */
  Drupal.ebAggrid.markRequired = (colDef) => {
    colDef.headerName = `${colDef.headerName} *`;
    colDef.headerClass = 'eb-header-required';
    return colDef;
  };

  /**
   * Create an actions column definition.
   *
   * @return {Object}
   *   Actions column definition.
   */
  Drupal.ebAggrid.createActionsColumn = () => ({
    headerName: '',
    field: '_actions',
    width: Drupal.ebAggrid.COLUMN_WIDTHS.ACTION,
    pinned: 'right',
    cellRenderer: Drupal.ebAggrid.cellRenderers.deleteButton,
    editable: false,
    sortable: false,
    filter: false
  });

  /**
   * Create a JSON settings column definition.
   *
   * @param {string} field
   *   The field name.
   * @param {string} headerName
   *   The header display name.
   * @param {Object} options
   *   Optional configuration.
   *
   * @return {Object}
   *   Column definition.
   */
  Drupal.ebAggrid.createJsonColumn = (field, headerName, options = {}) => ({
    field,
    headerName,
    editable: true,
    cellEditor: 'agLargeTextCellEditor',
    cellEditorPopup: true,
    cellEditorParams: {
      maxLength: options.maxLength || 5000,
      rows: options.rows || 12,
      cols: 60
    },
    // Custom cell renderer showing friendly summary.
    cellRenderer: (params) => {
      const value = params.data[field];
      if (!value || typeof value !== 'object') {
        return '<span class="eb-settings-empty">Click to add</span>';
      }
      const keys = Object.keys(value);
      if (keys.length === 0) {
        return '<span class="eb-settings-empty">Click to add</span>';
      }
      // Show count and first key preview.
      const preview = keys.slice(0, 2).map((key) => {
        let val = value[key];
        if (typeof val === 'boolean') {
          val = val ? '✓' : '✗';
        }
        else if (typeof val === 'object') {
          val = '{...}';
        }
        else if (typeof val === 'string' && val.length > 15) {
          val = `${val.substring(0, 12)}...`;
        }
        return `<span class="eb-setting-key">${key}</span>: ${val}`;
      }).join(', ');
      const more = keys.length > 2 ? ` <span class="eb-settings-more">+${keys.length - 2} more</span>` : '';
      return `<span class="eb-settings-preview">${preview}${more}</span>`;
    },
    // Tooltip showing full JSON on hover.
    tooltipValueGetter: (params) => {
      const value = params.data[field];
      if (!value || typeof value !== 'object' || Object.keys(value).length === 0) {
        return 'Click to add settings';
      }
      return JSON.stringify(value, null, 2);
    },
    // Value getter returns JSON string for the editor.
    valueGetter: (params) => {
      const value = params.data[field];
      if (value && typeof value === 'object') {
        return JSON.stringify(value, null, 2);
      }
      return value || '';
    },
    // Value setter parses JSON string back to object.
    valueSetter: (params) => {
      let { newValue } = params;
      if (newValue && typeof newValue === 'string') {
        newValue = newValue.trim();
        if (newValue === '') {
          params.data[field] = {};
          return true;
        }
        try {
          params.data[field] = JSON.parse(newValue);
        }
        catch (e) {
          // Keep existing value if JSON is invalid.
          console.warn(`Invalid JSON in ${field}:`, e.message);
          return false;
        }
      }
      else {
        params.data[field] = newValue || {};
      }
      return true;
    },
    width: options.width || Drupal.ebAggrid.COLUMN_WIDTHS.LARGE,
    ...(options.extra || {})
  });

  /**
   * Create a Settings column with Configure button.
   *
   * Replaces the old JSON columns with a user-friendly modal interface.
   *
   * @return {Object}
   *   Column definition.
   */
  Drupal.ebAggrid.createSettingsColumn = () => ({
    headerName: 'Settings',
    field: '_settings',
    headerTooltip: 'Click to configure field settings.',
    width: 130,
    editable: false,
    sortable: false,
    filter: false,
    cellRenderer: (params) => {
      const hasStorage = params.data.field_storage_settings &&
                       Object.keys(params.data.field_storage_settings).length > 0;
      const hasConfig = params.data.field_config_settings &&
                      Object.keys(params.data.field_config_settings).length > 0;

      // Count configured sections.
      const count = (hasStorage ? 1 : 0) + (hasConfig ? 1 : 0);
      const label = count > 0 ? Drupal.t('Configured (@count)', { '@count': count }) : Drupal.t('Configure');
      const cssClass = count > 0 ? 'eb-settings-btn eb-settings-btn--has-values' : 'eb-settings-btn';

      const button = document.createElement('button');
      button.type = 'button';
      button.className = cssClass;
      button.textContent = label;
      button.title = count > 0 ? Drupal.t('Click to modify settings') : Drupal.t('Click to add settings');

      return button;
    },
    onCellClicked: (params) => {
      // Prevent editing, open modal instead.
      if (Drupal.ebAggrid.openSettingsModal) {
        Drupal.ebAggrid.openSettingsModal(params.node, params.data);
      }
    }
  });

  /**
   * Create a Format Settings column with Configure button.
   *
   * Used in the Field Groups tab to configure format_settings via modal.
   *
   * @return {Object}
   *   Column definition.
   */
  Drupal.ebAggrid.createFormatSettingsColumn = () => ({
    headerName: 'Format Settings',
    field: '_format_settings',
    headerTooltip: 'Click to configure format settings for this field group.',
    width: 150,
    flex: 1,
    editable: false,
    sortable: false,
    filter: false,
    cellRenderer: (params) => {
      const formatSettings = params.data.format_settings;
      const hasSettings = formatSettings &&
                         typeof formatSettings === 'object' &&
                         Object.keys(formatSettings).length > 0;

      // Count configured settings.
      const count = hasSettings ? Object.keys(formatSettings).length : 0;
      const label = count > 0 ? Drupal.t('Configured (@count)', { '@count': count }) : Drupal.t('Configure');
      const cssClass = count > 0 ? 'eb-settings-btn eb-settings-btn--has-values' : 'eb-settings-btn';

      const button = document.createElement('button');
      button.type = 'button';
      button.className = cssClass;
      button.textContent = label;
      button.title = count > 0 ? Drupal.t('Click to modify format settings') : Drupal.t('Click to add format settings');

      return button;
    },
    onCellClicked: (params) => {
      // Prevent editing, open modal instead.
      if (Drupal.ebAggrid.openFormatSettingsModal) {
        Drupal.ebAggrid.openFormatSettingsModal(params.node, params.data);
      }
    }
  });

  /**
   * Create an entity type select column.
   *
   * @param {Array} entityTypes
   *   Available entity types.
   * @param {boolean} clearBundle
   *   Whether to clear bundle on change.
   *
   * @return {Object}
   *   Column definition.
   */
  Drupal.ebAggrid.createEntityTypeColumn = (entityTypes, clearBundle) => {
    const col = Drupal.ebAggrid.markRequired({
      field: 'entity_type',
      headerName: 'Entity Type',
      editable: true,
      cellEditor: 'agSelectCellEditor',
      cellEditorParams: {
        values: entityTypes || []
      },
      valueFormatter: Drupal.ebAggrid.entityTypeLabelFormatter,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.LARGE
    });

    if (clearBundle) {
      col.onCellValueChanged = (params) => {
        if (params.column.colId === 'entity_type') {
          params.node.setDataValue('bundle', '');
        }
      };
    }

    return col;
  };

  /**
   * Create a bundle select column.
   *
   * @return {Object}
   *   Column definition.
   */
  Drupal.ebAggrid.createBundleColumn = () => Drupal.ebAggrid.markRequired({
    field: 'bundle',
    headerName: 'Bundle',
    editable: true,
    cellEditor: 'agSelectCellEditor',
    cellEditorParams: (params) => {
      const entityType = params.data.entity_type;
      return Drupal.ebAggrid.getBundleEditorParams(entityType);
    },
    width: Drupal.ebAggrid.COLUMN_WIDTHS.MEDIUM
  });

  // ============================================
  // STANDARD COLUMN DEFINITIONS
  // ============================================

  /**
   * Get status column definition.
   *
   * @return {Object}
   *   Status column definition.
   */
  Drupal.ebAggrid.getStatusColumn = () => ({
    headerName: '',
    field: '_status',
    width: Drupal.ebAggrid.COLUMN_WIDTHS.ICON,
    pinned: 'left',
    editable: false,
    sortable: false,
    filter: false,
    cellRenderer: Drupal.ebAggrid.cellRenderers.statusIcon
  });

  /**
   * Get row number column definition.
   *
   * Shows total row count in header instead of "#".
   *
   * @return {Object}
   *   Row number column definition.
   */
  Drupal.ebAggrid.getRowNumberColumn = () => ({
    headerName: '#',
    headerValueGetter: (params) => {
      let count = 0;
      if (params.api) {
        params.api.forEachNode(() => count++);
      }
      return count > 0 ? `${count} rows` : '#';
    },
    headerClass: 'eb-row-count-header',
    headerTooltip: Drupal.t('Total rows in this tab'),
    field: '_rowNum',
    width: 80,
    minWidth: 70,
    pinned: 'left',
    editable: false,
    sortable: false,
    filter: false,
    suppressSizeToFit: true,
    valueGetter: (params) => params.node.rowIndex !== null ? params.node.rowIndex + 1 : '',
    cellClass: 'eb-row-number'
  });

  /**
   * Get drag handle column definition.
   *
   * @return {Object}
   *   Drag handle column definition.
   */
  Drupal.ebAggrid.getDragColumn = () => ({
    headerName: '',
    field: '_drag',
    width: Drupal.ebAggrid.COLUMN_WIDTHS.ICON,
    pinned: 'left',
    rowDrag: true,
    editable: false,
    sortable: false,
    filter: false,
    cellRenderer: Drupal.ebAggrid.cellRenderers.dragHandle
  });

  // ============================================
  // EXTENSION COLUMN HELPERS
  // ============================================

  /**
   * Create a column definition for an extension feature.
   *
   * When the extension module is disabled, the column is shown but
   * non-editable with a visual indicator and tooltip explaining the
   * required module.
   *
   * @param {Object} config
   *   Base column configuration.
   * @param {string} extensionKey
   *   The extension key in drupalSettings.ebAggrid.extensions (e.g., 'pathauto').
   * @param {string} moduleName
   *   Human-readable module name for tooltip (e.g., 'eb_pathauto').
   *
   * @return {Object}
   *   AG-Grid column definition with extension-aware behavior.
   */
  Drupal.ebAggrid.createExtensionColumn = (config, extensionKey, moduleName) => {
    const ext = drupalSettings.ebAggrid?.extensions?.[extensionKey];
    const enabled = ext?.enabled ?? false;

    // Build cell class - can be string, array, or function.
    const baseCellClass = config.cellClass;
    const cellClass = (params) => {
      const classes = [];

      // Add base cell class if it exists.
      if (baseCellClass) {
        if (typeof baseCellClass === 'function') {
          const result = baseCellClass(params);
          if (Array.isArray(result)) {
            classes.push(...result);
          }
          else if (result) {
            classes.push(result);
          }
        }
        else if (Array.isArray(baseCellClass)) {
          classes.push(...baseCellClass);
        }
        else {
          classes.push(baseCellClass);
        }
      }

      // Add disabled class if extension not enabled.
      if (!enabled) {
        classes.push('eb-extension-disabled');
      }

      return classes;
    };

    // Build tooltip for disabled cells.
    const disabledTooltip = Drupal.t('Install @module module to use this feature', { '@module': moduleName });

    // Tooltip getter for cells - shows validation errors first, then disabled message.
    // Must check validation errors to maintain consistency with other columns.
    const tooltipValueGetter = (params) => {
      // First check for validation errors (same as defaultColDef behavior).
      const errorMessage = Drupal.ebAggrid.getCellErrorMessage?.(params);
      if (errorMessage) {
        return errorMessage;
      }

      // If extension disabled, show the disabled tooltip.
      if (!enabled) {
        return disabledTooltip;
      }

      // Fall back to any configured tooltipValueGetter or null.
      if (config.tooltipValueGetter) {
        return typeof config.tooltipValueGetter === 'function'
          ? config.tooltipValueGetter(params)
          : config.tooltipValueGetter;
      }

      return null;
    };

    return {
      ...config,
      editable: enabled && (config.editable !== false),
      cellClass,
      tooltipValueGetter,
      headerTooltip: enabled
        ? (config.headerTooltip || null)
        : disabledTooltip
    };
  };

  // ============================================
  // GRID-SPECIFIC COLUMN DEFINITIONS
  // ============================================

  /**
   * Get bundle definition columns.
   *
   * Column order: Row#, Status, Entity Type, Label, Bundle ID, Description, ...
   * Label comes before Bundle ID to match the natural mental model.
   *
   * @param {Object} discovery
   *   Discovery data from drupalSettings.
   *
   * @return {Array}
   *   Array of AG-Grid column definitions.
   */
  Drupal.ebAggrid.getBundleColumns = (discovery) => [
    Drupal.ebAggrid.getRowNumberColumn(),
    Drupal.ebAggrid.getStatusColumn(),
    Drupal.ebAggrid.createEntityTypeColumn(discovery.entityTypes, false),
    // Label column - positioned before Bundle ID for natural flow.
    Drupal.ebAggrid.markRequired({
      field: 'label',
      headerName: 'Label',
      editable: true,
      width: 250,
      cellEditorParams: {
        placeholder: Drupal.t('Enter human-readable name')
      },
      onCellValueChanged: Drupal.ebAggrid.autogenerateBundleId
    }),
    // Bundle ID column - auto-generated from Label.
    Drupal.ebAggrid.markRequired({
      field: 'bundle_id',
      headerName: 'Bundle ID',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.STANDARD,
      onCellValueChanged: Drupal.ebAggrid.handleBundleIdManualEdit,
      cellClass: (params) => {
        const classes = [];
        // Add autogenerated indicator if not manually edited.
        if (params.data && !params.data._bundleIdManuallyEdited && params.data._previousAutoBundleId) {
          classes.push('eb-cell-autogenerated');
        }
        return classes;
      },
      tooltipValueGetter: (params) => {
        if (params.data && !params.data._bundleIdManuallyEdited && params.data._previousAutoBundleId) {
          return Drupal.t('Auto-generated from Label. Edit to override.');
        }
        return null;
      }
    }),
    {
      field: 'description',
      headerName: 'Description',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.XLARGE,
      flex: 1
    },
    // Extension columns - shown but disabled when module not enabled.
    Drupal.ebAggrid.createExtensionColumn({
      field: 'pathauto_pattern',
      headerName: 'Pathauto Pattern',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.LARGE
    }, 'pathauto', 'eb_pathauto'),
    Drupal.ebAggrid.createExtensionColumn({
      field: 'auto_entitylabel_status',
      headerName: 'Auto Label Status',
      editable: true,
      cellEditor: 'agSelectCellEditor',
      cellEditorParams: {
        values: ['', 'enabled', 'disabled', 'optional', 'prefilled']
      },
      width: 140
    }, 'auto_entitylabel', 'eb_auto_entitylabel'),
    Drupal.ebAggrid.createExtensionColumn({
      field: 'auto_entitylabel_pattern',
      headerName: 'Auto Label Pattern',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.LARGE
    }, 'auto_entitylabel', 'eb_auto_entitylabel'),
    Drupal.ebAggrid.createActionsColumn()
  ];

  /**
   * Create Widget column group for Fields tab.
   *
   * @param {Object} discovery
   *   Discovery data from drupalSettings.
   *
   * @return {Object}
   *   AG-Grid column group definition.
   */
  Drupal.ebAggrid.createWidgetColumnGroup = (discovery) => ({
    headerName: 'Widget (Form)',
    marryChildren: true,
    openByDefault: false,
    children: [
      {
        field: 'widget',
        headerName: 'Type',
        editable: true,
        cellEditor: 'agSelectCellEditor',
        cellEditorParams: (params) => {
          const fieldType = params.data?.field_type;
          const widgetsForType = fieldType
            ? (discovery.widgetsByFieldType?.[fieldType] || [])
            : [];
          return { values: ['', ...widgetsForType] };
        },
        valueFormatter: (params) => {
          if (!params.value) return '';
          return discovery.widgets?.[params.value]?.label || params.value;
        },
        width: Drupal.ebAggrid.COLUMN_WIDTHS.STANDARD,
        columnGroupShow: 'closed'
      }
    ]
  });

  /**
   * Create Formatter column group for Fields tab.
   *
   * @param {Object} discovery
   *   Discovery data from drupalSettings.
   *
   * @return {Object}
   *   AG-Grid column group definition.
   */
  Drupal.ebAggrid.createFormatterColumnGroup = (discovery) => ({
    headerName: 'Formatter (View)',
    marryChildren: true,
    openByDefault: false,
    children: [
      {
        field: 'formatter',
        headerName: 'Type',
        editable: true,
        cellEditor: 'agSelectCellEditor',
        cellEditorParams: (params) => {
          const fieldType = params.data?.field_type;
          const formattersForType = fieldType
            ? (discovery.formattersByFieldType?.[fieldType] || [])
            : [];
          return { values: ['', ...formattersForType] };
        },
        valueFormatter: (params) => {
          if (!params.value) return '';
          return discovery.formatters?.[params.value]?.label || params.value;
        },
        width: Drupal.ebAggrid.COLUMN_WIDTHS.STANDARD,
        columnGroupShow: 'closed'
      },
      {
        field: 'label_display',
        headerName: 'Label',
        editable: true,
        cellEditor: 'agSelectCellEditor',
        cellEditorParams: {
          values: ['above', 'inline', 'hidden', 'visually_hidden']
        },
        width: 100,
        columnGroupShow: 'open'
      }
    ]
  });

  /**
   * Create Group column group for Fields tab.
   *
   * This column group only appears when the eb_field_group module is enabled.
   * It allows assigning fields to field groups for default form/view modes.
   *
   * @return {Object|null}
   *   AG-Grid column group definition, or null if eb_field_group is not enabled.
   */
  Drupal.ebAggrid.createGroupColumnGroup = () => {
    const fieldGroupExt = drupalSettings.ebAggrid?.extensions?.field_group;
    if (!fieldGroupExt?.enabled) {
      return null;
    }

    // Build form group options with empty string for "none".
    const formGroupOptions = ['', ...(fieldGroupExt.formGroups || []).map((g) => g.name)];
    const viewGroupOptions = ['', ...(fieldGroupExt.viewGroups || []).map((g) => g.name)];

    return {
      headerName: 'Group',
      marryChildren: true,
      openByDefault: false,
      children: [
        {
          field: 'form_group',
          headerName: 'Form',
          editable: true,
          cellEditor: 'agSelectCellEditor',
          cellEditorParams: { values: formGroupOptions },
          valueFormatter: (params) => {
            if (!params.value) {
              return '';
            }
            const group = (fieldGroupExt.formGroups || []).find((g) => g.name === params.value);
            return group?.label || params.value;
          },
          width: Drupal.ebAggrid.COLUMN_WIDTHS.MEDIUM,
          columnGroupShow: 'closed'
        },
        {
          field: 'view_group',
          headerName: 'View',
          editable: true,
          cellEditor: 'agSelectCellEditor',
          cellEditorParams: { values: viewGroupOptions },
          valueFormatter: (params) => {
            if (!params.value) {
              return '';
            }
            const group = (fieldGroupExt.viewGroups || []).find((g) => g.name === params.value);
            return group?.label || params.value;
          },
          width: Drupal.ebAggrid.COLUMN_WIDTHS.MEDIUM,
          columnGroupShow: 'open'
        }
      ]
    };
  };

  /**
   * Get default settings for a widget from discovery data.
   *
   * @param {string} widgetId
   *   The widget plugin ID.
   *
   * @return {Object}
   *   Default settings object (may be empty).
   */
  Drupal.ebAggrid.getWidgetDefaultSettings = (widgetId) => {
    if (!widgetId) return {};
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    return discovery.widgets?.[widgetId]?.default_settings || {};
  };

  /**
   * Get default settings for a formatter from discovery data.
   *
   * @param {string} formatterId
   *   The formatter plugin ID.
   *
   * @return {Object}
   *   Default settings object (may be empty).
   */
  Drupal.ebAggrid.getFormatterDefaultSettings = (formatterId) => {
    if (!formatterId) return {};
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    return discovery.formatters?.[formatterId]?.default_settings || {};
  };

  /**
   * Populate widget settings with defaults when widget changes.
   *
   * @param {Object} params
   *   AG-Grid cell value changed params.
   * @param {string} newWidget
   *   The new widget value.
   */
  Drupal.ebAggrid.populateWidgetSettings = (params, newWidget) => {
    const defaultSettings = Drupal.ebAggrid.getWidgetDefaultSettings(newWidget);
    params.node.setDataValue('widget_settings', { ...defaultSettings });
  };

  /**
   * Populate formatter settings with defaults when formatter changes.
   *
   * @param {Object} params
   *   AG-Grid cell value changed params.
   * @param {string} newFormatter
   *   The new formatter value.
   */
  Drupal.ebAggrid.populateFormatterSettings = (params, newFormatter) => {
    const defaultSettings = Drupal.ebAggrid.getFormatterDefaultSettings(newFormatter);
    params.node.setDataValue('formatter_settings', { ...defaultSettings });
  };

  /**
   * Populate default widget and formatter when field type changes.
   *
   * @param {Object} params
   *   AG-Grid cell value changed params.
   * @param {string} newFieldType
   *   The new field type value.
   */
  Drupal.ebAggrid.populateDefaultWidgetFormatter = (params, newFieldType) => {
    const discovery = drupalSettings.ebAggrid?.discovery || {};

    // Set default widget if empty.
    const widgetsForType = discovery.widgetsByFieldType?.[newFieldType] || [];
    if (widgetsForType.length > 0 && !params.data.widget) {
      const defaultWidget = widgetsForType[0];
      params.node.setDataValue('widget', defaultWidget);
      // Populate widget settings with defaults from the widget.
      const widgetDefaults = Drupal.ebAggrid.getWidgetDefaultSettings(defaultWidget);
      params.node.setDataValue('widget_settings', { ...widgetDefaults });
    }

    // Set default formatter if empty.
    const formattersForType = discovery.formattersByFieldType?.[newFieldType] || [];
    if (formattersForType.length > 0 && !params.data.formatter) {
      const defaultFormatter = formattersForType[0];
      params.node.setDataValue('formatter', defaultFormatter);
      // Populate formatter settings with defaults from the formatter.
      const formatterDefaults = Drupal.ebAggrid.getFormatterDefaultSettings(defaultFormatter);
      params.node.setDataValue('formatter_settings', { ...formatterDefaults });
    }
  };

  /**
   * Get field definition columns.
   *
   * Column order: Target (combined), Label, Field Name, Field Type, Description, etc.
   * Includes autogeneration handlers for Field Name from Label.
   *
   * @param {Object} discovery
   *   Discovery data from drupalSettings.
   *
   * @return {Array}
   *   Array of AG-Grid column definitions.
   */
  Drupal.ebAggrid.getFieldColumns = (discovery) => [
    Drupal.ebAggrid.getRowNumberColumn(),
    Drupal.ebAggrid.getStatusColumn(),
    Drupal.ebAggrid.getDragColumn(),
    // Combined Target column (Entity Type + Bundle).
    Drupal.ebAggrid.createCombinedTargetColumn(),
    // Label comes BEFORE Field Name (natural mental model).
    Drupal.ebAggrid.markRequired({
      field: 'label',
      headerName: 'Label',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.LARGE,
      cellEditorParams: {
        placeholder: 'Enter human-readable name'
      }
    }),
    // Field Name is auto-generated from Label.
    Drupal.ebAggrid.markRequired({
      field: 'field_name',
      headerName: 'Field Name',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.STANDARD,
      valueSetter: Drupal.ebAggrid.valueHandlers.createPrefixSetter('field_', 'field_name')
    }),
    // Field Type with human-readable labels and icons.
    Drupal.ebAggrid.markRequired({
      field: 'field_type',
      headerName: 'Field Type',
      editable: true,
      cellEditor: 'agSelectCellEditor',
      cellEditorParams: {
        values: discovery.fieldTypes || []
      },
      valueFormatter: Drupal.ebAggrid.fieldTypeLabelFormatter,
      cellRenderer: Drupal.ebAggrid.fieldTypeIconRenderer,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.LARGE
    }),
    {
      field: 'description',
      headerName: 'Description',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.LARGE
    },
    {
      field: 'required',
      headerName: 'Required',
      editable: true,
      cellRenderer: 'agCheckboxCellRenderer',
      cellEditor: 'agCheckboxCellEditor',
      width: 90
    },
    {
      field: 'cardinality',
      headerName: 'Cardinality',
      editable: true,
      cellEditor: 'agSelectCellEditor',
      cellEditorParams: {
        values: ['1', '-1', '2', '3', '4', '5', '10']
      },
      valueParser: Drupal.ebAggrid.valueHandlers.intParser,
      width: 100
    },
    {
      field: 'translatable',
      headerName: 'Translatable',
      editable: true,
      cellRenderer: 'agCheckboxCellRenderer',
      cellEditor: 'agCheckboxCellEditor',
      width: 100
    },
    // Widget (Form) column group.
    Drupal.ebAggrid.createWidgetColumnGroup(discovery),
    // Formatter (View) column group.
    Drupal.ebAggrid.createFormatterColumnGroup(discovery),
    // Group column group (only if eb_field_group is enabled).
    Drupal.ebAggrid.createGroupColumnGroup(),
    // Settings column with Configure button.
    Drupal.ebAggrid.createSettingsColumn(),
    Drupal.ebAggrid.createActionsColumn()
  ].filter(Boolean);

  /**
   * Get field group definition columns.
   *
   * @param {Object} discovery
   *   Discovery data from drupalSettings.
   *
   * @return {Array}
   *   Array of AG-Grid column definitions.
   */
  Drupal.ebAggrid.getFieldGroupColumns = (discovery) => [
    Drupal.ebAggrid.getRowNumberColumn(),
    Drupal.ebAggrid.getStatusColumn(),
    Drupal.ebAggrid.getDragColumn(),
    // Combined Target column (Entity Type + Bundle).
    Drupal.ebAggrid.createCombinedTargetColumn(),
    {
      field: 'display_type',
      headerName: 'Display Type',
      editable: true,
      cellEditor: 'agSelectCellEditor',
      cellEditorParams: {
        values: discovery.displayTypes || ['form', 'view']
      },
      width: 110
    },
    Drupal.ebAggrid.markRequired({
      field: 'mode',
      headerName: 'Mode',
      editable: true,
      cellEditor: 'agSelectCellEditor',
      cellEditorParams: {
        values: discovery.displayModes || ['default']
      },
      width: 100
    }),
    Drupal.ebAggrid.markRequired({
      field: 'group_name',
      headerName: 'Group Name',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.STANDARD,
      valueSetter: Drupal.ebAggrid.valueHandlers.createPrefixSetter('group_', 'group_name')
    }),
    {
      field: 'label',
      headerName: 'Label',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.STANDARD
    },
    Drupal.ebAggrid.markRequired({
      field: 'format_type',
      headerName: 'Format Type',
      editable: true,
      cellEditor: 'agSelectCellEditor',
      cellEditorParams: {
        values: discovery.fieldGroupFormatTypes || []
      },
      width: Drupal.ebAggrid.COLUMN_WIDTHS.MEDIUM
    }),
    {
      field: 'parent',
      headerName: 'Parent',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.MEDIUM
    },
    {
      field: 'weight',
      headerName: 'Weight',
      editable: true,
      valueParser: Drupal.ebAggrid.valueHandlers.intParser,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.SMALL
    },
    // Format Settings column with Configure button (modal).
    Drupal.ebAggrid.createFormatSettingsColumn(),
    Drupal.ebAggrid.createActionsColumn()
  ];

  /**
   * Create a Display Settings column with Configure button.
   *
   * Opens modal for widget_settings and formatter_settings configuration.
   *
   * @return {Object}
   *   Column definition.
   */
  Drupal.ebAggrid.createDisplaySettingsColumn = () => ({
    headerName: 'Settings',
    field: '_display_settings',
    headerTooltip: 'Click to configure widget and formatter settings.',
    width: 130,
    editable: false,
    sortable: false,
    filter: false,
    cellRenderer: (params) => {
      const hasWidget = params.data.widget_settings &&
                       typeof params.data.widget_settings === 'object' &&
                       Object.keys(params.data.widget_settings).length > 0;
      const hasFormatter = params.data.formatter_settings &&
                          typeof params.data.formatter_settings === 'object' &&
                          Object.keys(params.data.formatter_settings).length > 0;

      // Count configured sections.
      const count = (hasWidget ? 1 : 0) + (hasFormatter ? 1 : 0);
      const label = count > 0 ? Drupal.t('Configured (@count)', { '@count': count }) : Drupal.t('Configure');
      const cssClass = count > 0 ? 'eb-settings-btn eb-settings-btn--has-values' : 'eb-settings-btn';

      const button = document.createElement('button');
      button.type = 'button';
      button.className = cssClass;
      button.textContent = label;
      button.title = count > 0 ? Drupal.t('Click to modify settings') : Drupal.t('Click to add settings');

      return button;
    },
    onCellClicked: (params) => {
      if (Drupal.ebAggrid.openDisplaySettingsModal) {
        Drupal.ebAggrid.openDisplaySettingsModal(params.node, params.data);
      }
    }
  });

  /**
   * Get display field definition columns.
   *
   * Column structure aligned with Fields tab for consistency.
   *
   * @param {Object} discovery
   *   Discovery data from drupalSettings.
   *
   * @return {Array}
   *   Array of AG-Grid column definitions.
   */
  Drupal.ebAggrid.getDisplayFieldColumns = (discovery) => [
    Drupal.ebAggrid.getRowNumberColumn(),
    Drupal.ebAggrid.getStatusColumn(),
    Drupal.ebAggrid.getDragColumn(),
    // Combined Target column (Entity Type + Bundle).
    Drupal.ebAggrid.createCombinedTargetColumn(),
    Drupal.ebAggrid.markRequired({
      field: 'display_type',
      headerName: 'Display Type',
      editable: true,
      cellEditor: 'agSelectCellEditor',
      cellEditorParams: {
        values: discovery.displayTypes || ['form', 'view']
      },
      width: 110
    }),
    // Mode column with dynamic filtering based on display_type.
    Drupal.ebAggrid.markRequired({
      field: 'mode',
      headerName: 'Mode',
      editable: true,
      cellEditor: 'agSelectCellEditor',
      cellEditorParams: (params) => {
        const displayType = params.data?.display_type;
        if (displayType === 'form') {
          return { values: discovery.formDisplayModes || ['default'] };
        }
        return { values: discovery.viewDisplayModes || ['default', 'teaser', 'full'] };
      },
      width: 120
    }),
    Drupal.ebAggrid.markRequired({
      field: 'field_name',
      headerName: 'Field Name',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.STANDARD
    }),
    // Widget (Form) column group - Type only.
    {
      headerName: 'Widget (Form)',
      marryChildren: true,
      openByDefault: false,
      children: [
        {
          field: 'widget',
          headerName: 'Type',
          editable: true,
          cellEditor: 'agSelectCellEditor',
          cellEditorParams: (params) => {
            const fieldName = params.data.field_name;
            const fieldData = Drupal.ebAggrid.findFieldDefinition(fieldName);
            const fieldType = fieldData?.field_type;
            const widgetsForType = fieldType && discovery.widgetsByFieldType
              ? discovery.widgetsByFieldType[fieldType] || []
              : Object.keys(discovery.widgets || {});
            return { values: ['', ...widgetsForType] };
          },
          valueFormatter: (params) => {
            if (!params.value) return '';
            return discovery.widgets?.[params.value]?.label || params.value;
          },
          width: Drupal.ebAggrid.COLUMN_WIDTHS.STANDARD,
          columnGroupShow: 'closed'
        }
      ]
    },
    // Formatter (View) column group - Type and Label.
    {
      headerName: 'Formatter (View)',
      marryChildren: true,
      openByDefault: false,
      children: [
        {
          field: 'formatter',
          headerName: 'Type',
          editable: true,
          cellEditor: 'agSelectCellEditor',
          cellEditorParams: (params) => {
            const fieldName = params.data.field_name;
            const fieldData = Drupal.ebAggrid.findFieldDefinition(fieldName);
            const fieldType = fieldData?.field_type;
            const formattersForType = fieldType && discovery.formattersByFieldType
              ? discovery.formattersByFieldType[fieldType] || []
              : Object.keys(discovery.formatters || {});
            return { values: ['', ...formattersForType] };
          },
          valueFormatter: (params) => {
            if (!params.value) return '';
            return discovery.formatters?.[params.value]?.label || params.value;
          },
          width: Drupal.ebAggrid.COLUMN_WIDTHS.STANDARD,
          columnGroupShow: 'closed'
        },
        {
          field: 'label_display',
          headerName: 'Label',
          editable: true,
          cellEditor: 'agSelectCellEditor',
          cellEditorParams: {
            values: ['above', 'inline', 'hidden', 'visually_hidden']
          },
          width: 100,
          columnGroupShow: 'open'
        }
      ]
    },
    // Standalone Weight column.
    {
      field: 'weight',
      headerName: 'Weight',
      editable: true,
      valueParser: Drupal.ebAggrid.valueHandlers.intParser,
      width: 80
    },
    // Standalone Group column.
    {
      field: 'group',
      headerName: 'Group',
      editable: true,
      width: 120
    },
    // Settings column with Configure button.
    Drupal.ebAggrid.createDisplaySettingsColumn(),
    Drupal.ebAggrid.createActionsColumn()
  ];

  /**
   * Get menu definition columns.
   *
   * @param {Object} discovery
   *   Discovery data from drupalSettings.
   *
   * @return {Array}
   *   Array of AG-Grid column definitions.
   */
  Drupal.ebAggrid.getMenuColumns = (discovery) => [
    Drupal.ebAggrid.getRowNumberColumn(),
    Drupal.ebAggrid.getStatusColumn(),
    Drupal.ebAggrid.markRequired({
      field: 'menu_id',
      headerName: 'Menu ID',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.STANDARD
    }),
    Drupal.ebAggrid.markRequired({
      field: 'label',
      headerName: 'Label',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.LARGE
    }),
    {
      field: 'description',
      headerName: 'Description',
      editable: true,
      width: Drupal.ebAggrid.COLUMN_WIDTHS.DESCRIPTION,
      flex: 1
    },
    Drupal.ebAggrid.createActionsColumn()
  ];

  // ============================================
  // HELPER FUNCTIONS
  // ============================================

  /**
   * Helper to find a field definition by field_name.
   *
   * @param {string} fieldName
   *   The field name to search for.
   *
   * @return {Object|null}
   *   The field definition object or null.
   */
  Drupal.ebAggrid.findFieldDefinition = (fieldName) => {
    if (!Drupal.ebAggrid.grids?.field) {
      return null;
    }

    const fieldGrid = Drupal.ebAggrid.grids.field;
    let result = null;

    fieldGrid.forEachNode((node) => {
      if (node.data?.field_name === fieldName) {
        result = node.data;
      }
    });

    return result;
  };

})(Drupal);
