/**
 * @file
 * AG-Grid cell renderers and value formatters for EB UI.
 *
 * This file contains shared rendering and formatting logic used by column
 * definitions. Separated from eb_ui.columns.js for maintainability.
 *
 * Includes:
 * - Cell renderers (delete button, status icon, drag handle)
 * - Value handlers (JSON, array, integer parsers/formatters)
 * - Label formatters (entity type, field type, target)
 */

(function (Drupal) {
  'use strict';

  /**
   * Initialize the ebUi namespace if not exists.
   */
  Drupal.ebAggrid = Drupal.ebAggrid || {};

  // ============================================
  // SHARED CELL RENDERERS
  // ============================================

  /**
   * Shared cell renderers to eliminate duplication.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.cellRenderers = {

    /**
     * Delete button renderer.
     *
     * @param {Object} params
     *   AG-Grid cell params.
     *
     * @return {HTMLElement}
     *   The delete button element.
     */
    deleteButton: (params) => {
      const button = document.createElement('button');
      button.type = 'button';
      button.innerHTML = '&times;';
      button.className = 'grid-delete-btn';
      button.title = Drupal.t('Delete row');
      button.addEventListener('click', (event) => {
        event.preventDefault();
        params.api.applyTransaction({ remove: [params.data] });
      });
      return button;
    },

    /**
     * Row status icon renderer.
     *
     * @param {Object} params
     *   AG-Grid cell params.
     *
     * @return {HTMLElement}
     *   The status icon element.
     */
    statusIcon: (params) => {
      const status = params.value || 'valid';
      const container = document.createElement('div');
      container.className = `eb-row-status eb-row-status--${status}`;

      const icon = document.createElement('span');
      icon.className = 'eb-row-status__icon';
      icon.textContent = Drupal.ebAggrid.STATUS_ICONS[status] || '\u25CF';

      const titles = {
        valid: Drupal.t('Valid'),
        warning: Drupal.t('Warning'),
        error: Drupal.t('Error'),
        modified: Drupal.t('Modified')
      };
      container.title = titles[status] || '';

      container.appendChild(icon);
      return container;
    },

    /**
     * Drag handle renderer.
     *
     * @return {HTMLElement}
     *   The drag handle element.
     */
    dragHandle: () => {
      const handle = document.createElement('span');
      handle.className = 'eb-drag-handle';
      handle.innerHTML = '\u22EE\u22EE';
      handle.title = Drupal.t('Drag to reorder');
      return handle;
    }
  };

  // ============================================
  // SHARED VALUE HANDLERS
  // ============================================

  /**
   * Shared value formatters and parsers.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.valueHandlers = {

    /**
     * Format JSON value for display.
     *
     * @param {Object} params
     *   AG-Grid params.
     *
     * @return {string}
     *   Formatted string.
     */
    jsonFormatter: (params) => {
      if (params.value && typeof params.value === 'object') {
        return JSON.stringify(params.value);
      }
      return params.value || '';
    },

    /**
     * Parse JSON value from input.
     *
     * @param {Object} params
     *   AG-Grid params.
     *
     * @return {Object|string}
     *   Parsed object or original string.
     */
    jsonParser: (params) => {
      try {
        return JSON.parse(params.newValue);
      }
      catch (error) {
        return params.newValue;
      }
    },

    /**
     * Parse integer value.
     *
     * @param {Object} params
     *   AG-Grid params.
     *
     * @return {number}
     *   Parsed integer.
     */
    intParser: (params) => parseInt(params.newValue, 10) || 0,

    /**
     * Create machine name setter with prefix.
     *
     * @param {string} prefix
     *   The required prefix (e.g., 'field_', 'group_').
     * @param {string} fieldName
     *   The field name to set.
     *
     * @return {Function}
     *   Value setter function.
     */
    createPrefixSetter: (prefix, fieldName) => (params) => {
      let { newValue: value } = params;
      if (value && !value.startsWith(prefix)) {
        value = prefix + value;
      }
      params.data[fieldName] = value;
      return true;
    },

    /**
     * Format array as comma-separated string.
     *
     * @param {Object} params
     *   AG-Grid params.
     *
     * @return {string}
     *   Comma-separated string.
     */
    arrayFormatter: (params) => {
      if (Array.isArray(params.value)) {
        return params.value.join(', ');
      }
      return params.value || '';
    },

    /**
     * Parse comma-separated string to array.
     *
     * @param {Object} params
     *   AG-Grid params.
     *
     * @return {Array}
     *   Parsed array.
     */
    arrayParser: (params) => {
      if (typeof params.newValue === 'string') {
        return params.newValue
          .split(',')
          .map((item) => item.trim())
          .filter((item) => item);
      }
      return params.newValue;
    }
  };

  // ============================================
  // ENTITY TYPE LABEL FORMATTER
  // ============================================

  /**
   * Format entity type to display label with machine name.
   *
   * Displays as "Label (machine_name)" for user-friendly identification.
   *
   * @param {Object} params
   *   AG-Grid value formatter params.
   *
   * @return {string}
   *   Formatted entity type string.
   */
  Drupal.ebAggrid.entityTypeLabelFormatter = (params) => {
    const id = params.value;
    if (!id) {
      return '';
    }

    const labels = drupalSettings.ebAggrid?.discovery?.entityTypeLabels || {};
    const label = labels[id] || id;
    return `${label} (${id})`;
  };

  // ============================================
  // FIELD TYPE LABEL FORMATTER
  // ============================================

  /**
   * Format field type to display human-readable label.
   *
   * @param {Object} params
   *   AG-Grid value formatter params.
   *
   * @return {string}
   *   Human-readable field type label.
   */
  Drupal.ebAggrid.fieldTypeLabelFormatter = (params) => {
    const id = params.value;
    if (!id) {
      return '';
    }

    const labels = drupalSettings.ebAggrid?.discovery?.fieldTypeLabels || {};
    return labels[id] || id;
  };

  /**
   * Render field type with icon.
   *
   * Displays field type with its Drupal core icon from Field UI.
   *
   * @param {Object} params
   *   AG-Grid cell renderer params.
   *
   * @return {string}
   *   HTML string with icon and label.
   */
  Drupal.ebAggrid.fieldTypeIconRenderer = (params) => {
    const fieldType = params.value;
    if (!fieldType) {
      return '';
    }

    const iconClass = Drupal.ebAggrid.getFieldTypeIconClass(fieldType);
    const labels = drupalSettings.ebAggrid?.discovery?.fieldTypeLabels || {};
    const label = labels[fieldType] || fieldType;

    return `<span class="eb-field-type">
      <span class="eb-field-type__icon ${iconClass}"></span>
      <span class="eb-field-type__label">${label}</span>
    </span>`;
  };

  /**
   * Format settings preview for collapsed view.
   *
   * @param {Object} params
   *   AG-Grid value formatter params.
   *
   * @return {string}
   *   Preview string showing first few key-value pairs.
   */
  Drupal.ebAggrid.settingsPreviewFormatter = (params) => {
    if (!params.value || typeof params.value !== 'object') {
      return params.value || '';
    }

    const keys = Object.keys(params.value);
    if (keys.length === 0) {
      return '(empty)';
    }

    // Show first 2-3 key values as preview.
    const preview = keys.slice(0, 3).map((key) => {
      let val = params.value[key];
      if (typeof val === 'boolean') {
        val = val ? 'yes' : 'no';
      }
      if (typeof val === 'object') {
        val = '{...}';
      }
      return `${key}: ${val}`;
    }).join(', ');

    return keys.length > 3 ? `${preview}, ...` : preview;
  };

  // ============================================
  // TARGET COLUMN HELPERS
  // ============================================

  /**
   * Build flat list of target values for dropdown.
   *
   * Includes BOTH:
   * 1. Existing bundles from Drupal (discovery.bundleLabels)
   * 2. Bundle definitions from the Bundles tab (live grid data + bundleDefinitions)
   *
   * @return {Array}
   *   Array of "entity_type:bundle" strings.
   */
  Drupal.ebAggrid.getTargetValues = () => {
    const settings = drupalSettings.ebAggrid || {};
    const discovery = settings.discovery || {};
    const values = [''];
    const seen = {};

    // 1. Add bundles from discovery data (existing in Drupal).
    const entityTypes = discovery.entityTypes || [];
    const bundleLabels = discovery.bundleLabels || {};

    entityTypes.forEach((entityType) => {
      const bundles = Object.keys(bundleLabels[entityType] || {});
      bundles.forEach((bundle) => {
        const key = `${entityType}:${bundle}`;
        if (!seen[key]) {
          values.push(key);
          seen[key] = true;
        }
      });
    });

    // 2. Add bundles from Bundles tab (new, not yet created).
    const bundleGridApi = Drupal.ebAggrid.grids?.bundle;
    let bundleDefinitions = [];

    if (bundleGridApi) {
      // Get live data from bundle grid.
      bundleGridApi.forEachNode((node) => {
        if (node.data?.entity_type && node.data?.bundle_id) {
          bundleDefinitions.push({
            entity_type: node.data.entity_type,
            bundle_id: node.data.bundle_id,
            label: node.data.label || node.data.bundle_id
          });
        }
      });
    }
    else {
      // Fall back to initial bundleDefinitions from page load.
      bundleDefinitions = settings.bundleDefinitions || [];
    }

    bundleDefinitions.forEach((def) => {
      const { entity_type: entityType, bundle_id: bundle, label } = def;
      if (entityType && bundle) {
        const key = `${entityType}:${bundle}`;
        if (!seen[key]) {
          values.push(key);
          seen[key] = true;
          // Also add to bundleLabels for formatter.
          if (!discovery.bundleLabels) {
            discovery.bundleLabels = {};
          }
          if (!discovery.bundleLabels[entityType]) {
            discovery.bundleLabels[entityType] = {};
          }
          discovery.bundleLabels[entityType][bundle] = label || bundle;
        }
      }
    });

    return values;
  };

  /**
   * Format target as "Bundle Label (Entity Type Label)".
   *
   * @param {Object} params
   *   AG-Grid value formatter params.
   *
   * @return {string}
   *   Formatted target string.
   */
  Drupal.ebAggrid.targetLabelFormatter = (params) => {
    const entityType = params.data?.entity_type;
    const bundle = params.data?.bundle;
    if (!entityType || !bundle) {
      return '';
    }

    const discovery = drupalSettings.ebAggrid?.discovery || {};
    const entityLabel = (discovery.entityTypeLabels || {})[entityType] || entityType;
    const bundleLabel = ((discovery.bundleLabels || {})[entityType] || {})[bundle] || bundle;

    return `${bundleLabel} (${entityLabel})`;
  };

  /**
   * Get formatted target values for dropdown display.
   *
   * @return {Array}
   *   Array of formatted labels like "Blog Post (Content)".
   */
  Drupal.ebAggrid.getTargetValuesFormatted = () => {
    const rawValues = Drupal.ebAggrid.getTargetValues();
    const discovery = drupalSettings.ebAggrid?.discovery || {};

    return rawValues.map((value) => {
      if (!value) {
        return '';
      }
      const parts = value.split(':');
      const entityType = parts[0] || '';
      const bundle = parts[1] || '';
      if (!entityType || !bundle) {
        return value;
      }
      const entityLabel = (discovery.entityTypeLabels || {})[entityType] || entityType;
      const bundleLabel = ((discovery.bundleLabels || {})[entityType] || {})[bundle] || bundle;
      return `${bundleLabel} (${entityLabel})`;
    });
  };

  /**
   * Get mapping from formatted label to raw value.
   *
   * @return {Object}
   *   Map of "Bundle Label (Entity Type Label)" => "entity_type:bundle".
   */
  Drupal.ebAggrid.getTargetLabelToValueMap = () => {
    const rawValues = Drupal.ebAggrid.getTargetValues();
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    const mapping = {};

    rawValues.forEach((value) => {
      if (!value) {
        mapping[''] = '';
        return;
      }
      const parts = value.split(':');
      const entityType = parts[0] || '';
      const bundle = parts[1] || '';
      if (!entityType || !bundle) {
        mapping[value] = value;
        return;
      }
      const entityLabel = (discovery.entityTypeLabels || {})[entityType] || entityType;
      const bundleLabel = ((discovery.bundleLabels || {})[entityType] || {})[bundle] || bundle;
      const formattedLabel = `${bundleLabel} (${entityLabel})`;
      mapping[formattedLabel] = value;
    });

    return mapping;
  };

})(Drupal);
