/**
 * @file
 * Shared constants for EB UI module.
 *
 * This file defines all constants, mappings, and configuration objects
 * that are shared across multiple modules to eliminate duplication.
 */

(function (Drupal) {
  'use strict';

  /**
   * Initialize the ebUi namespace if not exists.
   */
  Drupal.ebAggrid = Drupal.ebAggrid || {};

  /**
   * Timing constants for debouncing operations.
   *
   * Values are read from drupalSettings if available, with hardcoded fallbacks.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.TIMING = {
    /** Debounce delay for server-side validation (ms). */
    VALIDATION_DEBOUNCE_MS:
      drupalSettings.ebAggrid?.settings?.validation_debounce_ms || 500,
    /** Debounce delay for syncing grid data to hidden fields (ms). */
    SYNC_DEBOUNCE_MS:
      drupalSettings.ebAggrid?.settings?.sync_debounce_ms || 150,
    /** Delay before showing tooltips (ms). */
    TOOLTIP_DELAY_MS: 300
  };

  /**
   * Mapping from tab IDs to grid type identifiers.
   *
   * Used throughout the UI for navigation and data lookup.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.TAB_GRID_MAP = {
    bundle_definitions: 'bundle',
    field_definitions: 'field',
    field_group_definitions: 'field_group',
    display_field_definitions: 'display',
    menu_definitions: 'menu'
  };

  /**
   * Reverse mapping from grid types to tab IDs.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.GRID_TAB_MAP = {
    bundle: 'bundle_definitions',
    field: 'field_definitions',
    field_group: 'field_group_definitions',
    display: 'display_field_definitions',
    menu: 'menu_definitions'
  };

  /**
   * Mapping from grid types to hidden field element IDs.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.HIDDEN_FIELD_IDS = {
    bundle: 'bundle-definitions-data',
    field: 'field-definitions-data',
    field_group: 'field-group-definitions-data',
    display: 'display-field-definitions-data',
    menu: 'menu-definitions-data'
  };

  /**
   * Mapping from grid types to data keys in collectAllGridData output.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.GRID_DATA_KEYS = {
    bundle: 'bundle_definitions',
    field: 'field_definitions',
    field_group: 'field_group_definitions',
    display: 'display_field_definitions',
    menu: 'menu_definitions'
  };

  /**
   * Required fields configuration for each grid type.
   *
   * Used for client-side validation.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.REQUIRED_FIELDS = {
    bundle: ['entity_type', 'bundle_id', 'label'],
    field: ['entity_type', 'bundle', 'field_name', 'field_type', 'label'],
    field_group: ['entity_type', 'bundle', 'mode', 'group_name', 'format_type'],
    display: ['entity_type', 'bundle', 'display_type', 'mode', 'field_name'],
    menu: ['menu_id', 'label']
  };

  /**
   * Default row data templates for adding new rows.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.DEFAULT_ROW_DATA = {
    bundle: {
      entity_type: 'node',
      bundle_id: '',
      label: '',
      description: ''
    },
    field: {
      entity_type: 'node',
      bundle: '',
      field_name: 'field_',
      field_type: 'string',
      label: '',
      description: '',
      required: false,
      cardinality: 1,
      translatable: false,
      widget: '',
      formatter: '',
      widget_settings: {},
      formatter_settings: {},
      weight: 0,
      label_display: 'above',
      form_group: '',
      view_group: ''
    },
    field_group: {
      entity_type: 'node',
      bundle: '',
      display_type: 'form',
      mode: 'default',
      group_name: 'group_',
      label: '',
      format_type: 'details',
      weight: 0
    },
    display: {
      entity_type: 'node',
      bundle: '',
      display_type: 'view',
      mode: 'default',
      field_name: '',
      widget: '',
      formatter: '',
      widget_settings: {},
      formatter_settings: {},
      label_display: 'above',
      weight: 0
    },
    menu: {
      menu_id: '',
      label: '',
      description: ''
    }
  };

  /**
   * Row status values and their display properties.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.ROW_STATUS = {
    VALID: 'valid',
    WARNING: 'warning',
    ERROR: 'error',
    MODIFIED: 'modified'
  };

  /**
   * Row status display icons.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.STATUS_ICONS = {
    valid: '\u2713',
    warning: '\u26A0',
    error: '\u2717',
    modified: '\u25CF'
  };

  /**
   * Column width presets for consistent sizing.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.COLUMN_WIDTHS = {
    ICON: 40,
    ROW_NUMBER: 50,
    ACTION: 60,
    SMALL: 80,
    MEDIUM: 130,
    STANDARD: 150,
    LARGE: 200,
    XLARGE: 250,
    DESCRIPTION: 300
  };

  /**
   * Common entity types for prefetching bundles.
   *
   * @type {Array}
   */
  Drupal.ebAggrid.COMMON_ENTITY_TYPES = [
    'node',
    'taxonomy_term',
    'media',
    'paragraph',
    'user'
  ];

  /**
   * Internal field names that should be excluded from data operations.
   *
   * @type {Array}
   */
  Drupal.ebAggrid.INTERNAL_FIELDS = [
    '_status',
    '_actions',
    '_rowNum',
    '_drag'
  ];

  /**
   * Grid types that support row drag reordering.
   *
   * @type {Array}
   */
  Drupal.ebAggrid.DRAG_ENABLED_GRIDS = [
    'field',
    'field_group',
    'display'
  ];

  /**
   * Machine name field identifiers for validation.
   *
   * @type {Array}
   */
  Drupal.ebAggrid.MACHINE_NAME_FIELDS = [
    'bundle_id',
    'menu_id'
  ];

  /**
   * Error categorization patterns for grouping validation errors.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.ERROR_CATEGORIES = {
    'does not exist': {
      Bundle: 'Missing Bundles',
      Field: 'Missing Fields',
      'Entity type': 'Entity Type Issues',
      Menu: 'Menu Issues',
      default: 'Missing Dependencies'
    },
    'already exists': 'Duplicate Entries',
    required: 'Required Fields',
    Required: 'Required Fields',
    invalid: 'Invalid Values',
    Invalid: 'Invalid Values',
    default: 'Validation Errors'
  };

  /**
   * Mapping of field types to Drupal core field icon CSS classes.
   *
   * These icons are from field_ui/css/field_ui.icons.theme.css.
   * Fallback class for unknown types: 'field-option__icon'.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.FIELD_TYPE_ICONS = {
    // Text types.
    string: 'field-icon-plain_text',
    string_long: 'field-icon-plain_text',
    text: 'field-icon-formatted_text',
    text_long: 'field-icon-formatted_text',
    text_with_summary: 'field-icon-formatted_text',
    // Number types.
    integer: 'field-icon-number',
    decimal: 'field-icon-number',
    float: 'field-icon-number',
    list_integer: 'field-icon-number',
    list_float: 'field-icon-number',
    // Selection types.
    list_string: 'field-icon-selection_list',
    boolean: 'field-icon-boolean',
    // Reference types.
    entity_reference: 'field-icon-reference',
    entity_reference_revisions: 'field-icon-reference',
    // Date/time types.
    datetime: 'field-icon-date_time',
    timestamp: 'field-icon-date_time',
    daterange: 'field-icon-date_time',
    created: 'field-icon-date_time',
    changed: 'field-icon-date_time',
    // Media types.
    file: 'field-icon-file_upload',
    image: 'field-icon-file_upload',
    // Communication types.
    email: 'field-icon-email',
    telephone: 'field-icon-telephone',
    link: 'field-icon-link',
    // Other.
    comment: 'field-icon-comment'
  };

  /**
   * Get the icon CSS class for a field type.
   *
   * @param {string} fieldType
   *   The field type machine name.
   *
   * @return {string}
   *   The CSS class for the icon, or fallback class for unknown types.
   */
  Drupal.ebAggrid.getFieldTypeIconClass = (fieldType) =>
    Drupal.ebAggrid.FIELD_TYPE_ICONS[fieldType] || 'field-option__icon';

  /**
   * Helper function to get grid type from tab ID.
   *
   * @param {string} tabId
   *   The tab identifier.
   *
   * @return {string|null}
   *   The grid type or null if not found.
   */
  Drupal.ebAggrid.getGridTypeFromTab = (tabId) => Drupal.ebAggrid.TAB_GRID_MAP[tabId] || null;

  /**
   * Helper function to get tab ID from grid type.
   *
   * @param {string} gridType
   *   The grid type identifier.
   *
   * @return {string|null}
   *   The tab ID or null if not found.
   */
  Drupal.ebAggrid.getTabFromGridType = (gridType) => Drupal.ebAggrid.GRID_TAB_MAP[gridType] || null;

  /**
   * Check if a field is required for a grid type.
   *
   * @param {string} gridType
   *   The grid type.
   * @param {string} field
   *   The field name.
   *
   * @return {boolean}
   *   True if required.
   */
  Drupal.ebAggrid.isFieldRequired = (gridType, field) => {
    const required = Drupal.ebAggrid.REQUIRED_FIELDS[gridType] || [];
    return required.includes(field);
  };

  /**
   * Check if a field is internal (should be excluded from data operations).
   *
   * @param {string} fieldName
   *   The field name to check.
   *
   * @return {boolean}
   *   True if the field is internal.
   */
  Drupal.ebAggrid.isInternalField = (fieldName) => Drupal.ebAggrid.INTERNAL_FIELDS.includes(fieldName);

  /**
   * Check if a grid type supports row drag reordering.
   *
   * @param {string} gridType
   *   The grid type.
   *
   * @return {boolean}
   *   True if drag is enabled.
   */
  Drupal.ebAggrid.isDragEnabled = (gridType) => Drupal.ebAggrid.DRAG_ENABLED_GRIDS.includes(gridType);

  // ============================================
  // SESSION STORAGE UTILITIES
  // ============================================

  /**
   * Safely get value from sessionStorage with JSON parsing.
   *
   * @param {string} key
   *   The storage key.
   * @param {*} defaultValue
   *   Default value if key doesn't exist or on error.
   *
   * @return {*}
   *   Parsed value or default.
   */
  Drupal.ebAggrid.storageGet = (key, defaultValue) => {
    try {
      const stored = sessionStorage.getItem(key);
      return stored ? JSON.parse(stored) : (defaultValue !== undefined ? defaultValue : null);
    }
    catch (e) {
      return defaultValue !== undefined ? defaultValue : null;
    }
  };

  /**
   * Safely set value in sessionStorage with JSON serialization.
   *
   * @param {string} key
   *   The storage key.
   * @param {*} value
   *   Value to store (will be JSON stringified).
   *
   * @return {boolean}
   *   True if successful, false on error.
   */
  Drupal.ebAggrid.storageSet = (key, value) => {
    try {
      sessionStorage.setItem(key, JSON.stringify(value));
      return true;
    }
    catch (e) {
      return false;
    }
  };

  // ============================================
  // INTERNAL FIELDS UTILITIES
  // ============================================

  /**
   * Internal fields convention documentation.
   *
   * Fields prefixed with underscore (_) are internal to the grid and should
   * be excluded from data operations (serialization, comparison, export).
   * These include:
   *   - _status: Row validation status indicator
   *   - _actions: Delete button column
   *   - _rowNum: Row number display
   *   - _drag: Drag handle for reordering
   *   - _target: Combined entity_type:bundle virtual field
   *   - _settings: Settings modal trigger column
   *   - _bundleIdManuallyEdited: Tracks if bundle_id was manually edited
   *   - _previousAutoBundleId: Tracks auto-generated bundle_id
   *   - _fieldNameManuallyEdited: Tracks if field_name was manually edited
   *   - _previousAutoFieldName: Tracks auto-generated field_name
   */

  /**
   * Remove internal fields from a data object.
   *
   * Creates a new object without internal fields (prefixed with underscore).
   * Does not modify the original object.
   *
   * @param {Object} data
   *   The data object to clean.
   *
   * @return {Object}
   *   New object without internal fields.
   */
  Drupal.ebAggrid.stripInternalFields = (data) => {
    const cleaned = { ...data };
    Drupal.ebAggrid.INTERNAL_FIELDS.forEach((field) => {
      delete cleaned[field];
    });
    return cleaned;
  };

  /**
   * Get default row data for a grid type.
   *
   * @param {string} gridType
   *   The grid type.
   *
   * @return {Object}
   *   A copy of the default row data.
   */
  Drupal.ebAggrid.getDefaultRowData = (gridType) => {
    const defaults = Drupal.ebAggrid.DEFAULT_ROW_DATA[gridType];
    const rowData = defaults ? { ...defaults } : {};

    // For field grid type, populate default settings based on the default field type.
    if (gridType === 'field' && rowData.field_type) {
      const discovery = drupalSettings.ebAggrid?.discovery || {};
      const defaultSettings = discovery.fieldTypeDefaultSettings || {};
      const typeSettings = defaultSettings[rowData.field_type] || {};

      if (typeSettings.storage && Object.keys(typeSettings.storage).length > 0) {
        rowData.field_storage_settings = { ...typeSettings.storage };
      }
      if (typeSettings.config && Object.keys(typeSettings.config).length > 0) {
        rowData.field_config_settings = { ...typeSettings.config };
      }
    }

    return rowData;
  };

})(Drupal);
