/**
 * @file
 * Debug mode functionality for EB UI grids.
 *
 * Provides debug panels for grid rows using a floating panel approach
 * compatible with AG-Grid Community edition (no Enterprise features required).
 * Shows raw field data, settings, and validation state.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  Drupal.ebAggrid = Drupal.ebAggrid || {};

  /**
   * Track the currently open debug panel.
   *
   * @type {Object|null}
   */
  Drupal.ebAggrid._activeDebugPanel = null;

  /**
   * Check if debug mode is active.
   *
   * @return {boolean}
   *   TRUE if debug mode is enabled.
   */
  Drupal.ebAggrid.isDebugMode = () => drupalSettings.ebAggrid?.debugMode === true;

  /**
   * Get debug mode grid configuration.
   *
   * Adds row click handler for debug panel display.
   * Works with AG-Grid Community edition.
   *
   * @return {Object}
   *   AG-Grid configuration options for debug mode.
   */
  Drupal.ebAggrid.getDebugMasterDetailConfig = () => {
    if (!Drupal.ebAggrid.isDebugMode()) {
      return {};
    }

    return {
      onRowClicked: (params) => {
        // Only handle if click was not on an editable cell or button.
        if (params.event?.target) {
          const { target } = params.event;
          const tagName = target.tagName.toLowerCase();
          // Skip if clicking on input, button, select, or within an editor.
          if (tagName === 'input' || tagName === 'button' || tagName === 'select' || tagName === 'textarea') {
            return;
          }
          if (target.closest('.ag-cell-edit-wrapper') || target.closest('.eb-settings-btn')) {
            return;
          }
        }
        Drupal.ebAggrid.toggleDebugPanel(params.node, params.api);
      }
    };
  };

  /**
   * Get the debug column definition.
   *
   * This column shows a debug button to open the debug panel.
   *
   * @return {Object|null}
   *   Column definition or null if debug mode is not active.
   */
  Drupal.ebAggrid.getDebugExpandColumn = () => {
    if (!Drupal.ebAggrid.isDebugMode()) {
      return null;
    }

    return {
      headerName: '',
      field: '_debug',
      width: 36,
      minWidth: 36,
      maxWidth: 36,
      suppressHeaderMenuButton: true,
      suppressHeaderContextMenu: true,
      sortable: false,
      filter: false,
      resizable: false,
      pinned: 'left',
      cellRenderer: (params) => {
        const btn = document.createElement('button');
        btn.type = 'button';
        btn.className = 'eb-debug-row-btn';
        btn.innerHTML = '🔍';
        btn.title = Drupal.t('Show debug info for this row');
        btn.addEventListener('click', (e) => {
          e.stopPropagation();
          Drupal.ebAggrid.toggleDebugPanel(params.node, params.api);
        });
        return btn;
      }
    };
  };

  /**
   * Toggle debug panel for a row.
   *
   * @param {Object} node
   *   The AG-Grid row node.
   * @param {Object} api
   *   The AG-Grid API.
   */
  Drupal.ebAggrid.toggleDebugPanel = (node, api) => {
    const existingPanel = Drupal.ebAggrid._activeDebugPanel;

    // Close existing panel if open.
    if (existingPanel) {
      if (existingPanel.nodeId === node.id) {
        // Same row clicked - just close.
        Drupal.ebAggrid.closeDebugPanel();
        return;
      }
      Drupal.ebAggrid.closeDebugPanel();
    }

    // Open new panel.
    Drupal.ebAggrid.openDebugPanel(node, api);
  };

  /**
   * Open debug panel for a row.
   *
   * @param {Object} node
   *   The AG-Grid row node.
   * @param {Object} api
   *   The AG-Grid API.
   */
  Drupal.ebAggrid.openDebugPanel = (node, api) => {
    // Find the grid container.
    let gridContainer = document.querySelector('.eb-tab-panel--active .eb-grid');
    if (!gridContainer) {
      gridContainer = document.querySelector('.eb-grid');
    }
    if (!gridContainer) {
      return;
    }

    // Create panel element.
    const panel = document.createElement('div');
    panel.className = 'eb-debug-panel eb-debug-panel--floating';
    panel.innerHTML = Drupal.ebAggrid.renderDebugPanel(node.data, node);

    // Insert after grid container.
    gridContainer.parentNode.insertBefore(panel, gridContainer.nextSibling);

    // Store reference.
    Drupal.ebAggrid._activeDebugPanel = {
      element: panel,
      nodeId: node.id,
      api
    };

    // Scroll panel into view.
    setTimeout(() => {
      panel.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }, 50);
  };

  /**
   * Close the active debug panel.
   */
  Drupal.ebAggrid.closeDebugPanel = () => {
    const panel = Drupal.ebAggrid._activeDebugPanel;
    if (panel?.element?.parentNode) {
      panel.element.parentNode.removeChild(panel.element);
    }
    Drupal.ebAggrid._activeDebugPanel = null;
  };

  /**
   * Render the debug panel HTML for a row.
   *
   * @param {Object} data
   *   The row data object.
   * @param {Object} node
   *   The AG-Grid row node.
   *
   * @return {string}
   *   HTML string for the debug panel.
   */
  Drupal.ebAggrid.renderDebugPanel = (data, node) => {
    const html = [];
    const escape = Drupal.checkPlain;

    // Header with row info.
    html.push('<div class="eb-debug-panel__header">');
    html.push('<span class="eb-debug-panel__title">');
    html.push(Drupal.t('Debug: Row @index', { '@index': node.rowIndex + 1 }));
    html.push('</span>');
    html.push('<div class="eb-debug-panel__actions">');
    html.push(`<button type="button" class="eb-debug-panel__copy" onclick="Drupal.ebAggrid.copyDebugData(this)" data-row-index="${node.rowIndex}">`);
    html.push(Drupal.t('Copy JSON'));
    html.push('</button>');
    html.push(`<button type="button" class="eb-debug-panel__close" onclick="Drupal.ebAggrid.closeDebugPanel()" title="${Drupal.t('Close')}">`);
    html.push('×');
    html.push('</button>');
    html.push('</div>');
    html.push('</div>');

    // Content area with sections.
    html.push('<div class="eb-debug-panel__content">');

    // Determine which sections to show based on data keys.
    const sections = Drupal.ebAggrid.getDebugSections(data);

    sections.forEach((section) => {
      html.push('<div class="eb-debug-panel__section">');
      html.push(`<h4 class="eb-debug-panel__section-title">${escape(section.title)}</h4>`);

      if (section.type === 'json') {
        html.push(`<pre class="eb-debug-panel__json">${escape(JSON.stringify(section.data, null, 2))}</pre>`);
      }
      else if (section.type === 'table') {
        html.push(Drupal.ebAggrid.renderDebugTable(section.data));
      }

      html.push('</div>');
    });

    html.push('</div>');

    return html.join('');
  };

  /**
   * Get debug sections based on row data.
   *
   * @param {Object} data
   *   The row data object.
   *
   * @return {Array}
   *   Array of section definitions.
   */
  Drupal.ebAggrid.getDebugSections = (data) => {
    const sections = [];

    // Core fields (simple key-value pairs).
    const coreFields = {};
    const settingsFields = {};
    const otherFields = {};

    Object.keys(data).forEach((key) => {
      // Skip internal AG-Grid fields.
      if (key.startsWith('_') || key === 'ag-Grid-AutoColumn') {
        return;
      }

      const value = data[key];

      // Categorize fields.
      if (key.includes('settings') || key.includes('_settings')) {
        settingsFields[key] = value;
      }
      else if (typeof value === 'object' && value !== null) {
        otherFields[key] = value;
      }
      else {
        coreFields[key] = value;
      }
    });

    // Core fields as table.
    if (Object.keys(coreFields).length > 0) {
      sections.push({
        title: Drupal.t('Core Fields'),
        type: 'table',
        data: coreFields
      });
    }

    // Settings as JSON.
    if (Object.keys(settingsFields).length > 0) {
      sections.push({
        title: Drupal.t('Settings'),
        type: 'json',
        data: settingsFields
      });
    }

    // Other complex fields as JSON.
    if (Object.keys(otherFields).length > 0) {
      sections.push({
        title: Drupal.t('Other Data'),
        type: 'json',
        data: otherFields
      });
    }

    // Full raw data.
    sections.push({
      title: Drupal.t('Raw Data (Complete)'),
      type: 'json',
      data
    });

    return sections;
  };

  /**
   * Render a debug table for key-value pairs.
   *
   * @param {Object} data
   *   Key-value pairs to display.
   *
   * @return {string}
   *   HTML table string.
   */
  Drupal.ebAggrid.renderDebugTable = (data) => {
    const html = ['<table class="eb-debug-panel__table">'];
    html.push(`<thead><tr><th>${Drupal.t('Property')}</th><th>${Drupal.t('Value')}</th><th>${Drupal.t('Type')}</th></tr></thead>`);
    html.push('<tbody>');

    Object.keys(data).forEach((key) => {
      const value = data[key];
      let type = typeof value;
      let displayValue;

      if (value === null) {
        displayValue = '<em>null</em>';
        type = 'null';
      }
      else if (value === undefined) {
        displayValue = '<em>undefined</em>';
        type = 'undefined';
      }
      else if (type === 'boolean') {
        displayValue = value ? 'true' : 'false';
      }
      else if (type === 'object') {
        displayValue = `<code>${Drupal.checkPlain(JSON.stringify(value))}</code>`;
        type = Array.isArray(value) ? 'array' : 'object';
      }
      else {
        displayValue = Drupal.checkPlain(String(value));
      }

      html.push('<tr>');
      html.push(`<td class="eb-debug-panel__key">${Drupal.checkPlain(key)}</td>`);
      html.push(`<td class="eb-debug-panel__value">${displayValue}</td>`);
      html.push(`<td class="eb-debug-panel__type"><span class="eb-debug-type eb-debug-type--${type}">${type}</span></td>`);
      html.push('</tr>');
    });

    html.push('</tbody></table>');
    return html.join('');
  };

  /**
   * Copy debug data to clipboard.
   *
   * @param {HTMLElement} button
   *   The button element that was clicked.
   */
  Drupal.ebAggrid.copyDebugData = (button) => {
    const rowIndex = parseInt(button.getAttribute('data-row-index'), 10);

    // Find the active grid.
    const activeTab = document.querySelector('.eb-tab-panel--active');
    if (!activeTab) {
      return;
    }

    const tabName = activeTab.getAttribute('data-tab');
    const gridType = Drupal.ebAggrid.getGridTypeFromTabName(tabName);
    const gridApi = Drupal.ebAggrid.grids?.[gridType];

    if (!gridApi) {
      return;
    }

    const node = gridApi.getDisplayedRowAtIndex(rowIndex);
    if (!node?.data) {
      return;
    }

    const jsonStr = JSON.stringify(node.data, null, 2);

    // Use modern Clipboard API if available.
    if (navigator.clipboard?.writeText) {
      navigator.clipboard.writeText(jsonStr).then(() => {
        Drupal.ebAggrid.showCopyFeedback(button, true);
      }).catch(() => {
        Drupal.ebAggrid.showCopyFeedback(button, false);
      });
    }
    else {
      // Fallback for older browsers.
      const textarea = document.createElement('textarea');
      textarea.value = jsonStr;
      textarea.style.position = 'fixed';
      textarea.style.opacity = '0';
      document.body.appendChild(textarea);
      textarea.select();
      try {
        document.execCommand('copy');
        Drupal.ebAggrid.showCopyFeedback(button, true);
      }
      catch (e) {
        Drupal.ebAggrid.showCopyFeedback(button, false);
      }
      document.body.removeChild(textarea);
    }
  };

  /**
   * Show copy feedback on button.
   *
   * @param {HTMLElement} button
   *   The button element.
   * @param {boolean} success
   *   Whether the copy was successful.
   */
  Drupal.ebAggrid.showCopyFeedback = (button, success) => {
    const originalText = button.textContent;
    button.textContent = success ? Drupal.t('Copied!') : Drupal.t('Failed');
    button.classList.add(success ? 'eb-debug-panel__copy--success' : 'eb-debug-panel__copy--error');

    setTimeout(() => {
      button.textContent = originalText;
      button.classList.remove('eb-debug-panel__copy--success', 'eb-debug-panel__copy--error');
    }, 1500);
  };

  /**
   * Get grid type from tab name.
   *
   * @param {string} tabName
   *   The tab data attribute value.
   *
   * @return {string}
   *   The grid type identifier.
   */
  Drupal.ebAggrid.getGridTypeFromTabName = (tabName) => {
    const mapping = {
      bundle_definitions: 'bundle',
      field_definitions: 'field',
      field_group_definitions: 'field_group',
      display_field_definitions: 'display',
      menu_definitions: 'menu'
    };
    return mapping[tabName] || 'field';
  };

  /**
   * Add debug indicator to toolbar when debug mode is active.
   */
  Drupal.ebAggrid.addDebugIndicator = () => {
    if (!Drupal.ebAggrid.isDebugMode()) {
      return;
    }

    const toolbar = document.querySelector('.eb-toolbar');
    if (!toolbar) {
      return;
    }

    // Check if indicator already exists.
    if (toolbar.querySelector('.eb-debug-indicator')) {
      return;
    }

    const indicator = document.createElement('div');
    indicator.className = 'eb-debug-indicator';
    indicator.innerHTML = `<span class="eb-debug-indicator__icon">🔧</span><span class="eb-debug-indicator__text">${Drupal.t('Debug Mode')}</span>`;
    indicator.title = Drupal.t('Debug mode is active. Click any row expand arrow to see debug details.');

    // Insert at the beginning of toolbar.
    toolbar.insertBefore(indicator, toolbar.firstChild);
  };

})(Drupal, drupalSettings);
