/**
 * @file
 * Grid management for EB UI module.
 *
 * Handles grid initialization, lifecycle management, data synchronization,
 * and row operations for all AG-Grid instances.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  /**
   * Initialize the ebUi namespace if not exists.
   */
  Drupal.ebAggrid = Drupal.ebAggrid || {};

  /**
   * Storage for grid instances.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.grids = {};

  /**
   * Storage for resize event listeners for cleanup.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.resizeListeners = {};

  /**
   * Debounce timer for sync operations.
   *
   * @type {number|null}
   */
  Drupal.ebAggrid.syncDebounceTimer = null;

  /**
   * Default grid options shared across all grids.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.defaultGridOptions = {
    theme: typeof agGrid !== 'undefined' && agGrid.themeQuartz ?
      agGrid.themeQuartz.withParams({
        fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif',
        fontSize: '13px',
        headerFontSize: '13px',
        cellHorizontalPadding: 12,
        headerHeight: 40,
        rowHeight: 36,
        headerBackgroundColor: '#f5f5f5',
        oddRowBackgroundColor: '#fafafa',
        selectedRowBackgroundColor: '#e3f2fd',
        rangeSelectionBorderColor: '#1976d2'
      }) : 'legacy',
    rowSelection: {
      mode: 'multiRow',
      headerCheckbox: true,
      checkboxes: true
    },
    undoRedoCellEditing: true,
    undoRedoCellEditingLimit: 50,
    enableCellTextSelection: true,
    suppressScrollOnNewData: true,
    singleClickEdit: true,
    enterNavigatesVertically: true,
    enterNavigatesVerticallyAfterEdit: true,
    stopEditingWhenCellsLoseFocus: true,
    defaultColDef: {
      flex: 1,
      minWidth: 80,
      resizable: true,
      sortable: true,
      filter: true,
      editable: true,
      cellClassRules: {
        'eb-cell-error': (params) => Drupal.ebAggrid.hasCellError(params),
        'eb-cell-editable': (params) => params.colDef.editable !== false
      },
      tooltipValueGetter: (params) => Drupal.ebAggrid.getCellErrorMessage(params) || params.value
    },
    tabToNextCell: (params) => params.nextCellPosition,
    processCellForClipboard: (params) => Drupal.ebAggrid.processCopiedValue(params),
    processCellFromClipboard: (params) => Drupal.ebAggrid.processPastedValue(params),
    onCellValueChanged: (event) => {
      const gridType = event.context?.gridType;
      if (gridType) {
        Drupal.ebAggrid.validateCell(event);
        Drupal.ebAggrid.syncGridToHiddenFieldDebounced(event.api, gridType);
        Drupal.ebAggrid.markChangeCacheDirty();
        Drupal.ebAggrid.updateUnsavedIndicators();
        Drupal.ebAggrid.triggerDebouncedValidation(gridType);
      }
    },
    onRowDataUpdated: (event) => {
      const gridType = event.context?.gridType;
      if (gridType) {
        Drupal.ebAggrid.markChangeCacheDirty();
        Drupal.ebAggrid.updateUnsavedIndicators();
      }
    }
  };

  /**
   * Initialize a single grid.
   *
   * @param {string} gridType
   *   The type of grid (bundle, field, field_group, display, menu).
   * @param {string} containerId
   *   The DOM container ID.
   * @param {string} hiddenFieldId
   *   The hidden field ID for data storage.
   * @param {Array} columnDefs
   *   The column definitions.
   * @param {Array} rowData
   *   Initial row data.
   */
  Drupal.ebAggrid.initGrid = (gridType, containerId, hiddenFieldId, columnDefs, rowData) => {
    const container = document.getElementById(containerId);
    if (!container) {
      return;
    }

    // Check if grid already exists.
    if (Drupal.ebAggrid.grids[gridType]) {
      return;
    }

    // Enable row drag for grids with weight-based ordering.
    const rowDragEnabled = Drupal.ebAggrid.isDragEnabled(gridType);

    // Prepend debug expand column if debug mode is active.
    let finalColumnDefs = columnDefs;
    if (Drupal.ebAggrid.getDebugExpandColumn) {
      const debugColumn = Drupal.ebAggrid.getDebugExpandColumn();
      if (debugColumn) {
        finalColumnDefs = [debugColumn, ...columnDefs];
      }
    }

    const gridOptions = {
      ...Drupal.ebAggrid.defaultGridOptions,
      columnDefs: finalColumnDefs,
      rowData: rowData || [],
      rowDragManaged: rowDragEnabled,
      context: {
        gridType,
        hiddenFieldId
      },
      onRowDragEnd: rowDragEnabled ? (event) => {
        Drupal.ebAggrid.updateWeightsAfterDrag(event);
      } : undefined,
      onGridReady: (params) => {
        Drupal.ebAggrid.onGridReady(params, gridType, containerId);
      }
    };

    // Merge debug mode configuration if active.
    if (Drupal.ebAggrid.getDebugMasterDetailConfig) {
      const debugConfig = Drupal.ebAggrid.getDebugMasterDetailConfig();
      Object.assign(gridOptions, debugConfig);
    }

    // Create AG-Grid instance.
    const gridApi = agGrid.createGrid(container, gridOptions);
    Drupal.ebAggrid.grids[gridType] = gridApi;

    // Initial sync to hidden field.
    Drupal.ebAggrid.syncGridToHiddenField(gridApi, gridType);
  };

  /**
   * Grid ready callback handler.
   *
   * @param {Object} params
   *   AG-Grid ready params.
   * @param {string} gridType
   *   The grid type.
   * @param {string} containerId
   *   The container element ID.
   */
  Drupal.ebAggrid.onGridReady = (params, gridType, containerId) => {
    const gridContainer = document.getElementById(containerId);

    // Size columns if container is visible.
    if (gridContainer?.offsetWidth > 0) {
      params.api.sizeColumnsToFit();
    }

    // Create and store resize listener for cleanup.
    const resizeListener = () => {
      if (gridContainer?.offsetWidth > 0) {
        setTimeout(() => {
          params.api.sizeColumnsToFit();
        }, 100);
      }
    };
    window.addEventListener('resize', resizeListener);
    Drupal.ebAggrid.resizeListeners[gridType] = resizeListener;

    // Store initial data FIRST for change tracking.
    Drupal.ebAggrid.storeInitialData(params.api, gridType);

    // Set initial status for all rows.
    params.api.forEachNode((node) => {
      if (node.data) {
        node.data._status = Drupal.ebAggrid.ROW_STATUS.VALID;
      }
    });

    // Restore column group states from session storage.
    Drupal.ebAggrid.restoreColumnGroupStates(params.api, gridType);

    // Attach column group event listeners.
    Drupal.ebAggrid.attachColumnGroupListeners(params.api, gridType);

    // Run initial client-side validation.
    Drupal.ebAggrid.validateAllRows(gridType, params.api);

    // Refresh header to show row count.
    params.api.refreshHeader();
  };

  /**
   * Destroy a grid and clean up resources.
   *
   * @param {string} gridType
   *   The grid type to destroy.
   */
  Drupal.ebAggrid.destroyGrid = (gridType) => {
    const gridApi = Drupal.ebAggrid.grids[gridType];
    if (gridApi) {
      gridApi.destroy();
      delete Drupal.ebAggrid.grids[gridType];
    }

    // Remove resize listener.
    const resizeListener = Drupal.ebAggrid.resizeListeners[gridType];
    if (resizeListener) {
      window.removeEventListener('resize', resizeListener);
      delete Drupal.ebAggrid.resizeListeners[gridType];
    }

    // Clean up initial data.
    delete Drupal.ebAggrid.initialData[gridType];
  };

  /**
   * Destroy all grids and clean up resources.
   */
  Drupal.ebAggrid.destroyAllGrids = () => {
    Object.keys(Drupal.ebAggrid.grids).forEach((gridType) => {
      Drupal.ebAggrid.destroyGrid(gridType);
    });
  };

  /**
   * Get the currently active grid API.
   *
   * @return {Object|null}
   *   Active grid API or null.
   */
  Drupal.ebAggrid.getActiveGrid = () => {
    const activeTab = document.querySelector('.eb-tab-panel--active');
    if (!activeTab) {
      return null;
    }

    const tabId = activeTab.getAttribute('data-tab');
    const gridType = Drupal.ebAggrid.getGridTypeFromTab(tabId);
    return gridType ? Drupal.ebAggrid.grids[gridType] : null;
  };

  /**
   * Refresh all grids to update cell styling.
   */
  Drupal.ebAggrid.refreshAllGrids = () => {
    Object.keys(Drupal.ebAggrid.grids).forEach((gridType) => {
      const gridApi = Drupal.ebAggrid.grids[gridType];
      if (gridApi) {
        gridApi.refreshCells({ force: true });
      }
    });
  };

  /**
   * Sync grid data to hidden form field.
   *
   * For grids with drag-enabled reordering (field, field_group, display),
   * assigns weight based on row index to ensure proper ordering.
   *
   * @param {Object} gridApi
   *   The AG-Grid API instance.
   * @param {string} gridType
   *   The type of grid.
   */
  Drupal.ebAggrid.syncGridToHiddenField = (gridApi, gridType) => {
    const hiddenFieldId = Drupal.ebAggrid.HIDDEN_FIELD_IDS[gridType];
    const hiddenField = document.getElementById(hiddenFieldId);

    if (!hiddenField) {
      return;
    }

    const rowData = [];
    const isDragEnabled = Drupal.ebAggrid.isDragEnabled(gridType);
    let rowIndex = 0;

    gridApi.forEachNode((node) => {
      const data = { ...node.data };
      // Remove internal fields.
      Drupal.ebAggrid.INTERNAL_FIELDS.forEach((field) => {
        delete data[field];
      });

      // Assign weight from row index for drag-enabled grids.
      if (isDragEnabled) {
        data.weight = rowIndex;
      }

      // Remove empty string values to keep YAML clean.
      Object.keys(data).forEach((key) => {
        if (data[key] === '' || data[key] === null || data[key] === undefined) {
          delete data[key];
        }
      });
      if (Object.keys(data).length > 0) {
        rowData.push(data);
      }
      rowIndex++;
    });

    hiddenField.value = JSON.stringify(rowData);

    // Update tab row counts.
    if (typeof Drupal.ebAggrid.updateTabRowCounts === 'function') {
      Drupal.ebAggrid.updateTabRowCounts();
    }
  };

  /**
   * Debounced sync of grid data to hidden field.
   *
   * @param {Object} gridApi
   *   The AG-Grid API instance.
   * @param {string} gridType
   *   The type of grid.
   */
  Drupal.ebAggrid.syncGridToHiddenFieldDebounced = (gridApi, gridType) => {
    if (Drupal.ebAggrid.syncDebounceTimer) {
      clearTimeout(Drupal.ebAggrid.syncDebounceTimer);
    }

    const debounceMs = Drupal.ebAggrid.TIMING?.SYNC_DEBOUNCE_MS ?? 150;

    Drupal.ebAggrid.syncDebounceTimer = setTimeout(() => {
      Drupal.ebAggrid.syncGridToHiddenField(gridApi, gridType);
      Drupal.ebAggrid.syncDebounceTimer = null;
    }, debounceMs);
  };

  /**
   * Flush pending debounced sync immediately.
   */
  Drupal.ebAggrid.flushPendingSync = () => {
    if (Drupal.ebAggrid.syncDebounceTimer) {
      clearTimeout(Drupal.ebAggrid.syncDebounceTimer);
      Drupal.ebAggrid.syncDebounceTimer = null;
    }

    // Sync all grids immediately.
    Object.keys(Drupal.ebAggrid.grids).forEach((gridType) => {
      const gridApi = Drupal.ebAggrid.grids[gridType];
      if (gridApi) {
        Drupal.ebAggrid.syncGridToHiddenField(gridApi, gridType);
      }
    });
  };

  /**
   * Collect all grid data for submission/validation.
   *
   * For grids with drag-enabled reordering (field, field_group, display),
   * assigns weight based on row index to ensure proper ordering.
   *
   * @return {Object}
   *   Object containing all grid data.
   */
  Drupal.ebAggrid.collectAllGridData = () => {
    const allData = {};

    Object.keys(Drupal.ebAggrid.grids).forEach((gridType) => {
      const gridApi = Drupal.ebAggrid.grids[gridType];
      if (!gridApi) {
        return;
      }

      const dataKey = Drupal.ebAggrid.GRID_DATA_KEYS[gridType] || gridType;
      allData[dataKey] = [];

      // Check if this grid supports weight from row order.
      const isDragEnabled = Drupal.ebAggrid.isDragEnabled(gridType);
      let rowIndex = 0;

      gridApi.forEachNode((node) => {
        const rowData = { ...node.data };
        // Remove internal fields.
        Drupal.ebAggrid.INTERNAL_FIELDS.forEach((field) => {
          delete rowData[field];
        });

        // Assign weight from row index for drag-enabled grids.
        if (isDragEnabled) {
          rowData.weight = rowIndex;
        }

        // Remove empty form_group/view_group from field definitions.
        // These are optional and should not be saved as empty strings.
        if (gridType === 'field') {
          if (!rowData.form_group) {
            delete rowData.form_group;
          }
          if (!rowData.view_group) {
            delete rowData.view_group;
          }
        }

        allData[dataKey].push(rowData);
        rowIndex++;
      });
    });

    return allData;
  };

  /**
   * Add a new row to a grid.
   *
   * @param {string} gridType
   *   The grid type.
   */
  Drupal.ebAggrid.addRow = (gridType) => {
    const gridApi = Drupal.ebAggrid.grids[gridType];
    if (!gridApi) {
      return;
    }

    const newRowData = Drupal.ebAggrid.getDefaultRowData(gridType);
    const result = gridApi.applyTransaction({ add: [newRowData] });

    // Validate and set status for the new row.
    if (result.add?.length > 0) {
      const newNode = result.add[0];

      // Initialize validation errors for this grid type if needed.
      if (!Drupal.ebAggrid.validationErrors[gridType]) {
        Drupal.ebAggrid.validationErrors[gridType] = {};
      }

      // Validate the new row to show required field errors.
      const errors = Drupal.ebAggrid.validateRow(gridType, newNode);
      if (Object.keys(errors).length > 0) {
        Drupal.ebAggrid.validationErrors[gridType][newNode.id] = errors;
        // Set status to error since we have validation errors.
        newNode.data._status = Drupal.ebAggrid.ROW_STATUS.ERROR;
      }
      else {
        // No errors but it's a new row, show as modified.
        newNode.data._status = Drupal.ebAggrid.ROW_STATUS.MODIFIED;
      }

      setTimeout(() => {
        gridApi.ensureIndexVisible(newNode.rowIndex);
        // Focus the first required field that's empty.
        const required = Drupal.ebAggrid.REQUIRED_FIELDS[gridType] || [];
        let firstRequired = required[0];
        if (!firstRequired) {
          const colDefs = gridApi.getColumnDefs();
          if (colDefs?.length > 0) {
            firstRequired = colDefs[0].field;
          }
        }
        if (firstRequired) {
          gridApi.setFocusedCell(newNode.rowIndex, firstRequired);
        }
        // Refresh to show validation errors and status.
        gridApi.refreshCells({ rowNodes: [newNode], force: true });
        // Update tab indicators.
        Drupal.ebAggrid.updateTabErrorIndicators();
        // Update unsaved changes indicators.
        Drupal.ebAggrid.updateUnsavedIndicators();
      }, 100);
    }

    Drupal.ebAggrid.syncGridToHiddenField(gridApi, gridType);
    Drupal.ebAggrid.markChangeCacheDirty();
  };

  /**
   * Duplicate the current row.
   *
   * @param {Object} params
   *   AG-Grid params with node data.
   */
  Drupal.ebAggrid.duplicateRow = (params) => {
    if (!params.node?.data) {
      return;
    }

    const newData = { ...params.node.data };
    // Remove internal fields.
    Drupal.ebAggrid.INTERNAL_FIELDS.forEach((field) => {
      delete newData[field];
    });

    // Modify ID-like fields to indicate copy.
    if (newData.bundle_id) {
      newData.bundle_id = `${newData.bundle_id}_copy`;
    }
    if (newData.field_name?.startsWith('field_')) {
      newData.field_name = `${newData.field_name}_copy`;
    }
    if (newData.group_name?.startsWith('group_')) {
      newData.group_name = `${newData.group_name}_copy`;
    }
    if (newData.menu_id) {
      newData.menu_id = `${newData.menu_id}_copy`;
    }

    const result = params.api.applyTransaction({
      add: [newData],
      addIndex: params.node.rowIndex + 1
    });

    const gridType = params.api.gridOptions?.context?.gridType;
    if (gridType) {
      // Validate the duplicated row.
      if (result.add?.length > 0) {
        const newNode = result.add[0];

        // Initialize validation errors for this grid type if needed.
        if (!Drupal.ebAggrid.validationErrors[gridType]) {
          Drupal.ebAggrid.validationErrors[gridType] = {};
        }

        // Validate the new row.
        const errors = Drupal.ebAggrid.validateRow(gridType, newNode);
        if (Object.keys(errors).length > 0) {
          Drupal.ebAggrid.validationErrors[gridType][newNode.id] = errors;
          newNode.data._status = Drupal.ebAggrid.ROW_STATUS.ERROR;
        }
        else {
          newNode.data._status = Drupal.ebAggrid.ROW_STATUS.MODIFIED;
        }

        // Refresh to show validation status.
        params.api.refreshCells({ rowNodes: [newNode], force: true });
        Drupal.ebAggrid.updateTabErrorIndicators();
      }

      Drupal.ebAggrid.syncGridToHiddenField(params.api, gridType);
      Drupal.ebAggrid.markChangeCacheDirty();
      Drupal.ebAggrid.updateUnsavedIndicators();
    }
  };

  /**
   * Update weight values after row drag.
   *
   * @param {Object} event
   *   The row drag end event.
   */
  Drupal.ebAggrid.updateWeightsAfterDrag = (event) => {
    const { api: gridApi } = event;
    const gridType = gridApi.gridOptions?.context?.gridType;

    // Update weights based on new row order.
    let weight = 0;
    gridApi.forEachNode((node) => {
      if (node.data && node.data.weight !== weight) {
        node.data.weight = weight;
      }
      weight++;
    });

    // Sync to hidden field.
    if (gridType) {
      Drupal.ebAggrid.syncGridToHiddenField(gridApi, gridType);
      Drupal.ebAggrid.markChangeCacheDirty();
      Drupal.ebAggrid.updateUnsavedIndicators();
    }

    // Refresh to show updated weights.
    gridApi.refreshCells({ columns: ['weight'], force: true });
  };

  /**
   * Process pasted cell value - clean up data.
   *
   * @param {Object} params
   *   AG-Grid clipboard params.
   *
   * @return {string}
   *   Cleaned value.
   */
  Drupal.ebAggrid.processPastedValue = (params) => {
    let { value } = params;

    if (typeof value !== 'string') {
      return value;
    }

    // Trim whitespace.
    value = value.trim();

    // Handle boolean fields.
    if (params.column?.colDef) {
      const { colDef } = params.column;
      if (colDef.cellRenderer === 'agCheckboxCellRenderer') {
        const lowerVal = value.toLowerCase();
        return lowerVal === 'true' || lowerVal === '1' || lowerVal === 'yes';
      }

      // Handle numeric fields.
      if (colDef.field === 'cardinality' || colDef.field === 'weight') {
        value = value.replace(/[^\d-]/g, '');
        return parseInt(value, 10) || 0;
      }
    }

    return value;
  };

  /**
   * Process copied cell value for clipboard.
   *
   * @param {Object} params
   *   AG-Grid clipboard params.
   *
   * @return {string}
   *   Formatted value for clipboard.
   */
  Drupal.ebAggrid.processCopiedValue = (params) => {
    const { value } = params;

    if (typeof value === 'object' && value !== null) {
      return JSON.stringify(value);
    }

    if (typeof value === 'boolean') {
      return value ? 'true' : 'false';
    }

    return value;
  };

  /**
   * Restore column group states from session storage.
   *
   * @param {Object} gridApi
   *   The AG-Grid API.
   * @param {string} gridType
   *   The grid type.
   */
  Drupal.ebAggrid.restoreColumnGroupStates = (gridApi, gridType) => {
    const states = Drupal.ebAggrid.storageGet(`eb_column_groups_${gridType}`, {});
    Object.keys(states).forEach((groupId) => {
      gridApi.setColumnGroupOpened(groupId, states[groupId]);
    });
  };

  /**
   * Attach column group event listeners for persistence.
   *
   * @param {Object} gridApi
   *   The AG-Grid API.
   * @param {string} gridType
   *   The grid type.
   */
  Drupal.ebAggrid.attachColumnGroupListeners = (gridApi, gridType) => {
    gridApi.addEventListener('columnGroupOpened', (event) => {
      const stateKey = `eb_column_groups_${gridType}`;
      const states = Drupal.ebAggrid.storageGet(stateKey, {});
      states[event.columnGroup.getGroupId()] = event.columnGroup.isExpanded();
      Drupal.ebAggrid.storageSet(stateKey, states);
    });
  };

})(Drupal, drupalSettings);
