/**
 * @file
 * Main AG-Grid initialization for EB UI.
 *
 * This file orchestrates the EB UI module, initializing grids,
 * attaching behaviors, and coordinating between extracted sub-modules:
 * - constants.js: Shared constants and mappings
 * - validation.js: Cell and row validation
 * - changes.js: Change tracking and indicators
 * - modals.js: Modal dialogs
 * - grid.js: Grid lifecycle management
 * - columns.js: Column definitions
 */

(function (Drupal, drupalSettings, once) {
  'use strict';

  /**
   * Storage for grid instances and validation state.
   *
   * @type {Object}
   */
  Drupal.ebAggrid = Drupal.ebAggrid || {};
  Drupal.ebAggrid.grids = {};
  Drupal.ebAggrid.validationErrors = {};
  Drupal.ebAggrid.serverValidationErrors = {};
  Drupal.ebAggrid.validationTimer = null;
  Drupal.ebAggrid.isValidating = false;
  // Bundle cache is managed by Drupal.ebUi (eb_ui/base library).
  Drupal.ebAggrid.isSubmitting = false;
  Drupal.ebAggrid.isResizing = false;

  /**
   * Height constraints for resizable grid.
   */
  Drupal.ebAggrid.RESIZE_CONSTRAINTS = {
    MIN_HEIGHT: 300,
    MAX_HEIGHT: 1500,
    DEFAULT_HEIGHT: 700
  };

  /**
   * Escape HTML entities to prevent XSS attacks.
   *
   * This is a fallback for when Drupal.checkPlain is not available.
   * Always prefer Drupal.checkPlain when available.
   *
   * @param {string} text
   *   The text to escape.
   *
   * @return {string}
   *   The escaped text safe for HTML insertion.
   */
  Drupal.ebAggrid.escapeHtml = (text) => {
    if (text === null || text === undefined) {
      return '';
    }
    // Use Drupal.checkPlain if available (it handles edge cases better).
    if (typeof Drupal.checkPlain === 'function') {
      return Drupal.checkPlain(String(text));
    }
    // Fallback implementation.
    const div = document.createElement('div');
    div.textContent = String(text);
    return div.innerHTML;
  };

  /**
   * Create a text node safely (no HTML interpretation).
   *
   * @param {string} text
   *   The text content.
   *
   * @return {Text}
   *   A text node.
   */
  Drupal.ebAggrid.createTextNode = (text) => document.createTextNode(text === null || text === undefined ? '' : String(text));

  // ============================================
  // SHARED HELPERS (Aliases to Drupal.ebUi)
  // ============================================
  // These are provided by eb_ui/base library. We create aliases here
  // for backwards compatibility with other eb_aggrid modules.

  /**
   * Alias to Drupal.ebUi.createElement.
   */
  Drupal.ebAggrid.createElement = (...args) => Drupal.ebUi.createElement(...args);

  /**
   * Alias to Drupal.ebUi.resetSelectWithPlaceholder.
   */
  Drupal.ebAggrid.resetSelectWithPlaceholder = (...args) => Drupal.ebUi.resetSelectWithPlaceholder(...args);

  /**
   * Alias to Drupal.ebUi.getCsrfToken.
   */
  Drupal.ebAggrid.getCsrfToken = () => Drupal.ebUi.getCsrfToken();

  /**
   * Alias to Drupal.ebUi.getPostHeaders.
   */
  Drupal.ebAggrid.getPostHeaders = () => Drupal.ebUi.getPostHeaders();

  /**
   * Alias to Drupal.ebUi.fetchJson.
   */
  Drupal.ebAggrid.fetchJson = (...args) => Drupal.ebUi.fetchJson(...args);

  /**
   * Custom theme based on Quartz using the v33 Theming API.
   *
   * Uses USWDS-inspired colors for Federal Government accessibility compliance.
   *
   * @see https://www.ag-grid.com/javascript-data-grid/theming-api/
   * @see https://designsystem.digital.gov/design-tokens/color/theme-tokens/
   */
  const ebTheme = agGrid.themeQuartz.withParams({
    // USWDS primary blue for accent color.
    accentColor: '#005ea2',
    borderRadius: 4,
    // Header styling.
    headerBackgroundColor: '#f0f0f0',
    headerTextColor: '#1b1b1b',
    // Row interaction colors.
    rowHoverColor: 'rgba(0, 94, 162, 0.04)',
    selectedRowBackgroundColor: 'rgba(0, 94, 162, 0.08)',
    // Typography.
    fontSize: 14,
    headerFontSize: 14,
    headerFontWeight: 600,
    // Cell padding for touch targets.
    cellHorizontalPadding: 12,
    cellVerticalPadding: 10,
    // Spacing for 44px minimum row height (WCAG).
    spacing: 8,
    rowHeight: 44
  });

  /**
   * Default grid options shared by all grids.
   *
   * References functions from extracted modules.
   *
   * @type {Object}
   */
  const defaultGridOptions = {
    theme: ebTheme,
    // Enable browser tooltips for headerTooltip support.
    enableBrowserTooltips: true,
    defaultColDef: {
      sortable: true,
      filter: true,
      resizable: true,
      editable: true,
      singleClickEdit: true,
      cellClass: (params) => {
        const classes = [];
        // Add error class if cell has validation error.
        if (Drupal.ebAggrid.hasCellError(params)) {
          classes.push('eb-cell-error');
        }
        // Add editable indicator class.
        if (params.colDef.editable !== false) {
          classes.push('eb-cell-editable');
        }
        return classes;
      },
      // Tooltip for validation errors.
      tooltipValueGetter: (params) => Drupal.ebAggrid.getCellErrorMessage(params)
    },
    tooltipShowDelay: Drupal.ebAggrid.TIMING?.TOOLTIP_DELAY_MS ?? 300,
    rowSelection: {
      mode: 'multiRow',
      checkboxes: false,
      headerCheckbox: false,
      enableClickSelection: false
    },
    animateRows: true,
    stopEditingWhenCellsLoseFocus: true,
    undoRedoCellEditing: true,
    undoRedoCellEditingLimit: 50,
    enterNavigatesVerticallyAfterEdit: true,
    tabToNextCell: (params) => params.nextCellPosition,
    // Copy/paste handling.
    processCellFromClipboard: (params) => Drupal.ebAggrid.processPastedValue(params),
    processCellForClipboard: (params) => Drupal.ebAggrid.processCopiedValue(params),
    onCellValueChanged: (event) => {
      Drupal.ebAggrid.validateCell(event);
      // Use debounced sync to prevent excessive serialization during rapid edits.
      Drupal.ebAggrid.syncGridToHiddenFieldDebounced(event.api, event.context.gridType);
      // Trigger debounced server-side validation.
      Drupal.ebAggrid.triggerDebouncedValidation(event.context.gridType);
      // Update row status indicator with explicit gridType.
      Drupal.ebAggrid.updateRowStatus(event.api, event.node, event.context.gridType);
      // Update unsaved changes indicators.
      Drupal.ebAggrid.updateUnsavedIndicators();

      // Grid-specific autogeneration handlers.
      const { gridType } = event.context || {};
      if (gridType === 'bundle') {
        // Bundle ID autogeneration from Label.
        Drupal.ebAggrid.autogenerateBundleId(event);
        Drupal.ebAggrid.handleBundleIdManualEdit(event);
      }
      else if (gridType === 'field') {
        // Field Name autogeneration from Label.
        Drupal.ebAggrid.handleLabelChangeForFieldName(event);
        Drupal.ebAggrid.handleFieldNameManualEdit(event);
        // Field type change populates default settings and widget/formatter.
        Drupal.ebAggrid.handleFieldTypeChange(event);
        // Widget/Formatter changes populate their respective default settings.
        Drupal.ebAggrid.handleWidgetChange(event);
        Drupal.ebAggrid.handleFormatterChange(event);
      }
    },
    onRowDataUpdated: (event) => {
      Drupal.ebAggrid.syncGridToHiddenField(event.api, event.context.gridType);
    }
  };

  /**
   * Make default grid options available globally for grid.js.
   */
  Drupal.ebAggrid.defaultGridOptions = defaultGridOptions;

  /**
   * Get bundle editor params for dependent dropdown.
   *
   * @param {string} entityType
   *   The entity type ID.
   *
   * @return {Object}
   *   Cell editor params with values array.
   */
  Drupal.ebAggrid.getBundleEditorParams = (entityType) => {
    if (!entityType) {
      return { values: [''] };
    }

    if (Drupal.ebUi.bundleCache[entityType]) {
      return { values: ['', ...Drupal.ebUi.bundleCache[entityType]] };
    }

    // Fetch bundles asynchronously if not cached.
    Drupal.ebAggrid.fetchBundlesForEntityType(entityType);

    return { values: [''] };
  };

  /**
   * Alias to Drupal.ebUi.fetchBundlesForEntityType.
   */
  Drupal.ebAggrid.fetchBundlesForEntityType = (...args) => Drupal.ebUi.fetchBundlesForEntityType(...args);

  /**
   * Prefetch bundles for common entity types.
   *
   * Uses the server-provided list of available entity types from discovery,
   * filtered to only include the most commonly used ones to avoid
   * unnecessary requests for entity types without modules installed.
   */
  Drupal.ebAggrid.prefetchCommonBundles = () => {
    // Get available entity types from server-side discovery (module is installed).
    const availableTypes = drupalSettings.ebAggrid?.discovery?.entityTypes || [];

    // Only prefetch commonly used types that are actually available.
    const commonTypes = Drupal.ebAggrid.COMMON_ENTITY_TYPES || ['node', 'taxonomy_term', 'media', 'paragraph', 'user'];

    // Filter to only types that exist in the system.
    const typesToFetch = commonTypes.filter((type) => availableTypes.includes(type));

    typesToFetch.forEach((entityType) => {
      Drupal.ebAggrid.fetchBundlesForEntityType(entityType);
    });
  };

  /**
   * Get validation errors for display.
   *
   * @return {Object}
   *   All validation errors by grid type.
   */
  Drupal.ebAggrid.getValidationErrors = () => {
    const allErrors = {};
    let hasErrors = false;

    Object.keys(Drupal.ebAggrid.validationErrors).forEach((gridType) => {
      const gridErrors = Drupal.ebAggrid.validationErrors[gridType];
      Object.keys(gridErrors).forEach((rowId) => {
        const rowErrors = gridErrors[rowId];
        if (Object.keys(rowErrors).length > 0) {
          if (!allErrors[gridType]) {
            allErrors[gridType] = {};
          }
          allErrors[gridType][rowId] = rowErrors;
          hasErrors = true;
        }
      });
    });

    return hasErrors ? allErrors : null;
  };

  /**
   * Undo last edit in active grid.
   */
  Drupal.ebAggrid.undo = () => {
    const gridApi = Drupal.ebAggrid.getActiveGrid();
    if (gridApi) {
      gridApi.undoCellEditing();
    }
  };

  /**
   * Redo last undone edit in active grid.
   */
  Drupal.ebAggrid.redo = () => {
    const gridApi = Drupal.ebAggrid.getActiveGrid();
    if (gridApi) {
      gridApi.redoCellEditing();
    }
  };

  /**
   * Revert all changes in active grid.
   */
  Drupal.ebAggrid.revertAll = () => {
    const gridApi = Drupal.ebAggrid.getActiveGrid();
    if (gridApi && confirm(Drupal.t('Revert all changes in this grid?'))) {
      // Not fully supported - would need to reload from hidden field.
      gridApi.undoCellEditing();
    }
  };

  /**
   * Apply quick filter to active grid.
   *
   * @param {string} filterText
   *   The filter text.
   */
  Drupal.ebAggrid.applyQuickFilter = (filterText) => {
    const gridApi = Drupal.ebAggrid.getActiveGrid();
    if (gridApi) {
      gridApi.setGridOption('quickFilterText', filterText);
    }
  };

  /**
   * Toggle fullscreen mode for the grid container.
   *
   * Uses the native Browser Fullscreen API.
   */
  Drupal.ebAggrid.toggleFullscreen = () => {
    const container = document.querySelector('.eb-spreadsheet-container');
    if (!container) {
      return;
    }

    if (!Drupal.ebAggrid.isFullscreen()) {
      // Enter fullscreen.
      const requestFn = container.requestFullscreen ||
                       container.webkitRequestFullscreen ||
                       container.mozRequestFullScreen ||
                       container.msRequestFullscreen;

      if (requestFn) {
        requestFn.call(container).catch((err) => {
          console.warn('Fullscreen request failed:', err);
        });
      }
    }
    else {
      // Exit fullscreen.
      const exitFn = document.exitFullscreen ||
                    document.webkitExitFullscreen ||
                    document.mozCancelFullScreen ||
                    document.msExitFullscreen;

      if (exitFn) {
        exitFn.call(document);
      }
    }
  };

  /**
   * Check if currently in fullscreen mode.
   *
   * @return {boolean}
   *   TRUE if in fullscreen mode.
   */
  Drupal.ebAggrid.isFullscreen = () => {
    return !!(document.fullscreenElement ||
             document.webkitFullscreenElement ||
             document.mozFullScreenElement ||
             document.msFullscreenElement);
  };

  /**
   * Update fullscreen button state and icon.
   */
  Drupal.ebAggrid.updateFullscreenButton = () => {
    const btn = document.querySelector('.eb-fullscreen-btn');
    if (!btn) {
      return;
    }

    const icon = btn.querySelector('.eb-fullscreen-btn__icon');
    const isFullscreen = Drupal.ebAggrid.isFullscreen();

    if (isFullscreen) {
      btn.classList.add('eb-fullscreen-btn--active');
      btn.setAttribute('title', Drupal.t('Exit fullscreen (Escape)'));
      if (icon) {
        icon.textContent = '⛶';
      }
    }
    else {
      btn.classList.remove('eb-fullscreen-btn--active');
      btn.setAttribute('title', Drupal.t('Toggle fullscreen (F11)'));
      if (icon) {
        icon.textContent = '⛶';
      }
    }

    // Refresh grids after fullscreen change to adjust sizing.
    Drupal.ebAggrid.refreshAllGrids();
  };

  /**
   * Refresh all grid instances to recalculate size.
   */
  Drupal.ebAggrid.refreshAllGrids = () => {
    Object.keys(Drupal.ebAggrid.grids).forEach((gridType) => {
      const gridApi = Drupal.ebAggrid.grids[gridType];
      if (gridApi) {
        setTimeout(() => {
          gridApi.sizeColumnsToFit();
          gridApi.refreshHeader();
        }, 100);
      }
    });
  };

  /**
   * Initialize resize handle for adjustable grid height.
   */
  Drupal.ebAggrid.initResizeHandle = () => {
    const handle = document.querySelector('.eb-resize-handle');
    const tabPanels = document.querySelector('.eb-tab-panels');

    if (!handle || !tabPanels) {
      return;
    }

    let startY = 0;
    let startHeight = 0;

    // Restore saved height on init.
    const savedHeight = Drupal.ebAggrid.storageGet('ebUi_gridHeight', null);
    if (savedHeight) {
      const height = parseInt(savedHeight, 10);
      if (height >= Drupal.ebAggrid.RESIZE_CONSTRAINTS.MIN_HEIGHT &&
          height <= Drupal.ebAggrid.RESIZE_CONSTRAINTS.MAX_HEIGHT) {
        tabPanels.style.minHeight = `${height}px`;
        // Also update grid min-height.
        document.querySelectorAll('.eb-grid').forEach((grid) => {
          grid.style.minHeight = `${height - 50}px`;
        });
      }
    }

    const onMouseDown = (e) => {
      e.preventDefault();
      Drupal.ebAggrid.isResizing = true;
      startY = e.clientY;
      startHeight = tabPanels.offsetHeight;
      handle.classList.add('eb-resize-handle--active');
      document.body.style.cursor = 'ns-resize';
      document.body.style.userSelect = 'none';

      document.addEventListener('mousemove', onMouseMove);
      document.addEventListener('mouseup', onMouseUp);
    };

    const onMouseMove = (e) => {
      if (!Drupal.ebAggrid.isResizing) {
        return;
      }

      const delta = e.clientY - startY;
      let newHeight = startHeight + delta;

      // Enforce constraints.
      newHeight = Math.max(Drupal.ebAggrid.RESIZE_CONSTRAINTS.MIN_HEIGHT, newHeight);
      newHeight = Math.min(Drupal.ebAggrid.RESIZE_CONSTRAINTS.MAX_HEIGHT, newHeight);

      tabPanels.style.minHeight = `${newHeight}px`;

      // Update grid min-height.
      document.querySelectorAll('.eb-grid').forEach((grid) => {
        grid.style.minHeight = `${newHeight - 50}px`;
      });

      // Update ARIA value.
      handle.setAttribute('aria-valuenow', newHeight);
    };

    const onMouseUp = () => {
      Drupal.ebAggrid.isResizing = false;
      handle.classList.remove('eb-resize-handle--active');
      document.body.style.cursor = '';
      document.body.style.userSelect = '';

      document.removeEventListener('mousemove', onMouseMove);
      document.removeEventListener('mouseup', onMouseUp);

      // Save preference.
      const currentHeight = tabPanels.offsetHeight;
      Drupal.ebAggrid.storageSet('ebUi_gridHeight', currentHeight);

      // Refresh grids after resize.
      Drupal.ebAggrid.refreshAllGrids();
    };

    // Mouse events.
    handle.addEventListener('mousedown', onMouseDown);

    // Touch support for mobile.
    handle.addEventListener('touchstart', (e) => {
      if (e.touches.length === 1) {
        const touch = e.touches[0];
        onMouseDown({
          preventDefault: () => e.preventDefault(),
          clientY: touch.clientY
        });
      }
    }, { passive: false });

    document.addEventListener('touchmove', (e) => {
      if (Drupal.ebAggrid.isResizing && e.touches.length === 1) {
        const touch = e.touches[0];
        onMouseMove({ clientY: touch.clientY });
      }
    }, { passive: true });

    document.addEventListener('touchend', () => {
      if (Drupal.ebAggrid.isResizing) {
        onMouseUp();
      }
    });
  };

  /**
   * Handle keyboard shortcuts.
   *
   * @param {KeyboardEvent} e
   *   The keyboard event.
   */
  Drupal.ebAggrid.handleKeyboardShortcut = (e) => {
    // F11 = Toggle fullscreen.
    if (e.keyCode === 122) {
      e.preventDefault();
      Drupal.ebAggrid.toggleFullscreen();
      return;
    }

    // Ctrl/Cmd + Z = Undo.
    if ((e.ctrlKey || e.metaKey) && e.keyCode === 90 && !e.shiftKey) {
      e.preventDefault();
      Drupal.ebAggrid.undo();
      return;
    }

    // Ctrl/Cmd + Shift + Z or Ctrl/Cmd + Y = Redo.
    if ((e.ctrlKey || e.metaKey) && (e.keyCode === 90 && e.shiftKey || e.keyCode === 89)) {
      e.preventDefault();
      Drupal.ebAggrid.redo();
      return;
    }

    // Ctrl/Cmd + D = Duplicate row.
    if ((e.ctrlKey || e.metaKey) && e.keyCode === 68) {
      e.preventDefault();
      const gridApi = Drupal.ebAggrid.getActiveGrid();
      if (gridApi) {
        const focusedCell = gridApi.getFocusedCell();
        if (focusedCell) {
          const rowNode = gridApi.getDisplayedRowAtIndex(focusedCell.rowIndex);
          if (rowNode) {
            Drupal.ebAggrid.duplicateRow({
              api: gridApi,
              node: rowNode,
              context: gridApi.gridOptions.context
            });
          }
        }
      }
      return;
    }

    // Delete key = Delete selected rows.
    if (e.keyCode === 46) {
      const gridApi = Drupal.ebAggrid.getActiveGrid();
      if (gridApi) {
        const selectedNodes = gridApi.getSelectedNodes();
        if (selectedNodes.length > 0) {
          e.preventDefault();
          if (confirm(Drupal.t('Delete @count selected row(s)?', { '@count': selectedNodes.length }))) {
            const rowsToRemove = selectedNodes.map((node) => node.data);
            gridApi.applyTransaction({ remove: rowsToRemove });
            Drupal.ebAggrid.syncGridToHiddenField(gridApi, gridApi.gridOptions.context.gridType);
            Drupal.ebAggrid.updateUnsavedIndicators();
          }
        }
      }
    }
  };

  /**
   * Switch to a specific tab.
   *
   * @param {string} tabId
   *   The tab identifier.
   */
  Drupal.ebAggrid.switchTab = (tabId) => {
    // Update tab buttons.
    const tabs = document.querySelectorAll('.eb-sheet-tab');
    tabs.forEach((tab) => {
      if (tab.getAttribute('data-tab') === tabId) {
        tab.classList.add('eb-sheet-tab--active');
        tab.setAttribute('aria-selected', 'true');
      }
      else {
        tab.classList.remove('eb-sheet-tab--active');
        tab.setAttribute('aria-selected', 'false');
      }
    });

    // Update tab panels.
    const panels = document.querySelectorAll('.eb-tab-panel');
    panels.forEach((panel) => {
      if (panel.getAttribute('data-tab') === tabId) {
        panel.classList.add('eb-tab-panel--active');
      }
      else {
        panel.classList.remove('eb-tab-panel--active');
      }
    });

    // Resize the grid in the newly visible panel and refresh header.
    const gridType = Drupal.ebAggrid.TAB_GRID_MAP[tabId];

    if (gridType && Drupal.ebAggrid.grids[gridType]) {
      const gridApi = Drupal.ebAggrid.grids[gridType];
      setTimeout(() => {
        gridApi.sizeColumnsToFit();
        // Refresh header to update row count display.
        gridApi.refreshHeader();
      }, 50);
    }

    // Store current tab in session storage for persistence.
    Drupal.ebAggrid.storageSet('ebUi_activeTab', tabId);
  };

  /**
   * Update row counts displayed on tabs.
   */
  Drupal.ebAggrid.updateTabRowCounts = () => {
    const tabGridMap = Drupal.ebAggrid.TAB_GRID_MAP;

    Object.keys(tabGridMap).forEach((tabId) => {
      const gridType = tabGridMap[tabId];
      const tab = document.querySelector(`.eb-sheet-tab[data-tab="${tabId}"]`);
      if (!tab) {
        return;
      }

      let count = 0;
      const gridApi = Drupal.ebAggrid.grids[gridType];
      if (gridApi) {
        gridApi.forEachNode(() => {
          count++;
        });
      }
      else {
        // Grid not yet initialized, check settings.
        const settings = drupalSettings.ebAggrid;
        if (settings) {
          const dataKey = {
            bundle: 'bundleDefinitions',
            field: 'fieldDefinitions',
            field_group: 'fieldGroupDefinitions',
            display: 'displayFieldDefinitions',
            menu: 'menuDefinitions'
          }[gridType];
          if (dataKey && settings[dataKey]) {
            count = settings[dataKey].length;
          }
        }
      }

      // Update or create count badge.
      let countBadge = tab.querySelector('.eb-sheet-tab__count');
      if (count > 0) {
        if (!countBadge) {
          countBadge = document.createElement('span');
          countBadge.className = 'eb-sheet-tab__count';
          tab.appendChild(countBadge);
        }
        countBadge.textContent = count;
      }
      else if (countBadge) {
        countBadge.remove();
      }
    });

    // Refresh grid header to update row count display.
    Drupal.ebAggrid.refreshRowCountHeader();
  };

  /**
   * Refresh the row count displayed in grid headers.
   *
   * This updates the # column header which shows total row count.
   */
  Drupal.ebAggrid.refreshRowCountHeader = () => {
    // Get active tab's grid and refresh its header.
    const activeTab = document.querySelector('.eb-sheet-tab--active');
    if (!activeTab) {
      return;
    }

    const tabId = activeTab.getAttribute('data-tab');
    const gridType = Drupal.ebAggrid.TAB_GRID_MAP?.[tabId];

    if (gridType) {
      const gridApi = Drupal.ebAggrid.grids[gridType];
      if (gridApi) {
        gridApi.refreshHeader();
      }
    }
  };

  /**
   * Save column group state to session storage.
   *
   * @param {string} gridType
   *   The grid type.
   * @param {string} groupId
   *   The column group ID.
   * @param {boolean} isOpen
   *   Whether the group is open.
   */
  Drupal.ebAggrid.saveColumnGroupState = (gridType, groupId, isOpen) => {
    const stateKey = `ebUi_columnGroups_${gridType}`;
    const state = Drupal.ebAggrid.storageGet(stateKey, {});
    state[groupId] = isOpen;
    Drupal.ebAggrid.storageSet(stateKey, state);
  };

  /**
   * Get column group state from session storage.
   *
   * @param {string} gridType
   *   The grid type.
   * @param {string} groupId
   *   The column group ID.
   *
   * @return {boolean|null}
   *   The saved state or null if not saved.
   */
  Drupal.ebAggrid.getColumnGroupState = (gridType, groupId) => {
    const stateKey = `ebUi_columnGroups_${gridType}`;
    const state = Drupal.ebAggrid.storageGet(stateKey, {});
    return Object.prototype.hasOwnProperty.call(state, groupId) ? state[groupId] : null;
  };

  /**
   * Drupal behavior for EB UI grid initialization.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.ebUiGrid = {
    attach: (context) => {
      const settings = drupalSettings.ebAggrid;
      if (!settings?.discovery) {
        return;
      }

      const { discovery } = settings;

      // Prefetch bundles for common entity types.
      once('eb-prefetch', 'body', context).forEach(() => {
        Drupal.ebAggrid.prefetchCommonBundles();
      });

      // Add debug mode indicator if debug mode is active.
      once('eb-debug-indicator', 'body', context).forEach(() => {
        if (Drupal.ebAggrid.addDebugIndicator) {
          Drupal.ebAggrid.addDebugIndicator();
        }
      });

      // Initialize Bundle grid.
      once('eb-bundle-grid', '#eb-bundle-grid', context).forEach(() => {
        Drupal.ebAggrid.initGrid(
          'bundle',
          'eb-bundle-grid',
          'bundle-definitions-data',
          Drupal.ebAggrid.getBundleColumns(discovery),
          settings.bundleDefinitions || []
        );
      });

      // Initialize Field grid.
      // Sort by entity_type first (for visual grouping), then by weight within each group.
      once('eb-field-grid', '#eb-field-grid', context).forEach(() => {
        const fieldData = (settings.fieldDefinitions || [])
          .slice()
          .sort((a, b) => {
            // Primary sort: entity_type (alphabetical)
            const entityCompare = (a.entity_type || '').localeCompare(b.entity_type || '');
            if (entityCompare !== 0) {
              return entityCompare;
            }
            // Secondary sort: bundle (alphabetical within same entity_type)
            const bundleCompare = (a.bundle || '').localeCompare(b.bundle || '');
            if (bundleCompare !== 0) {
              return bundleCompare;
            }
            // Tertiary sort: weight (numeric within same bundle)
            return (a.weight || 0) - (b.weight || 0);
          });
        Drupal.ebAggrid.initGrid(
          'field',
          'eb-field-grid',
          'field-definitions-data',
          Drupal.ebAggrid.getFieldColumns(discovery),
          fieldData
        );
      });

      // Initialize Field Group grid (only if extension is enabled).
      once('eb-field-group-grid', '#eb-field-group-grid', context).forEach((gridElement) => {
        const fieldGroupExt = settings.extensions?.field_group;

        if (!fieldGroupExt?.enabled) {
          // Show disabled notice instead of grid.
          const notice = document.createElement('div');
          notice.className = 'eb-extension-notice';
          notice.innerHTML = `
            <div class="eb-extension-notice__icon">📁</div>
            <p>The <strong>Field Groups</strong> feature requires the
            <code>eb_field_group</code> module to be installed and enabled.</p>
            <p>Enable the module to organize fields into collapsible groups,
            tabs, accordions, and other containers.</p>
          `;
          gridElement.parentNode.insertBefore(notice, gridElement);
          gridElement.style.display = 'none';

          // Hide the "Add Field Group" button if present.
          const addButton = gridElement.closest('.eb-tab-panel')?.querySelector('[data-grid-target="field_group"]');
          if (addButton) {
            addButton.style.display = 'none';
          }
          return;
        }

        Drupal.ebAggrid.initGrid(
          'field_group',
          'eb-field-group-grid',
          'field-group-definitions-data',
          Drupal.ebAggrid.getFieldGroupColumns(discovery),
          settings.fieldGroupDefinitions || []
        );
      });

      // Initialize Display Field grid.
      // Filter out 'default' mode entries - users configure these in the Fields tab.
      // Sort by entity_type, bundle, mode, then weight for visual grouping.
      once('eb-display-grid', '#eb-display-grid', context).forEach(() => {
        const displayData = (settings.displayFieldDefinitions || [])
          .filter((row) => row.mode !== 'default')
          .sort((a, b) => {
            // Primary sort: entity_type
            const entityCompare = (a.entity_type || '').localeCompare(b.entity_type || '');
            if (entityCompare !== 0) {
              return entityCompare;
            }
            // Secondary sort: bundle
            const bundleCompare = (a.bundle || '').localeCompare(b.bundle || '');
            if (bundleCompare !== 0) {
              return bundleCompare;
            }
            // Tertiary sort: mode
            const modeCompare = (a.mode || '').localeCompare(b.mode || '');
            if (modeCompare !== 0) {
              return modeCompare;
            }
            // Quaternary sort: weight
            return (a.weight || 0) - (b.weight || 0);
          });
        Drupal.ebAggrid.initGrid(
          'display',
          'eb-display-grid',
          'display-field-definitions-data',
          Drupal.ebAggrid.getDisplayFieldColumns(discovery),
          displayData
        );
      });

      // Initialize Menu grid.
      once('eb-menu-grid', '#eb-menu-grid', context).forEach(() => {
        Drupal.ebAggrid.initGrid(
          'menu',
          'eb-menu-grid',
          'menu-definitions-data',
          Drupal.ebAggrid.getMenuColumns(discovery),
          settings.menuDefinitions || []
        );
      });

      // Attach add row button handlers.
      once('eb-add-buttons', '[data-grid-action="add-row"]', context).forEach((button) => {
        button.addEventListener('click', (e) => {
          e.preventDefault();
          const gridTarget = button.getAttribute('data-grid-target');
          Drupal.ebAggrid.addRow(gridTarget);
        });
      });
    }
  };

  /**
   * Sync all grids before form submission.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.ebUiFormSubmit = {
    attach: (context) => {
      once('eb-form-submit', 'form#eb-aggrid-form', context).forEach((form) => {
        form.addEventListener('submit', (e) => {
          // Flush any pending debounced syncs immediately.
          Drupal.ebAggrid.flushPendingSync();

          // Check for validation errors.
          const errors = Drupal.ebAggrid.getValidationErrors();
          if (errors) {
            e.preventDefault();
            const errorMsg = Drupal.t('Please fix validation errors before saving.');
            alert(errorMsg);
          }
          else {
            // Form is being submitted, disable beforeunload warning.
            Drupal.ebAggrid.isSubmitting = true;
          }
        });
      });
    }
  };

  /**
   * Unsaved changes warning using native browser dialog.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.ebUiUnsavedWarning = {
    attach: (context) => {
      once('eb-unsaved-warning', 'form#eb-aggrid-form', context).forEach(() => {
        // Use native browser beforeunload dialog.
        window.addEventListener('beforeunload', (e) => {
          if (Drupal.ebAggrid.isSubmitting) {
            return;
          }

          if (Drupal.ebAggrid.hasUnsavedChanges()) {
            e.preventDefault();
            e.returnValue = '';
            return '';
          }
        });
      });
    }
  };

  /**
   * Prevent form submission on Enter key in grids and add keyboard shortcuts.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.ebUiKeyboard = {
    attach: (context) => {
      once('eb-keyboard', '.eb-grid', context).forEach((grid) => {
        grid.addEventListener('keydown', (e) => {
          // Allow Enter for cell editing, but prevent form submission.
          if (e.keyCode === 13) {
            e.stopPropagation();
          }

          // Handle keyboard shortcuts.
          Drupal.ebAggrid.handleKeyboardShortcut(e);
        });
      });
    }
  };

  /**
   * Toolbar button handlers.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.ebUiToolbar = {
    attach: (context) => {
      // Undo button.
      once('eb-undo-btn', '.eb-undo-btn', context).forEach((btn) => {
        btn.addEventListener('click', (e) => {
          e.preventDefault();
          Drupal.ebAggrid.undo();
        });
      });

      // Redo button.
      once('eb-redo-btn', '.eb-redo-btn', context).forEach((btn) => {
        btn.addEventListener('click', (e) => {
          e.preventDefault();
          Drupal.ebAggrid.redo();
        });
      });

      // Revert All button.
      once('eb-revert-btn', '.eb-revert-btn', context).forEach((btn) => {
        btn.addEventListener('click', (e) => {
          e.preventDefault();
          Drupal.ebAggrid.revertAll();
        });
      });

      // Validate All button.
      once('eb-validate-btn', '.eb-validate-btn', context).forEach((btn) => {
        btn.addEventListener('click', (e) => {
          e.preventDefault();
          Drupal.ebAggrid.validateAll().then((result) => {
            Drupal.ebUi.openValidationModal(result);
          });
        });
      });

      // Preview button.
      once('eb-preview-btn', '.eb-preview-btn', context).forEach((btn) => {
        btn.addEventListener('click', (e) => {
          e.preventDefault();

          // Helper to build payload from grid data.
          const buildPayload = () => {
            const allData = Drupal.ebAggrid.collectAllGridData();
            const settingsEl = document.getElementById('eb-settings-data');
            const settings = settingsEl?.value ? JSON.parse(settingsEl.value) : {};
            return { ...settings, ...allData };
          };

          // Open the preview modal with validate-first flow.
          Drupal.ebUi.openPreviewModal({
            validateFn: () => {
              return Drupal.ebAggrid.validateAll();
            },
            previewFn: () => {
              return Drupal.ebUi.fetchJson('/eb/api/preview', {
                method: 'POST',
                body: JSON.stringify(buildPayload())
              });
            }
          });
        });
      });

      // Quick filter input.
      once('eb-quick-filter', '.eb-quick-filter', context).forEach((input) => {
        let debounceTimer = null;
        input.addEventListener('input', (e) => {
          clearTimeout(debounceTimer);
          debounceTimer = setTimeout(() => {
            Drupal.ebAggrid.applyQuickFilter(e.target.value);
          }, 200);
        });
      });

      // Quick filter clear button.
      once('eb-filter-clear', '.eb-filter-clear', context).forEach((btn) => {
        btn.addEventListener('click', (e) => {
          e.preventDefault();
          const input = document.querySelector('.eb-quick-filter');
          if (input) {
            input.value = '';
            Drupal.ebAggrid.applyQuickFilter('');
          }
        });
      });

      // Import from Drupal button.
      once('eb-import-drupal-btn', '.eb-import-drupal-btn', context).forEach((btn) => {
        btn.addEventListener('click', (e) => {
          e.preventDefault();
          const gridType = btn.getAttribute('data-grid-target');
          if (gridType) {
            Drupal.ebUi.openImportModal({
              targetGridType: gridType,
              onImport: (config, entityType, bundle, targetGrid) => {
                Drupal.ebAggrid.mergeImportedConfig(config, targetGrid, entityType, bundle);
              }
            });
          }
        });
      });

      // Help button toggle.
      once('eb-help-btn', '.eb-help-btn', context).forEach((btn) => {
        btn.addEventListener('click', (e) => {
          e.preventDefault();
          const helpPanel = document.querySelector('.eb-help-panel');
          if (helpPanel) {
            const isVisible = helpPanel.style.display !== 'none';
            helpPanel.style.display = isVisible ? 'none' : 'block';
            btn.setAttribute('aria-expanded', isVisible ? 'false' : 'true');
          }
        });
      });

      // Fullscreen button toggle.
      once('eb-fullscreen-btn', '.eb-fullscreen-btn', context).forEach((btn) => {
        btn.addEventListener('click', (e) => {
          e.preventDefault();
          Drupal.ebAggrid.toggleFullscreen();
        });
      });
    }
  };

  /**
   * Excel-like bottom sheet tabs functionality.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.ebUiSheetTabs = {
    attach: (context) => {
      once('eb-sheet-tabs', '.eb-sheet-tabs', context).forEach((tabContainer) => {
        const tabs = tabContainer.querySelectorAll('.eb-sheet-tab');

        tabs.forEach((tab) => {
          tab.addEventListener('click', (e) => {
            e.preventDefault();
            const targetTab = tab.getAttribute('data-tab');
            Drupal.ebAggrid.switchTab(targetTab);
          });
        });

        // Initialize row counts on tabs.
        Drupal.ebAggrid.updateTabRowCounts();
      });

      // Keyboard navigation for tabs (arrow keys).
      once('eb-tab-keyboard', '.eb-sheet-tab', context).forEach((tab) => {
        tab.addEventListener('keydown', (e) => {
          const tabs = document.querySelectorAll('.eb-sheet-tab');
          const currentIndex = Array.from(tabs).indexOf(e.target);
          let newIndex = currentIndex;

          if (e.key === 'ArrowRight' || e.key === 'ArrowDown') {
            e.preventDefault();
            newIndex = (currentIndex + 1) % tabs.length;
          }
          else if (e.key === 'ArrowLeft' || e.key === 'ArrowUp') {
            e.preventDefault();
            newIndex = (currentIndex - 1 + tabs.length) % tabs.length;
          }
          else if (e.key === 'Home') {
            e.preventDefault();
            newIndex = 0;
          }
          else if (e.key === 'End') {
            e.preventDefault();
            newIndex = tabs.length - 1;
          }

          if (newIndex !== currentIndex) {
            tabs[newIndex].focus();
            tabs[newIndex].click();
          }
        });
      });
    }
  };

  /**
   * Restore active tab from session storage.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.ebUiRestoreTab = {
    attach: (context) => {
      once('eb-restore-tab', '.eb-spreadsheet-container', context).forEach(() => {
        const savedTab = Drupal.ebAggrid.storageGet('ebUi_activeTab', null);
        if (savedTab) {
          // Wait for grids to initialize.
          setTimeout(() => {
            Drupal.ebAggrid.switchTab(savedTab);
          }, 200);
        }
      });
    }
  };

  /**
   * Fullscreen mode handling.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.ebUiFullscreen = {
    attach: (context) => {
      once('eb-fullscreen', 'body', context).forEach(() => {
        // Listen for fullscreen change events.
        const fullscreenEvents = [
          'fullscreenchange',
          'webkitfullscreenchange',
          'mozfullscreenchange',
          'MSFullscreenChange'
        ];

        fullscreenEvents.forEach((eventName) => {
          document.addEventListener(eventName, () => {
            Drupal.ebAggrid.updateFullscreenButton();
          });
        });
      });
    }
  };

  /**
   * Resizable grid container.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.ebUiResizeHandle = {
    attach: (context) => {
      once('eb-resize-handle', '.eb-resize-handle', context).forEach(() => {
        Drupal.ebAggrid.initResizeHandle();
      });
    }
  };

})(Drupal, drupalSettings, once);
