/**
 * @file
 * Grid-specific modal management for EB Aggrid module.
 *
 * This file contains grid-specific modals only. Shared modals (validation,
 * preview, import) are provided by eb_ui/modals library.
 *
 * Grid-specific modals:
 * - Settings modal (field storage/config settings)
 * - Format settings modal (field group format settings)
 * - Display settings modal (widget/formatter settings)
 */

(function (Drupal, drupalSettings) {
  'use strict';

  /**
   * Initialize the ebAggrid namespace if not exists.
   */
  Drupal.ebAggrid = Drupal.ebAggrid || {};

  // ============================================
  // MERGE IMPORTED CONFIG (Grid-specific)
  // ============================================

  /**
   * Merge imported configuration into grid.
   *
   * @param {Object} config
   *   The imported configuration.
   * @param {string} gridType
   *   The target grid type.
   * @param {string} entityType
   *   The entity type.
   * @param {string} bundle
   *   The bundle.
   */
  Drupal.ebAggrid.mergeImportedConfig = (config, gridType, entityType, bundle) => {
    const gridApi = Drupal.ebAggrid.grids[gridType];
    if (!gridApi) {
      return;
    }

    const rowsToAdd = [];

    // Map configuration to grid row format based on grid type.
    if (gridType === 'field' && config.field_definitions) {
      config.field_definitions.forEach((field) => {
        rowsToAdd.push({
          entity_type: entityType,
          bundle,
          field_name: field.field_name,
          field_type: field.field_type,
          label: field.label,
          description: field.description || '',
          required: field.required || false,
          cardinality: field.cardinality || 1,
          translatable: false,
          field_storage_settings: field.storage_settings || {},
          field_config_settings: field.settings || {}
        });
      });
    }
    else if (gridType === 'field_group' && config.field_group_definitions) {
      config.field_group_definitions.forEach((group) => {
        rowsToAdd.push({
          entity_type: entityType,
          bundle,
          display_type: 'form',
          mode: 'default',
          group_name: group.group_name,
          label: group.label,
          format_type: group.format_type,
          weight: group.weight || 0,
          format_settings: group.format_settings || {}
        });
      });
    }
    else if (gridType === 'display' && config.display_field_definitions) {
      config.display_field_definitions.forEach((display) => {
        rowsToAdd.push({
          entity_type: entityType,
          bundle,
          display_type: display.display_mode || 'form',
          mode: 'default',
          field_name: display.field_name,
          widget: display.type || '',
          widget_settings: display.settings || {},
          weight: display.weight || 0,
          group: ''
        });
      });
    }

    if (rowsToAdd.length > 0) {
      gridApi.applyTransaction({ add: rowsToAdd });
      Drupal.ebAggrid.syncGridToHiddenField(gridApi, gridType);
      alert(Drupal.t('Imported @count items.', { '@count': rowsToAdd.length }));
    }
    else {
      alert(Drupal.t('No items to import for this configuration.'));
    }
  };

  // ============================================
  // SETTINGS MODAL (Field storage/config settings)
  // ============================================

  /**
   * Store reference to current settings modal context.
   *
   * @type {Object|null}
   */
  Drupal.ebAggrid._settingsModalNode = null;
  Drupal.ebAggrid._settingsModalData = null;

  /**
   * Open settings modal for a field row.
   *
   * @param {Object} node
   *   The AG-Grid row node.
   * @param {Object} rowData
   *   The row data object.
   */
  Drupal.ebAggrid.openSettingsModal = (node, rowData) => {
    const fieldType = rowData.field_type;
    if (!fieldType) {
      alert(Drupal.t('Please select a Field Type first.'));
      return;
    }

    Drupal.ebUi.openModal('eb-settings-modal', Drupal.ebAggrid.createSettingsModal);

    const modal = document.getElementById('eb-settings-modal');

    // Store reference to node for saving.
    Drupal.ebAggrid._settingsModalNode = node;
    Drupal.ebAggrid._settingsModalData = rowData;

    // Update modal title.
    const fieldLabel = rowData.label || rowData.field_name || 'Field';
    const typeLabel = Drupal.ebAggrid.getFieldTypeLabel(fieldType);
    const title = modal.querySelector('.eb-modal__title');
    if (title) {
      title.textContent = Drupal.t('Configure @type: @label', {
        '@type': typeLabel,
        '@label': fieldLabel
      });
    }

    // Render form.
    const content = modal.querySelector('.eb-modal__content');
    if (content && Drupal.ebAggrid.renderSettingsForm) {
      content.innerHTML = Drupal.ebAggrid.renderSettingsForm(fieldType, rowData);

      // Attach event listeners to form elements.
      if (Drupal.ebAggrid.attachSettingsFormListeners) {
        Drupal.ebAggrid.attachSettingsFormListeners(modal, fieldType);
      }

      // Initialize AJAX loading for widget/formatter settings panels.
      if (Drupal.ebAggrid.initPluginSettingsPanels) {
        Drupal.ebAggrid.initPluginSettingsPanels(modal, rowData);
      }
    }
  };

  /**
   * Create settings modal element.
   */
  Drupal.ebAggrid.createSettingsModal = () => {
    const modal = Drupal.ebUi.createModal({
      id: 'eb-settings-modal',
      title: Drupal.t('Configure Field Settings'),
      dialogClass: 'eb-modal__dialog--settings',
      footerButtons: [
        { label: Drupal.t('Cancel'), class: 'eb-modal__cancel' },
        { label: Drupal.t('Apply Settings'), primary: true, class: 'eb-settings-apply' }
      ]
    });

    // Additional event listeners.
    const cancelBtn = modal.querySelector('.eb-modal__cancel');
    const applyBtn = modal.querySelector('.eb-settings-apply');

    if (cancelBtn) {
      cancelBtn.addEventListener('click', Drupal.ebAggrid.closeSettingsModal);
    }

    if (applyBtn) {
      applyBtn.addEventListener('click', Drupal.ebAggrid.applySettingsModal);
    }
  };

  /**
   * Close settings modal.
   */
  Drupal.ebAggrid.closeSettingsModal = () => {
    Drupal.ebUi.closeModal('eb-settings-modal');
    Drupal.ebAggrid._settingsModalNode = null;
    Drupal.ebAggrid._settingsModalData = null;
  };

  /**
   * Apply settings from modal to grid row.
   */
  Drupal.ebAggrid.applySettingsModal = () => {
    const modal = document.getElementById('eb-settings-modal');
    const node = Drupal.ebAggrid._settingsModalNode;
    const rowData = Drupal.ebAggrid._settingsModalData;

    if (!modal || !node || !rowData) {
      Drupal.ebAggrid.closeSettingsModal();
      return;
    }

    const fieldType = rowData.field_type;

    // Collect form values for storage/config (client-side forms).
    let storageSettings = {};
    let configSettings = {};

    if (Drupal.ebAggrid.collectFormValues) {
      storageSettings = Drupal.ebAggrid.collectFormValues(modal, 'storage', fieldType);
      configSettings = Drupal.ebAggrid.collectFormValues(modal, 'config', fieldType);
    }

    // Collect widget settings from Drupal-rendered AJAX form.
    let widgetSettings = rowData.widget_settings || {};
    const widgetContainer = modal.querySelector('.eb-plugin-settings-container[data-plugin-type="widget"]');
    if (widgetContainer && Drupal.ebAggrid.collectDrupalFormSettings) {
      const collected = Drupal.ebAggrid.collectDrupalFormSettings(widgetContainer);
      // The form may wrap settings in various ways; normalize them.
      const normalized = Drupal.ebAggrid.normalizePluginSettings(collected);
      if (Object.keys(normalized).length > 0) {
        widgetSettings = { ...widgetSettings, ...normalized };
      }
    }

    // Collect formatter settings from Drupal-rendered AJAX form.
    let formatterSettings = rowData.formatter_settings || {};
    const formatterContainer = modal.querySelector('.eb-plugin-settings-container[data-plugin-type="formatter"]');
    if (formatterContainer && Drupal.ebAggrid.collectDrupalFormSettings) {
      const collected = Drupal.ebAggrid.collectDrupalFormSettings(formatterContainer);
      // The form may wrap settings in various ways; normalize them.
      const normalized = Drupal.ebAggrid.normalizePluginSettings(collected);
      if (Object.keys(normalized).length > 0) {
        formatterSettings = { ...formatterSettings, ...normalized };
      }
    }

    // Merge with existing values (preserve values not in form).
    const existingStorage = rowData.field_storage_settings || {};
    const existingConfig = rowData.field_config_settings || {};

    const newStorage = { ...existingStorage, ...storageSettings };
    const newConfig = { ...existingConfig, ...configSettings };

    // Update node data directly (setDataValue doesn't work well for objects).
    node.data.field_storage_settings = newStorage;
    node.data.field_config_settings = newConfig;
    node.data.widget_settings = widgetSettings;
    node.data.formatter_settings = formatterSettings;

    // Refresh the row to reflect changes and sync to hidden field.
    const api = Drupal.ebAggrid.grids?.field;
    if (api) {
      // Use applyTransaction to properly notify AG-Grid of the data change.
      api.applyTransaction({ update: [node.data] });
      // Force refresh of the Settings cell to update button label.
      api.refreshCells({
        rowNodes: [node],
        columns: ['_settings'],
        force: true
      });
      // Sync changes to hidden field for form submission.
      Drupal.ebAggrid.syncGridToHiddenField(api, 'field');
    }

    // Close modal.
    Drupal.ebAggrid.closeSettingsModal();
  };

  /**
   * Normalize plugin settings collected from Drupal forms.
   *
   * Drupal Form API may wrap settings in nested structures. This function
   * flattens them to a simple key-value object.
   *
   * @param {Object} settings
   *   The raw collected settings.
   *
   * @return {Object}
   *   Normalized settings object.
   */
  Drupal.ebAggrid.normalizePluginSettings = (settings) => {
    // If settings has a 'settings' key, unwrap it.
    if (settings.settings && typeof settings.settings === 'object') {
      return settings.settings;
    }
    return settings;
  };

  /**
   * Get human-readable field type label.
   *
   * @param {string} fieldType
   *   The field type ID.
   *
   * @return {string}
   *   The human-readable label.
   */
  Drupal.ebAggrid.getFieldTypeLabel = (fieldType) => {
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    const labels = discovery.fieldTypeLabels || {};
    return labels[fieldType] || fieldType;
  };

  // ============================================
  // FORMAT SETTINGS MODAL (Field Groups)
  // ============================================

  /**
   * Store reference to current format settings modal context.
   *
   * @type {Object|null}
   */
  Drupal.ebAggrid._formatSettingsModalNode = null;
  Drupal.ebAggrid._formatSettingsModalData = null;

  /**
   * Open format settings modal for a field group row.
   *
   * @param {Object} node
   *   The AG-Grid row node.
   * @param {Object} rowData
   *   The row data object.
   */
  Drupal.ebAggrid.openFormatSettingsModal = (node, rowData) => {
    const formatType = rowData.format_type;
    if (!formatType) {
      alert(Drupal.t('Please select a Format Type first.'));
      return;
    }

    Drupal.ebUi.openModal('eb-format-settings-modal', Drupal.ebAggrid.createFormatSettingsModal);

    const modal = document.getElementById('eb-format-settings-modal');

    // Store reference to node for saving.
    Drupal.ebAggrid._formatSettingsModalNode = node;
    Drupal.ebAggrid._formatSettingsModalData = rowData;

    // Update modal title.
    const groupLabel = rowData.label || rowData.group_name || 'Field Group';
    const title = modal.querySelector('.eb-modal__title');
    if (title) {
      title.textContent = Drupal.t('Configure Format Settings: @label', {
        '@label': groupLabel
      });
    }

    // Show loading state.
    const content = modal.querySelector('.eb-modal__content');
    content.innerHTML = `<div class="eb-modal__loading">${Drupal.t('Loading settings form...')}</div>`;

    // Fetch settings form via AJAX.
    const displayType = rowData.display_type || 'form';
    // Use node.data to get the latest data (rowData might be stale reference).
    let currentSettings = node.data.format_settings || {};

    // If no label in format_settings, use the field group's label as initial value.
    if (!currentSettings.label && node.data.label) {
      currentSettings = { ...currentSettings, label: node.data.label };
    }

    Drupal.ebUi.fetchJson(`/eb/api/format-settings/${encodeURIComponent(formatType)}`, {
      method: 'POST',
      headers: Drupal.ebUi.getPostHeaders(),
      body: JSON.stringify({
        display_type: displayType,
        current_settings: currentSettings
      })
    })
      .then((result) => {
        if (!result.success) {
          content.innerHTML = `<div class="eb-modal__error">${Drupal.t('Error: @message', { '@message': result.message || 'Unknown error' })}</div>`;
          return;
        }

        if (!result.hasSettings) {
          content.innerHTML = `<div class="eb-modal__message">${result.message || Drupal.t('No configurable settings for this format type.')}</div>`;
          // Disable apply button.
          const applyBtn = modal.querySelector('.eb-format-settings-apply');
          if (applyBtn) {
            applyBtn.disabled = true;
          }
          return;
        }

        // Render form HTML.
        content.innerHTML = result.html;

        // Enable apply button.
        const applyBtn = modal.querySelector('.eb-format-settings-apply');
        if (applyBtn) {
          applyBtn.disabled = false;
        }
      })
      .catch((error) => {
        content.innerHTML = `<div class="eb-modal__error">${Drupal.t('Error loading settings: @error', { '@error': error.message })}</div>`;
      });
  };

  /**
   * Create format settings modal element.
   */
  Drupal.ebAggrid.createFormatSettingsModal = () => {
    const modal = Drupal.ebUi.createModal({
      id: 'eb-format-settings-modal',
      title: Drupal.t('Configure Format Settings'),
      dialogClass: 'eb-modal__dialog--settings',
      footerButtons: [
        { label: Drupal.t('Cancel'), class: 'eb-modal__cancel' },
        { label: Drupal.t('Apply Settings'), primary: true, class: 'eb-format-settings-apply' }
      ]
    });

    // Additional event listeners.
    const cancelBtn = modal.querySelector('.eb-modal__cancel');
    const applyBtn = modal.querySelector('.eb-format-settings-apply');

    if (cancelBtn) {
      cancelBtn.addEventListener('click', Drupal.ebAggrid.closeFormatSettingsModal);
    }

    if (applyBtn) {
      applyBtn.addEventListener('click', Drupal.ebAggrid.applyFormatSettingsModal);
    }
  };

  /**
   * Close format settings modal.
   */
  Drupal.ebAggrid.closeFormatSettingsModal = () => {
    Drupal.ebUi.closeModal('eb-format-settings-modal');
    Drupal.ebAggrid._formatSettingsModalNode = null;
    Drupal.ebAggrid._formatSettingsModalData = null;
  };

  /**
   * Apply format settings from modal to grid row.
   */
  Drupal.ebAggrid.applyFormatSettingsModal = () => {
    const modal = document.getElementById('eb-format-settings-modal');
    const node = Drupal.ebAggrid._formatSettingsModalNode;
    const rowData = Drupal.ebAggrid._formatSettingsModalData;

    if (!modal || !node || !rowData) {
      Drupal.ebAggrid.closeFormatSettingsModal();
      return;
    }

    // Collect form values from the Drupal-rendered form.
    const container = modal.querySelector('.eb-plugin-settings-form');
    let formatSettings = rowData.format_settings || {};

    if (container && Drupal.ebAggrid.collectDrupalFormSettings) {
      const collected = Drupal.ebAggrid.collectDrupalFormSettings(container);
      // Normalize settings (may be wrapped in 'settings' key).
      const normalized = Drupal.ebAggrid.normalizePluginSettings(collected);
      if (Object.keys(normalized).length > 0) {
        formatSettings = { ...formatSettings, ...normalized };
      }
    }

    // Update node data.
    node.data.format_settings = formatSettings;

    // Refresh the row to reflect changes and sync to hidden field.
    const api = Drupal.ebAggrid.grids?.field_group;
    if (api) {
      // Use applyTransaction to properly notify AG-Grid of the data change.
      api.applyTransaction({ update: [node.data] });
      // Force refresh of the Format Settings cell to update button label.
      api.refreshCells({
        rowNodes: [node],
        columns: ['_format_settings'],
        force: true
      });
      // Sync changes to hidden field for form submission.
      Drupal.ebAggrid.syncGridToHiddenField(api, 'field_group');
    }

    // Close modal.
    Drupal.ebAggrid.closeFormatSettingsModal();
  };

  /**
   * Get human-readable format type label.
   *
   * @param {string} formatType
   *   The format type ID.
   *
   * @return {string}
   *   The human-readable label.
   */
  Drupal.ebAggrid.getFormatTypeLabel = (formatType) => {
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    const formatTypes = discovery.fieldGroupFormatTypes || [];
    // Format types are a simple array, so just return the ID for now.
    return formatType || '';
  };

  // ============================================
  // DISPLAY SETTINGS MODAL
  // ============================================

  /**
   * Open the display settings modal for a display field definition.
   *
   * @param {Object} node
   *   The AG-Grid row node.
   * @param {Object} rowData
   *   The row data object.
   */
  Drupal.ebAggrid.openDisplaySettingsModal = (node, rowData) => {
    // Check if widget or formatter is selected.
    if (!rowData.widget && !rowData.formatter) {
      alert(Drupal.t('Please select a Widget or Formatter type first.'));
      return;
    }

    Drupal.ebUi.openModal('eb-display-settings-modal', Drupal.ebAggrid.createDisplaySettingsModal);

    const modal = document.getElementById('eb-display-settings-modal');
    if (!modal) {
      console.error('Display settings modal not found');
      return;
    }

    // Store reference to node for saving.
    Drupal.ebAggrid._displaySettingsModalNode = node;
    Drupal.ebAggrid._displaySettingsModalData = rowData;

    // Update modal title with context.
    const fieldName = rowData.field_name || 'Field';
    const displayType = rowData.display_type || 'view';
    const mode = rowData.mode || 'default';
    const title = modal.querySelector('.eb-modal__title');
    if (title) {
      title.textContent = Drupal.t('Display Settings: @field (@mode @type)', {
        '@field': fieldName,
        '@mode': mode,
        '@type': displayType
      });
    }

    // Render the modal content.
    const content = modal.querySelector('.eb-modal__content');
    if (content) {
      const formHtml = Drupal.ebAggrid.renderDisplaySettingsForm(rowData);
      content.innerHTML = formHtml;

      // Initialize AJAX loading for widget/formatter settings panels.
      Drupal.ebAggrid.initDisplaySettingsPanels(modal, rowData);
    }
    else {
      console.error('Modal content container not found');
    }
  };

  /**
   * Render the display settings form HTML.
   *
   * @param {Object} rowData
   *   The row data.
   *
   * @return {string}
   *   The form HTML.
   */
  Drupal.ebAggrid.renderDisplaySettingsForm = (rowData) => {
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    const widgetLabel = rowData.widget ? (discovery.widgets?.[rowData.widget]?.label || rowData.widget) : '';
    const formatterLabel = rowData.formatter ? (discovery.formatters?.[rowData.formatter]?.label || rowData.formatter) : '';

    let html = '<div class="eb-display-settings-form">';

    // Widget Settings panel - use --active to make visible.
    html += '<div class="eb-display-settings-section">';
    html += '<h3 class="eb-display-settings-section__title">' + Drupal.t('Widget Settings') + '</h3>';
    if (rowData.widget) {
      html += '<div class="eb-display-settings-section__subtitle">' + Drupal.t('Type: @type', { '@type': widgetLabel }) + '</div>';
      html += '<div class="eb-plugin-settings-container" data-plugin-type="widget">';
      html += '<div class="eb-plugin-settings-loading">' + Drupal.t('Loading settings...') + '</div>';
      html += '</div>';
    }
    else {
      html += '<div class="eb-display-settings-section__empty">' + Drupal.t('No widget selected. Select a widget type in the grid to configure its settings.') + '</div>';
    }
    html += '</div>';

    // Formatter Settings panel.
    html += '<div class="eb-display-settings-section">';
    html += '<h3 class="eb-display-settings-section__title">' + Drupal.t('Formatter Settings') + '</h3>';
    if (rowData.formatter) {
      html += '<div class="eb-display-settings-section__subtitle">' + Drupal.t('Type: @type', { '@type': formatterLabel }) + '</div>';
      html += '<div class="eb-plugin-settings-container" data-plugin-type="formatter">';
      html += '<div class="eb-plugin-settings-loading">' + Drupal.t('Loading settings...') + '</div>';
      html += '</div>';
    }
    else {
      html += '<div class="eb-display-settings-section__empty">' + Drupal.t('No formatter selected. Select a formatter type in the grid to configure its settings.') + '</div>';
    }
    html += '</div>';

    html += '</div>';
    return html;
  };

  /**
   * Initialize display settings panels with AJAX-loaded forms.
   *
   * @param {HTMLElement} modal
   *   The modal element.
   * @param {Object} rowData
   *   The row data.
   */
  Drupal.ebAggrid.initDisplaySettingsPanels = async (modal, rowData) => {
    // Try to determine the field type from the field definition.
    const fieldName = rowData.field_name;
    const fieldData = Drupal.ebAggrid.findFieldDefinition?.(fieldName);
    const fieldType = fieldData?.field_type || 'string';
    const entityType = rowData.entity_type || 'node';
    const bundle = rowData.bundle || '';

    const context = {
      field_type: fieldType,
      entity_type: entityType,
      bundle: bundle
    };

    // Load widget settings if widget is set.
    if (rowData.widget && Drupal.ebAggrid.fetchPluginSettingsForm) {
      const widgetContainer = modal.querySelector('.eb-plugin-settings-container[data-plugin-type="widget"]');
      if (widgetContainer) {
        try {
          const result = await Drupal.ebAggrid.fetchPluginSettingsForm('widget', rowData.widget, {
            ...context,
            current_settings: rowData.widget_settings || {}
          });

          if (result.html) {
            widgetContainer.innerHTML = result.html;
            // Attach Drupal behaviors to the new content.
            if (typeof Drupal.attachBehaviors === 'function') {
              Drupal.attachBehaviors(widgetContainer);
            }
          }
          else if (result.hasSettings === false) {
            widgetContainer.innerHTML = '<div class="eb-plugin-settings-empty">' +
              Drupal.t('This widget has no configurable settings.') + '</div>';
          }
        }
        catch (error) {
          widgetContainer.innerHTML = '<div class="eb-plugin-settings-error">' +
            Drupal.t('Failed to load widget settings.') + '</div>';
        }
      }
    }

    // Load formatter settings if formatter is set.
    if (rowData.formatter && Drupal.ebAggrid.fetchPluginSettingsForm) {
      const formatterContainer = modal.querySelector('.eb-plugin-settings-container[data-plugin-type="formatter"]');
      if (formatterContainer) {
        try {
          const result = await Drupal.ebAggrid.fetchPluginSettingsForm('formatter', rowData.formatter, {
            ...context,
            current_settings: rowData.formatter_settings || {}
          });

          if (result.html) {
            formatterContainer.innerHTML = result.html;
            // Attach Drupal behaviors to the new content.
            if (typeof Drupal.attachBehaviors === 'function') {
              Drupal.attachBehaviors(formatterContainer);
            }
          }
          else if (result.hasSettings === false) {
            formatterContainer.innerHTML = '<div class="eb-plugin-settings-empty">' +
              Drupal.t('This formatter has no configurable settings.') + '</div>';
          }
        }
        catch (error) {
          formatterContainer.innerHTML = '<div class="eb-plugin-settings-error">' +
            Drupal.t('Failed to load formatter settings.') + '</div>';
        }
      }
    }
  };

  /**
   * Create display settings modal element.
   */
  Drupal.ebAggrid.createDisplaySettingsModal = () => {
    const modal = Drupal.ebUi.createModal({
      id: 'eb-display-settings-modal',
      title: Drupal.t('Display Field Settings'),
      dialogClass: 'eb-modal__dialog--settings',
      footerButtons: [
        { label: Drupal.t('Cancel'), class: 'eb-modal__cancel' },
        { label: Drupal.t('Apply Settings'), primary: true, class: 'eb-display-settings-apply' }
      ]
    });

    // Event listeners.
    const cancelBtn = modal.querySelector('.eb-modal__cancel');
    const applyBtn = modal.querySelector('.eb-display-settings-apply');

    if (cancelBtn) {
      cancelBtn.addEventListener('click', Drupal.ebAggrid.closeDisplaySettingsModal);
    }

    if (applyBtn) {
      applyBtn.addEventListener('click', Drupal.ebAggrid.applyDisplaySettingsModal);
    }
  };

  /**
   * Close display settings modal.
   */
  Drupal.ebAggrid.closeDisplaySettingsModal = () => {
    Drupal.ebUi.closeModal('eb-display-settings-modal');
    Drupal.ebAggrid._displaySettingsModalNode = null;
    Drupal.ebAggrid._displaySettingsModalData = null;
  };

  /**
   * Apply display settings from modal to grid row.
   */
  Drupal.ebAggrid.applyDisplaySettingsModal = () => {
    const modal = document.getElementById('eb-display-settings-modal');
    const node = Drupal.ebAggrid._displaySettingsModalNode;
    const rowData = Drupal.ebAggrid._displaySettingsModalData;

    if (!modal || !node || !rowData) {
      Drupal.ebAggrid.closeDisplaySettingsModal();
      return;
    }

    // Collect widget settings from Drupal-rendered AJAX form.
    let widgetSettings = rowData.widget_settings || {};
    const widgetContainer = modal.querySelector('.eb-plugin-settings-container[data-plugin-type="widget"]');
    if (widgetContainer && Drupal.ebAggrid.collectDrupalFormSettings) {
      const collected = Drupal.ebAggrid.collectDrupalFormSettings(widgetContainer);
      const normalized = Drupal.ebAggrid.normalizePluginSettings(collected);
      if (Object.keys(normalized).length > 0) {
        widgetSettings = { ...widgetSettings, ...normalized };
      }
    }

    // Collect formatter settings from Drupal-rendered AJAX form.
    let formatterSettings = rowData.formatter_settings || {};
    const formatterContainer = modal.querySelector('.eb-plugin-settings-container[data-plugin-type="formatter"]');
    if (formatterContainer && Drupal.ebAggrid.collectDrupalFormSettings) {
      const collected = Drupal.ebAggrid.collectDrupalFormSettings(formatterContainer);
      const normalized = Drupal.ebAggrid.normalizePluginSettings(collected);
      if (Object.keys(normalized).length > 0) {
        formatterSettings = { ...formatterSettings, ...normalized };
      }
    }

    // Update node data.
    node.data.widget_settings = widgetSettings;
    node.data.formatter_settings = formatterSettings;

    // Refresh the row and sync to hidden field.
    const api = Drupal.ebAggrid.grids?.display;
    if (api) {
      api.applyTransaction({ update: [node.data] });
      api.refreshCells({
        rowNodes: [node],
        columns: ['_display_settings'],
        force: true
      });
      Drupal.ebAggrid.syncGridToHiddenField(api, 'display');
    }

    Drupal.ebAggrid.closeDisplaySettingsModal();
  };

})(Drupal, drupalSettings);
