/**
 * @file
 * Field-type-specific settings form builders for EB UI.
 *
 * Provides dynamic form rendering based on schemas from PHP DiscoveryService.
 * Widget/formatter settings are fetched server-side via AJAX to use Drupal's
 * native settingsForm() rendering.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  /**
   * Initialize the ebUi namespace if not exists.
   */
  Drupal.ebAggrid = Drupal.ebAggrid || {};

  /**
   * Track current AJAX requests for cancellation.
   *
   * @type {Object}
   */
  Drupal.ebAggrid._pluginSettingsRequests = {
    widget: null,
    formatter: null
  };

  /**
   * Fetch plugin settings form from server via AJAX.
   *
   * @param {string} pluginType
   *   The plugin type ('widget' or 'formatter').
   * @param {string} pluginId
   *   The plugin ID (e.g., 'string_textfield').
   * @param {Object} context
   *   Context data including field_type, current_settings, entity_type, bundle.
   *
   * @return {Promise<Object>}
   *   Promise resolving to {success, hasSettings, html|message}.
   */
  Drupal.ebAggrid.fetchPluginSettingsForm = async (pluginType, pluginId, context) => {
    // Cancel any existing request for this plugin type.
    if (Drupal.ebAggrid._pluginSettingsRequests[pluginType]) {
      Drupal.ebAggrid._pluginSettingsRequests[pluginType].abort();
    }

    const controller = new AbortController();
    Drupal.ebAggrid._pluginSettingsRequests[pluginType] = controller;

    const url = `/eb/api/plugin-settings/${encodeURIComponent(pluginType)}/${encodeURIComponent(pluginId)}`;

    try {
      const response = await fetch(url, {
        method: 'POST',
        headers: Drupal.ebUi.getPostHeaders(),
        body: JSON.stringify(context),
        signal: controller.signal
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || `HTTP ${response.status}`);
      }

      return await response.json();
    }
    catch (error) {
      if (error.name === 'AbortError') {
        // Request was cancelled, don't treat as error.
        return { success: false, cancelled: true };
      }
      throw error;
    }
    finally {
      Drupal.ebAggrid._pluginSettingsRequests[pluginType] = null;
    }
  };

  /**
   * Render loading state HTML for plugin settings panel.
   *
   * @param {string} pluginType
   *   The plugin type ('widget' or 'formatter').
   *
   * @return {string}
   *   HTML string for loading state.
   */
  Drupal.ebAggrid.renderPluginSettingsLoading = (pluginType) => {
    return `
      <div class="eb-settings-loading">
        <div class="eb-settings-loading__spinner"></div>
        <span class="eb-settings-loading__text">${Drupal.t('Loading @type settings...', { '@type': pluginType })}</span>
      </div>
    `;
  };

  /**
   * Render error state HTML for plugin settings panel.
   *
   * @param {string} pluginType
   *   The plugin type ('widget' or 'formatter').
   * @param {string} message
   *   The error message.
   *
   * @return {string}
   *   HTML string for error state.
   */
  Drupal.ebAggrid.renderPluginSettingsError = (pluginType, message) => {
    return `
      <div class="eb-settings-error">
        <div class="eb-settings-error__message">${Drupal.checkPlain(message)}</div>
        <button type="button" class="eb-settings-error__retry button button--small">
          ${Drupal.t('Retry')}
        </button>
      </div>
    `;
  };

  /**
   * Load and render plugin settings form into a panel.
   *
   * @param {HTMLElement} panel
   *   The panel element to render into.
   * @param {string} pluginType
   *   The plugin type ('widget' or 'formatter').
   * @param {string} pluginId
   *   The plugin ID.
   * @param {Object} rowData
   *   The current row data with field context.
   */
  Drupal.ebAggrid.loadPluginSettingsForm = async (panel, pluginType, pluginId, rowData) => {
    if (!pluginId) {
      panel.innerHTML = `<p class="eb-settings-empty">${Drupal.t('Select a @type type first in the grid.', { '@type': pluginType })}</p>`;
      return;
    }

    // Show loading state.
    panel.innerHTML = Drupal.ebAggrid.renderPluginSettingsLoading(pluginType);
    panel.classList.add('eb-settings-panel--loading');

    // Build context for the request.
    const context = {
      field_type: rowData.field_type,
      current_settings: rowData[`${pluginType}_settings`] || {},
      entity_type: rowData.entity_type || 'node',
      bundle: rowData.bundle || 'article',
      field_config_settings: rowData.field_config_settings || {},
      field_storage_settings: rowData.field_storage_settings || {}
    };

    try {
      const result = await Drupal.ebAggrid.fetchPluginSettingsForm(pluginType, pluginId, context);

      panel.classList.remove('eb-settings-panel--loading');

      if (result.cancelled) {
        // Request was cancelled, do nothing (another request is in progress).
        return;
      }

      if (!result.success) {
        panel.innerHTML = Drupal.ebAggrid.renderPluginSettingsError(pluginType, result.message || 'Unknown error');
        Drupal.ebAggrid.attachRetryHandler(panel, pluginType, pluginId, rowData);
        return;
      }

      if (!result.hasSettings) {
        panel.innerHTML = `<p class="eb-settings-empty">${Drupal.checkPlain(result.message)}</p>`;
        return;
      }

      // Insert the server-rendered HTML.
      panel.innerHTML = result.html;

      // Attach Drupal behaviors to the new content.
      Drupal.attachBehaviors(panel, drupalSettings);
    }
    catch (error) {
      panel.classList.remove('eb-settings-panel--loading');
      panel.innerHTML = Drupal.ebAggrid.renderPluginSettingsError(pluginType, error.message);
      Drupal.ebAggrid.attachRetryHandler(panel, pluginType, pluginId, rowData);
    }
  };

  /**
   * Attach retry handler to error panel.
   *
   * @param {HTMLElement} panel
   *   The panel element.
   * @param {string} pluginType
   *   The plugin type.
   * @param {string} pluginId
   *   The plugin ID.
   * @param {Object} rowData
   *   The row data.
   */
  Drupal.ebAggrid.attachRetryHandler = (panel, pluginType, pluginId, rowData) => {
    const retryBtn = panel.querySelector('.eb-settings-error__retry');
    if (retryBtn) {
      retryBtn.addEventListener('click', () => {
        Drupal.ebAggrid.loadPluginSettingsForm(panel, pluginType, pluginId, rowData);
      });
    }
  };

  /**
   * Collect settings from a Drupal-rendered form.
   *
   * Extracts values from form elements using Form API naming conventions.
   *
   * @param {HTMLElement} container
   *   The container element with the Drupal form.
   *
   * @return {Object}
   *   Collected settings as nested object.
   */
  Drupal.ebAggrid.collectDrupalFormSettings = (container) => {
    const settings = {};

    if (!container) {
      return settings;
    }

    // Find all form elements in the container.
    const inputs = container.querySelectorAll('input, textarea, select');

    inputs.forEach((input) => {
      // Skip buttons and submit inputs.
      if (input.type === 'button' || input.type === 'submit') {
        return;
      }

      const name = input.name;
      if (!name) {
        return;
      }

      // Parse the name attribute to extract the key path.
      // Names can be like: settings[key], settings[parent][child], checkboxes[key].
      let value = Drupal.ebAggrid.getInputValue(input);

      // Skip unchecked multi-select checkboxes (checkboxes where value matches the key).
      // Boolean checkboxes (value="1") should always be included to capture false values.
      if (input.type === 'checkbox' && !input.checked) {
        const inputValue = input.value;
        // If value is not "1" or empty, it's a multi-select checkbox - skip if unchecked.
        if (inputValue !== '1' && inputValue !== '') {
          return;
        }
      }

      // Handle nested names like 'settings[handler_settings][target_bundles][article]'.
      Drupal.ebAggrid.setNestedFormValue(settings, name, value, input);
    });

    return settings;
  };

  /**
   * Get value from an input element.
   *
   * @param {HTMLElement} input
   *   The input element.
   *
   * @return {*}
   *   The extracted value.
   */
  Drupal.ebAggrid.getInputValue = (input) => {
    switch (input.type) {
      case 'checkbox':
        return input.checked;
      case 'number':
        return input.value !== '' ? parseFloat(input.value) : null;
      case 'select-multiple':
        return Array.from(input.selectedOptions).map(opt => opt.value);
      default:
        return input.value;
    }
  };

  /**
   * Set a value in a nested object based on form field name.
   *
   * Handles Form API naming like 'settings[key]' or 'settings[a][b]'.
   *
   * @param {Object} obj
   *   The object to modify.
   * @param {string} name
   *   The form field name (e.g., 'settings[key]').
   * @param {*} value
   *   The value to set.
   * @param {HTMLElement} input
   *   The input element (for checkbox handling).
   */
  Drupal.ebAggrid.setNestedFormValue = (obj, name, value, input) => {
    // Parse name like 'settings[handler_settings][target_bundles][article]'.
    const match = name.match(/^([^\[]+)(.*)$/);
    if (!match) {
      return;
    }

    const baseName = match[1];
    const rest = match[2];

    // If no brackets, just set directly.
    if (!rest) {
      if (input.type === 'checkbox') {
        obj[baseName] = input.checked;
      }
      else {
        obj[baseName] = value;
      }
      return;
    }

    // Parse all the bracket keys.
    const keys = [];
    const bracketRegex = /\[([^\]]*)\]/g;
    let bracketMatch;
    while ((bracketMatch = bracketRegex.exec(rest)) !== null) {
      keys.push(bracketMatch[1]);
    }

    // Build the nested structure.
    let current = obj;
    if (!current[baseName]) {
      current[baseName] = {};
    }
    current = current[baseName];

    for (let i = 0; i < keys.length - 1; i++) {
      const key = keys[i];
      if (!current[key]) {
        current[key] = {};
      }
      current = current[key];
    }

    // Set the final value.
    const lastKey = keys[keys.length - 1];
    if (input.type === 'checkbox') {
      // Detect checkbox type based on value attribute.
      // Boolean checkboxes (from Drupal Form API) have value="1".
      // Multi-select checkboxes have value matching the option key.
      const inputValue = input.value;
      if (inputValue === '1' || inputValue === '') {
        // Boolean checkbox - store true/false.
        current[lastKey] = input.checked;
      }
      else {
        // Multi-select checkbox - only store checked values with key as value.
        if (input.checked) {
          current[lastKey] = inputValue;
        }
      }
    }
    else {
      current[lastKey] = value;
    }
  };

  /**
   * Get settings schema for a field type.
   *
   * Schema comes from PHP DiscoveryService, NOT hardcoded.
   *
   * @param {string} fieldType
   *   The field type ID.
   *
   * @return {Object|null}
   *   The schema object or null if not found.
   */
  Drupal.ebAggrid.getSettingsSchema = (fieldType) => {
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    const schemas = discovery.fieldTypeFormSchemas || {};
    return schemas[fieldType] || null;
  };

  /**
   * Get a nested value from an object using dot notation path.
   *
   * @param {Object} obj
   *   The object to read from.
   * @param {string} path
   *   Dot-separated path like 'handler_settings.target_bundles'.
   *
   * @return {*}
   *   The value at the path, or undefined if not found.
   */
  Drupal.ebAggrid.getNestedValue = (obj, path) => {
    if (!obj || !path) {
      return undefined;
    }
    const parts = path.split('.');
    let current = obj;
    for (const part of parts) {
      if (current === null || current === undefined) {
        return undefined;
      }
      current = current[part];
    }
    return current;
  };

  /**
   * Set a nested value in an object using dot notation path.
   *
   * Creates intermediate objects as needed.
   *
   * @param {Object} obj
   *   The object to modify.
   * @param {string} path
   *   Dot-separated path like 'handler_settings.target_bundles'.
   * @param {*} value
   *   The value to set.
   */
  Drupal.ebAggrid.setNestedValue = (obj, path, value) => {
    if (!obj || !path) {
      return;
    }
    const parts = path.split('.');
    let current = obj;
    for (let i = 0; i < parts.length - 1; i++) {
      if (current[parts[i]] === undefined || current[parts[i]] === null) {
        current[parts[i]] = {};
      }
      current = current[parts[i]];
    }
    current[parts[parts.length - 1]] = value;
  };

  // NOTE: generateSchemaFromSettings() and renderPluginSettingsFields() were
  // removed in favor of server-side AJAX rendering. Widget/formatter settings
  // forms are now fetched via fetchPluginSettingsForm() and use Drupal's
  // native settingsForm() method for full Form API support.

  /**
   * Render tabs for settings modal.
   *
   * @param {string} activeTab
   *   The tab to show as active.
   * @param {Object} tabConfig
   *   Configuration for which tabs to show.
   *
   * @return {string}
   *   HTML string for tabs.
   */
  Drupal.ebAggrid.renderSettingsTabs = (activeTab = 'storage', tabConfig = {}) => {
    const tabs = [
      { id: 'storage', label: Drupal.t('Storage'), show: tabConfig.showStorage !== false },
      { id: 'config', label: Drupal.t('Instance'), show: tabConfig.showConfig !== false },
      { id: 'widget', label: Drupal.t('Widget'), show: tabConfig.showWidget !== false },
      { id: 'formatter', label: Drupal.t('Formatter'), show: tabConfig.showFormatter !== false }
    ];

    const visibleTabs = tabs.filter(t => t.show);
    if (visibleTabs.length === 0) return '';

    let html = '<div class="eb-settings-tabs">';
    visibleTabs.forEach((tab) => {
      const activeClass = tab.id === activeTab ? ' eb-settings-tab--active' : '';
      html += `<button type="button" class="eb-settings-tab${activeClass}" data-tab="${tab.id}">${tab.label}</button>`;
    });
    html += '</div>';
    return html;
  };

  /**
   * Render JSON settings editor for widget/formatter settings.
   *
   * @param {string} fieldId
   *   The field ID for the textarea.
   * @param {Object} settings
   *   The current settings object.
   * @param {string} label
   *   Label for the editor.
   *
   * @return {string}
   *   HTML string for JSON editor.
   */
  Drupal.ebAggrid.renderJsonSettingsEditor = (fieldId, settings, label) => {
    const jsonValue = settings && Object.keys(settings).length > 0
      ? JSON.stringify(settings, null, 2)
      : '{}';
    const escapedValue = Drupal.checkPlain(jsonValue);

    return `
      <div class="eb-settings-field">
        <label for="${fieldId}" class="eb-settings-label">${Drupal.checkPlain(label)}</label>
        <textarea id="${fieldId}" class="eb-settings-json-editor" rows="8">${escapedValue}</textarea>
        <p class="eb-settings-description">${Drupal.t('Enter settings as JSON. Use {} for empty settings.')}</p>
      </div>
    `;
  };

  /**
   * Render settings form HTML for a field type.
   *
   * Storage and Config settings are rendered client-side from schema.
   * Widget and Formatter settings are loaded via AJAX from Drupal's native
   * settingsForm() method for full Form API support.
   *
   * @param {string} fieldType
   *   The field type ID.
   * @param {Object} rowData
   *   The current row data containing field_storage_settings, field_config_settings,
   *   widget, formatter, widget_settings, and formatter_settings.
   *
   * @return {string}
   *   HTML string for the form.
   */
  Drupal.ebAggrid.renderSettingsForm = (fieldType, rowData) => {
    const schema = Drupal.ebAggrid.getSettingsSchema(fieldType);
    const hasStorage = schema?.storage && Object.keys(schema.storage).length > 0;
    const hasConfig = schema?.config && Object.keys(schema.config).length > 0;

    // Store current values for use by getDynamicOptions during initial render.
    const renderContext = {
      storage: rowData.field_storage_settings || {},
      config: rowData.field_config_settings || {}
    };

    // Determine which tabs to show.
    const tabConfig = {
      showStorage: hasStorage,
      showConfig: hasConfig,
      showWidget: true,
      showFormatter: true
    };

    // Determine first active tab.
    let activeTab = 'storage';
    if (!hasStorage && hasConfig) activeTab = 'config';
    else if (!hasStorage && !hasConfig) activeTab = 'widget';

    let html = '';

    // Render tabs.
    html += Drupal.ebAggrid.renderSettingsTabs(activeTab, tabConfig);

    // Storage Settings Panel (client-side).
    if (hasStorage) {
      const activeClass = activeTab === 'storage' ? ' eb-settings-panel--active' : '';
      html += `<div class="eb-settings-panel${activeClass}" data-panel="storage">`;
      html += '<div class="eb-settings-section">';
      html += `<h3 class="eb-settings-section__title">${Drupal.t('Storage Settings')}</h3>`;
      html += `<p class="eb-settings-section__description">${Drupal.t('Database-level settings. Cannot be changed after field is created.')}</p>`;
      html += Drupal.ebAggrid.renderFormFields(schema.storage, rowData.field_storage_settings || {}, 'storage', renderContext);
      html += '</div>';
      html += '</div>';
    }

    // Config/Instance Settings Panel (client-side).
    if (hasConfig) {
      const activeClass = activeTab === 'config' ? ' eb-settings-panel--active' : '';
      html += `<div class="eb-settings-panel${activeClass}" data-panel="config">`;
      html += '<div class="eb-settings-section">';
      html += `<h3 class="eb-settings-section__title">${Drupal.t('Instance Settings')}</h3>`;
      html += `<p class="eb-settings-section__description">${Drupal.t('Per-bundle settings. Can be changed after field is created.')}</p>`;
      html += Drupal.ebAggrid.renderFormFields(schema.config, rowData.field_config_settings || {}, 'config', renderContext);
      html += '</div>';
      html += '</div>';
    }

    // Widget Settings Panel (AJAX-loaded).
    const widgetActiveClass = activeTab === 'widget' ? ' eb-settings-panel--active' : '';
    html += `<div class="eb-settings-panel${widgetActiveClass}" data-panel="widget">`;
    html += '<div class="eb-settings-section">';
    html += `<h3 class="eb-settings-section__title">${Drupal.t('Widget Settings')}</h3>`;
    if (rowData.widget) {
      const discovery = drupalSettings.ebAggrid?.discovery || {};
      const widgetInfo = discovery.widgets?.[rowData.widget] || {};
      const widgetLabel = widgetInfo.label || rowData.widget;
      html += `<p class="eb-settings-section__description">${Drupal.t('Settings for widget: @widget', { '@widget': widgetLabel })}</p>`;
    }
    // Container for AJAX-loaded content.
    html += `<div class="eb-plugin-settings-container" data-plugin-type="widget" data-plugin-id="${Drupal.checkPlain(rowData.widget || '')}">`;
    html += Drupal.ebAggrid.renderPluginSettingsLoading('widget');
    html += '</div>';
    html += '</div>';
    html += '</div>';

    // Formatter Settings Panel (AJAX-loaded).
    const formatterActiveClass = activeTab === 'formatter' ? ' eb-settings-panel--active' : '';
    html += `<div class="eb-settings-panel${formatterActiveClass}" data-panel="formatter">`;
    html += '<div class="eb-settings-section">';
    html += `<h3 class="eb-settings-section__title">${Drupal.t('Formatter Settings')}</h3>`;
    if (rowData.formatter) {
      const discovery = drupalSettings.ebAggrid?.discovery || {};
      const formatterInfo = discovery.formatters?.[rowData.formatter] || {};
      const formatterLabel = formatterInfo.label || rowData.formatter;
      html += `<p class="eb-settings-section__description">${Drupal.t('Settings for formatter: @formatter', { '@formatter': formatterLabel })}</p>`;
    }
    // Container for AJAX-loaded content.
    html += `<div class="eb-plugin-settings-container" data-plugin-type="formatter" data-plugin-id="${Drupal.checkPlain(rowData.formatter || '')}">`;
    html += Drupal.ebAggrid.renderPluginSettingsLoading('formatter');
    html += '</div>';
    html += '</div>';
    html += '</div>';

    // If no content at all.
    if (!hasStorage && !hasConfig && !rowData.widget && !rowData.formatter) {
      return `<p class="eb-settings-no-config">${Drupal.t('No configurable settings available. Select a field type first.')}</p>`;
    }

    return html;
  };

  /**
   * Initialize AJAX loading for plugin settings panels.
   *
   * Called after the settings form HTML is inserted into the DOM.
   *
   * @param {HTMLElement} modal
   *   The modal element.
   * @param {Object} rowData
   *   The row data with field context.
   */
  Drupal.ebAggrid.initPluginSettingsPanels = (modal, rowData) => {
    // Load widget settings.
    const widgetContainer = modal.querySelector('.eb-plugin-settings-container[data-plugin-type="widget"]');
    if (widgetContainer && rowData.widget) {
      Drupal.ebAggrid.loadPluginSettingsForm(widgetContainer, 'widget', rowData.widget, rowData);
    }
    else if (widgetContainer) {
      widgetContainer.innerHTML = `<p class="eb-settings-empty">${Drupal.t('Select a widget type first in the grid.')}</p>`;
    }

    // Load formatter settings.
    const formatterContainer = modal.querySelector('.eb-plugin-settings-container[data-plugin-type="formatter"]');
    if (formatterContainer && rowData.formatter) {
      Drupal.ebAggrid.loadPluginSettingsForm(formatterContainer, 'formatter', rowData.formatter, rowData);
    }
    else if (formatterContainer) {
      formatterContainer.innerHTML = `<p class="eb-settings-empty">${Drupal.t('Select a formatter type first in the grid.')}</p>`;
    }
  };

  /**
   * Render form fields from schema.
   *
   * @param {Object} schema
   *   The schema object with field definitions.
   * @param {Object} values
   *   Current values for the fields.
   * @param {string} prefix
   *   Either 'storage' or 'config'.
   * @param {Object} renderContext
   *   Optional context with all current values for dynamic options.
   *
   * @return {string}
   *   HTML string for the form fields.
   */
  Drupal.ebAggrid.renderFormFields = (schema, values, prefix, renderContext) => {
    let html = '';

    Object.keys(schema).forEach((fieldName) => {
      const fieldDef = schema[fieldName];
      // Support nested paths like 'handler_settings.target_bundles'.
      const value = fieldDef.nested_path
        ? Drupal.ebAggrid.getNestedValue(values, fieldDef.nested_path)
        : values[fieldName];
      const inputId = `eb-settings-${prefix}-${fieldName}`;
      const inputName = `${prefix}[${fieldName}]`;

      html += '<div class="eb-settings-field">';

      // Label (except for checkboxes which have inline labels).
      if (fieldDef.type !== 'checkbox') {
        html += `<label for="${inputId}" class="eb-settings-label">`;
        html += Drupal.checkPlain(fieldDef.label || fieldName);
        if (fieldDef.required) {
          html += ' <span class="eb-required">*</span>';
        }
        html += '</label>';
      }

      // Render appropriate input type.
      switch (fieldDef.type) {
        case 'select':
          html += Drupal.ebAggrid.renderSelect(inputId, inputName, fieldDef, value, prefix, fieldName, renderContext);
          break;
        case 'number':
          html += Drupal.ebAggrid.renderNumber(inputId, inputName, fieldDef, value);
          break;
        case 'checkbox':
          html += Drupal.ebAggrid.renderCheckbox(inputId, inputName, fieldDef, value);
          break;
        case 'checkboxes':
          html += Drupal.ebAggrid.renderCheckboxes(inputId, inputName, fieldDef, value, prefix, fieldName, renderContext);
          break;
        case 'keyvalue':
          html += Drupal.ebAggrid.renderKeyValue(inputId, inputName, fieldDef, value);
          break;
        case 'text':
        default:
          html += Drupal.ebAggrid.renderText(inputId, inputName, fieldDef, value);
      }

      // Description.
      if (fieldDef.description) {
        html += `<div class="eb-settings-description">${Drupal.checkPlain(fieldDef.description)}</div>`;
      }

      html += '</div>';
    });

    return html;
  };

  /**
   * Render a select dropdown.
   *
   * @param {string} id
   *   The input ID.
   * @param {string} name
   *   The input name.
   * @param {Object} fieldDef
   *   The field definition from schema.
   * @param {*} value
   *   The current value.
   * @param {string} prefix
   *   The prefix (storage or config).
   * @param {string} fieldName
   *   The field name.
   * @param {Object} renderContext
   *   Optional context with all current values for dynamic options.
   *
   * @return {string}
   *   HTML string.
   */
  Drupal.ebAggrid.renderSelect = (id, name, fieldDef, value, prefix, fieldName, renderContext) => {
    let html = `<select id="${id}" name="${name}" class="eb-settings-select"`;
    html += ` data-field="${fieldName}" data-prefix="${prefix}"`;
    if (fieldDef.required) {
      html += ' required';
    }
    html += '>';

    // Add empty option.
    html += `<option value="">${Drupal.t('- Select -')}</option>`;

    // Get options from schema or dynamic source.
    const options = Drupal.ebAggrid.getFieldOptions(fieldDef, prefix, fieldName, renderContext);

    Object.keys(options).forEach((optionValue) => {
      const optionLabel = options[optionValue];
      const selected = value === optionValue ? ' selected' : '';
      html += `<option value="${Drupal.checkPlain(optionValue)}"${selected}>`;
      html += Drupal.checkPlain(optionLabel);
      html += '</option>';
    });

    html += '</select>';
    return html;
  };

  /**
   * Render a number input.
   *
   * @param {string} id
   *   The input ID.
   * @param {string} name
   *   The input name.
   * @param {Object} fieldDef
   *   The field definition from schema.
   * @param {*} value
   *   The current value.
   *
   * @return {string}
   *   HTML string.
   */
  Drupal.ebAggrid.renderNumber = (id, name, fieldDef, value) => {
    const currentValue = value !== undefined && value !== null ? value : (fieldDef.default || '');
    let html = `<input type="number" id="${id}" name="${name}"`;
    html += ' class="eb-settings-input eb-settings-input--number"';
    html += ` value="${Drupal.checkPlain(String(currentValue))}"`;

    if (fieldDef.min !== undefined) {
      html += ` min="${fieldDef.min}"`;
    }
    if (fieldDef.max !== undefined) {
      html += ` max="${fieldDef.max}"`;
    }
    if (fieldDef.required) {
      html += ' required';
    }
    html += '>';
    return html;
  };

  /**
   * Render a single checkbox.
   *
   * @param {string} id
   *   The input ID.
   * @param {string} name
   *   The input name.
   * @param {Object} fieldDef
   *   The field definition from schema.
   * @param {*} value
   *   The current value.
   *
   * @return {string}
   *   HTML string.
   */
  Drupal.ebAggrid.renderCheckbox = (id, name, fieldDef, value) => {
    const checked = value === true || value === 1 || value === '1' ? ' checked' : '';
    let html = '<div class="eb-settings-checkbox">';
    html += `<input type="checkbox" id="${id}" name="${name}" value="1"${checked}>`;
    html += ` <label for="${id}" class="eb-settings-checkbox-label">`;
    html += Drupal.checkPlain(fieldDef.label || name);
    if (fieldDef.required) {
      html += ' <span class="eb-required">*</span>';
    }
    html += '</label>';
    html += '</div>';
    return html;
  };

  /**
   * Render multiple checkboxes (for bundles, etc.).
   *
   * @param {string} id
   *   The input ID.
   * @param {string} name
   *   The input name.
   * @param {Object} fieldDef
   *   The field definition from schema.
   * @param {*} value
   *   The current value (array of selected keys).
   * @param {string} prefix
   *   The prefix (storage or config).
   * @param {string} fieldName
   *   The field name.
   * @param {Object} renderContext
   *   Optional context with all current values for dynamic options.
   *
   * @return {string}
   *   HTML string.
   */
  Drupal.ebAggrid.renderCheckboxes = (id, name, fieldDef, value, prefix, fieldName, renderContext) => {
    let selectedValues = value || {};
    if (Array.isArray(value)) {
      selectedValues = {};
      value.forEach((v) => { selectedValues[v] = v; });
    }

    // Get options from schema or dynamic source.
    const options = Drupal.ebAggrid.getFieldOptions(fieldDef, prefix, fieldName, renderContext);

    let html = `<div class="eb-settings-checkboxes" id="${id}-wrapper"`;
    html += ` data-field="${fieldName}" data-prefix="${prefix}">`;

    if (Object.keys(options).length === 0) {
      html += `<p class="eb-settings-checkboxes-empty">${Drupal.t('No options available. Select a target type first.')}</p>`;
    }
    else {
      Object.keys(options).forEach((optionValue, index) => {
        const optionLabel = options[optionValue];
        const checked = selectedValues[optionValue] ? ' checked' : '';
        const checkId = `${id}-${index}`;
        html += '<div class="eb-settings-checkbox-item">';
        html += `<input type="checkbox" id="${checkId}" name="${name}[${optionValue}]"`;
        html += ` value="${Drupal.checkPlain(optionValue)}"${checked}>`;
        html += ` <label for="${checkId}">${Drupal.checkPlain(optionLabel)}</label>`;
        html += '</div>';
      });
    }

    html += '</div>';
    return html;
  };

  /**
   * Render a key-value editor (for allowed_values).
   *
   * @param {string} id
   *   The input ID.
   * @param {string} name
   *   The input name.
   * @param {Object} fieldDef
   *   The field definition from schema.
   * @param {*} value
   *   The current value (object with key-value pairs).
   *
   * @return {string}
   *   HTML string.
   */
  Drupal.ebAggrid.renderKeyValue = (id, name, fieldDef, value) => {
    const pairs = value || {};
    let html = `<div class="eb-settings-keyvalue" id="${id}-wrapper">`;

    html += '<div class="eb-settings-keyvalue-items">';
    const keys = Object.keys(pairs);
    if (keys.length === 0) {
      // Add one empty row.
      html += Drupal.ebAggrid.renderKeyValueRow(name, '', '', 0);
    }
    else {
      keys.forEach((key, index) => {
        html += Drupal.ebAggrid.renderKeyValueRow(name, key, pairs[key], index);
      });
    }
    html += '</div>';

    html += `<button type="button" class="eb-settings-keyvalue-add button button--small"`;
    html += ` data-target="${id}-wrapper">`;
    html += Drupal.t('+ Add value');
    html += '</button>';

    html += '</div>';
    return html;
  };

  /**
   * Render a single key-value row.
   *
   * @param {string} name
   *   The input name base.
   * @param {string} key
   *   The key.
   * @param {string} value
   *   The value.
   * @param {number} index
   *   The row index.
   *
   * @return {string}
   *   HTML string.
   */
  Drupal.ebAggrid.renderKeyValueRow = (name, key, value, index) => {
    let html = '<div class="eb-settings-keyvalue-row">';
    html += '<input type="text" class="eb-settings-input eb-settings-keyvalue-key"';
    html += ` placeholder="${Drupal.t('Key')}" value="${Drupal.checkPlain(key)}"`;
    html += ` data-role="key" data-index="${index}">`;
    html += '<input type="text" class="eb-settings-input eb-settings-keyvalue-value"';
    html += ` placeholder="${Drupal.t('Label')}" value="${Drupal.checkPlain(String(value))}"`;
    html += ` data-role="value" data-index="${index}">`;
    html += `<button type="button" class="eb-settings-keyvalue-remove" title="${Drupal.t('Remove')}">&times;</button>`;
    html += '</div>';
    return html;
  };

  /**
   * Render a text input.
   *
   * @param {string} id
   *   The input ID.
   * @param {string} name
   *   The input name.
   * @param {Object} fieldDef
   *   The field definition from schema.
   * @param {*} value
   *   The current value.
   *
   * @return {string}
   *   HTML string.
   */
  Drupal.ebAggrid.renderText = (id, name, fieldDef, value) => {
    const currentValue = value !== undefined && value !== null ? value : (fieldDef.default || '');
    let html = `<input type="text" id="${id}" name="${name}"`;
    html += ' class="eb-settings-input eb-settings-input--text"';
    html += ` value="${Drupal.checkPlain(String(currentValue))}"`;
    if (fieldDef.required) {
      html += ' required';
    }
    html += '>';
    return html;
  };

  /**
   * Get options for a field from schema or dynamic source.
   *
   * @param {Object} fieldDef
   *   The field definition from schema.
   * @param {string} prefix
   *   The prefix (storage or config).
   * @param {string} fieldName
   *   The field name.
   * @param {Object} renderContext
   *   Optional context with all current values for dynamic options.
   *
   * @return {Object}
   *   Options as key-label pairs.
   */
  Drupal.ebAggrid.getFieldOptions = (fieldDef, prefix, fieldName, renderContext) => {
    // If static options are defined in schema, use them.
    if (fieldDef.options && typeof fieldDef.options === 'object') {
      return fieldDef.options;
    }

    // Handle dynamic options sources.
    if (fieldDef.options_source) {
      return Drupal.ebAggrid.getDynamicOptions(fieldDef.options_source, prefix, fieldName, renderContext);
    }

    return {};
  };

  /**
   * Get dynamic options from various sources.
   *
   * @param {string} source
   *   The options source identifier.
   * @param {string} prefix
   *   The prefix (storage or config).
   * @param {string} fieldName
   *   The field name.
   * @param {Object} renderContext
   *   Optional context with all current values for dynamic options.
   *
   * @return {Object}
   *   Options as key-label pairs.
   */
  Drupal.ebAggrid.getDynamicOptions = (source, prefix, fieldName, renderContext) => {
    switch (source) {
      case 'entityTypes':
        return Drupal.ebAggrid.getEntityTypeOptions();

      case 'bundlesForTargetType': {
        // First try to get target_type from renderContext (during initial render).
        let targetType = null;
        if (renderContext?.storage?.target_type) {
          targetType = renderContext.storage.target_type;
        }
        // Fallback to reading from DOM (for subsequent renders/changes).
        if (!targetType) {
          targetType = Drupal.ebAggrid.getSettingsFormValue('storage', 'target_type');
        }
        if (targetType) {
          return Drupal.ebAggrid.getBundleOptionsForType(targetType);
        }
        return {};
      }

      case 'textFormats':
        return Drupal.ebAggrid.getTextFormatOptions();

      default:
        return {};
    }
  };

  /**
   * Get entity type options for select fields.
   *
   * @return {Object}
   *   Entity types as id-label pairs.
   */
  Drupal.ebAggrid.getEntityTypeOptions = () => {
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    const entityTypes = discovery.entityTypes || [];
    const entityTypeLabels = discovery.entityTypeLabels || {};
    const options = {};

    entityTypes.forEach((entityType) => {
      options[entityType] = entityTypeLabels[entityType] || entityType;
    });

    return options;
  };

  /**
   * Get text format options for formatted text fields.
   *
   * @return {Object}
   *   Text formats as id-label pairs.
   */
  Drupal.ebAggrid.getTextFormatOptions = () => {
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    return discovery.textFormats || {};
  };

  /**
   * Get bundle options for a specific entity type.
   *
   * Includes both existing bundles from Drupal AND bundles from Bundles tab.
   *
   * @param {string} entityType
   *   The entity type ID.
   *
   * @return {Object}
   *   Bundles as id-label pairs.
   */
  Drupal.ebAggrid.getBundleOptionsForType = (entityType) => {
    const options = {};
    const discovery = drupalSettings.ebAggrid?.discovery || {};
    const bundleLabels = discovery.bundleLabels || {};

    // Add existing bundles from discovery.
    const existingBundles = bundleLabels[entityType] || {};
    Object.keys(existingBundles).forEach((bundleId) => {
      options[bundleId] = existingBundles[bundleId];
    });

    // Add bundles from Bundles tab (new, not yet created).
    const bundleGridApi = Drupal.ebAggrid.grids?.bundle;
    if (bundleGridApi) {
      bundleGridApi.forEachNode((node) => {
        if (node.data?.entity_type === entityType && node.data?.bundle_id) {
          const bundleId = node.data.bundle_id;
          if (!options[bundleId]) {
            options[bundleId] = node.data.label || bundleId;
          }
        }
      });
    }

    return options;
  };

  // ============================================
  // FIELD VALUE EXTRACTORS
  // ============================================

  /**
   * Field value extractors by type.
   *
   * Each extractor function takes a modal element and input ID,
   * and returns the extracted value for that field type.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.fieldExtractors = {
    /**
     * Extract checkbox value.
     *
     * @param {HTMLElement} modal
     *   The modal element.
     * @param {string} inputId
     *   The input element ID.
     *
     * @return {boolean}
     *   True if checked, false otherwise.
     */
    checkbox: (modal, inputId) => {
      const element = modal.querySelector(`#${inputId}`);
      return element ? element.checked : false;
    },

    /**
     * Extract multi-checkbox values.
     *
     * @param {HTMLElement} modal
     *   The modal element.
     * @param {string} inputId
     *   The input wrapper ID (without -wrapper suffix).
     *
     * @return {Object|null}
     *   Object of selected values or null if none selected.
     */
    checkboxes: (modal, inputId) => {
      const wrapper = modal.querySelector(`#${inputId}-wrapper`);
      if (!wrapper) {
        return null;
      }
      const selected = {};
      wrapper.querySelectorAll('input[type="checkbox"]:checked').forEach((cb) => {
        selected[cb.value] = cb.value;
      });
      return Object.keys(selected).length > 0 ? selected : null;
    },

    /**
     * Extract key-value pairs.
     *
     * @param {HTMLElement} modal
     *   The modal element.
     * @param {string} inputId
     *   The input wrapper ID (without -wrapper suffix).
     *
     * @return {Object}
     *   Object of key-value pairs.
     */
    keyvalue: (modal, inputId) => {
      const wrapper = modal.querySelector(`#${inputId}-wrapper`);
      if (!wrapper) {
        return {};
      }
      const keyvalues = {};
      wrapper.querySelectorAll('.eb-settings-keyvalue-row').forEach((row) => {
        const keyInput = row.querySelector('[data-role="key"]');
        const valueInput = row.querySelector('[data-role="value"]');
        if (keyInput && valueInput && keyInput.value.trim()) {
          keyvalues[keyInput.value.trim()] = valueInput.value;
        }
      });
      return Object.keys(keyvalues).length > 0 ? keyvalues : {};
    },

    /**
     * Extract number value.
     *
     * @param {HTMLElement} modal
     *   The modal element.
     * @param {string} inputId
     *   The input element ID.
     *
     * @return {number|null}
     *   Parsed number or null if empty.
     */
    number: (modal, inputId) => {
      const element = modal.querySelector(`#${inputId}`);
      if (!element || element.value === '') {
        return null;
      }
      return parseInt(element.value, 10);
    },

    /**
     * Extract select value.
     *
     * @param {HTMLElement} modal
     *   The modal element.
     * @param {string} inputId
     *   The input element ID.
     *
     * @return {string|null}
     *   Selected value or null if empty.
     */
    select: (modal, inputId) => {
      const element = modal.querySelector(`#${inputId}`);
      if (!element || element.value === '') {
        return null;
      }
      return element.value;
    },

    /**
     * Extract text value.
     *
     * @param {HTMLElement} modal
     *   The modal element.
     * @param {string} inputId
     *   The input element ID.
     *
     * @return {string|null}
     *   Text value or null if empty.
     */
    text: (modal, inputId) => {
      const element = modal.querySelector(`#${inputId}`);
      if (!element || element.value === '') {
        return null;
      }
      return element.value;
    }
  };

  /**
   * Get a value from the current settings form.
   *
   * @param {string} prefix
   *   The prefix (storage or config).
   * @param {string} fieldName
   *   The field name.
   *
   * @return {*}
   *   The current value or null.
   */
  Drupal.ebAggrid.getSettingsFormValue = (prefix, fieldName) => {
    const modal = document.getElementById('eb-settings-modal');
    if (!modal) {
      return null;
    }

    const input = modal.querySelector(`[data-field="${fieldName}"][data-prefix="${prefix}"]`);
    if (input) {
      return input.value;
    }

    return null;
  };

  /**
   * Collect form values from the settings modal.
   *
   * Uses type-specific extractors for cleaner code structure.
   *
   * @param {HTMLElement} modal
   *   The modal element.
   * @param {string} prefix
   *   Either 'storage' or 'config'.
   * @param {string} fieldType
   *   The field type ID.
   *
   * @return {Object}
   *   Collected values as an object.
   */
  Drupal.ebAggrid.collectFormValues = (modal, prefix, fieldType) => {
    const values = {};
    const schema = Drupal.ebAggrid.getSettingsSchema(fieldType);
    if (!schema || !schema[prefix]) {
      return values;
    }

    Object.keys(schema[prefix]).forEach((fieldName) => {
      const fieldDef = schema[prefix][fieldName];
      const inputId = `eb-settings-${prefix}-${fieldName}`;

      // Use type-specific extractor, defaulting to text extractor.
      const extractor = Drupal.ebAggrid.fieldExtractors[fieldDef.type] ||
                      Drupal.ebAggrid.fieldExtractors.text;
      const fieldValue = extractor(modal, inputId);

      // Handle nested paths like 'handler_settings.target_bundles'.
      if (fieldDef.nested_path && fieldValue !== null) {
        Drupal.ebAggrid.setNestedValue(values, fieldDef.nested_path, fieldValue);
      }
      else if (fieldValue !== null) {
        values[fieldName] = fieldValue;
      }
    });

    return values;
  };

  // NOTE: collectPluginSettings() was removed in favor of collectDrupalFormSettings()
  // which handles Drupal Form API rendered forms from the AJAX endpoint.

  /**
   * Attach event listeners to the settings form.
   *
   * @param {HTMLElement} modal
   *   The modal element.
   * @param {string} fieldType
   *   The field type ID.
   */
  Drupal.ebAggrid.attachSettingsFormListeners = (modal, fieldType) => {
    // Handle tab switching.
    modal.querySelectorAll('.eb-settings-tab').forEach((tab) => {
      tab.addEventListener('click', (e) => {
        const tabName = e.target.dataset.tab;
        // Hide all panels, show selected.
        modal.querySelectorAll('.eb-settings-panel').forEach((p) => {
          p.classList.remove('eb-settings-panel--active');
        });
        const targetPanel = modal.querySelector(`[data-panel="${tabName}"]`);
        if (targetPanel) {
          targetPanel.classList.add('eb-settings-panel--active');
        }
        // Update tab active state.
        modal.querySelectorAll('.eb-settings-tab').forEach((t) => {
          t.classList.remove('eb-settings-tab--active');
        });
        e.target.classList.add('eb-settings-tab--active');
      });
    });

    // Handle target_type change to rebuild bundle options.
    const targetTypeSelect = modal.querySelector('[data-field="target_type"][data-prefix="storage"]');
    if (targetTypeSelect) {
      targetTypeSelect.addEventListener('change', () => {
        Drupal.ebAggrid.rebuildBundleCheckboxes(modal, fieldType);
      });
    }

    // Handle key-value add button.
    modal.querySelectorAll('.eb-settings-keyvalue-add').forEach((btn) => {
      btn.addEventListener('click', () => {
        const targetId = btn.getAttribute('data-target');
        const wrapper = document.getElementById(targetId);
        if (wrapper) {
          const itemsContainer = wrapper.querySelector('.eb-settings-keyvalue-items');
          const rows = itemsContainer.querySelectorAll('.eb-settings-keyvalue-row');
          const newIndex = rows.length;
          const newRow = document.createElement('div');
          newRow.innerHTML = Drupal.ebAggrid.renderKeyValueRow('', '', '', newIndex);
          itemsContainer.appendChild(newRow.firstChild);
          // Attach remove listener to new row.
          Drupal.ebAggrid.attachKeyValueRemoveListeners(itemsContainer);
        }
      });
    });

    // Attach remove listeners to existing key-value rows.
    modal.querySelectorAll('.eb-settings-keyvalue').forEach((wrapper) => {
      Drupal.ebAggrid.attachKeyValueRemoveListeners(wrapper);
    });
  };

  /**
   * Attach remove listeners to key-value rows.
   *
   * @param {HTMLElement} container
   *   The key-value container element.
   */
  Drupal.ebAggrid.attachKeyValueRemoveListeners = (container) => {
    container.querySelectorAll('.eb-settings-keyvalue-remove').forEach((btn) => {
      btn.addEventListener('click', () => {
        const row = btn.closest('.eb-settings-keyvalue-row');
        if (row) {
          row.remove();
        }
      });
    });
  };

  /**
   * Rebuild bundle checkboxes when target_type changes.
   *
   * Completely rebuilds the checkbox HTML based on available bundles
   * for the selected target entity type.
   *
   * @param {HTMLElement} modal
   *   The modal element.
   * @param {string} fieldType
   *   The field type ID.
   */
  Drupal.ebAggrid.rebuildBundleCheckboxes = (modal, fieldType) => {
    const targetTypeSelect = modal.querySelector('[data-field="target_type"][data-prefix="storage"]');
    if (!targetTypeSelect) {
      return;
    }

    const newTargetType = targetTypeSelect.value;
    const bundleWrapper = modal.querySelector('[data-field="target_bundles"][data-prefix="config"]');
    if (!bundleWrapper) {
      return;
    }

    // Get bundle options for new target type.
    const options = newTargetType ? Drupal.ebAggrid.getBundleOptionsForType(newTargetType) : {};

    // Rebuild checkboxes.
    let html = '';
    if (Object.keys(options).length === 0) {
      html = `<p class="eb-settings-checkboxes-empty">${Drupal.t('No bundles available. Select a target type first.')}</p>`;
    }
    else {
      Object.keys(options).forEach((optionValue, index) => {
        const optionLabel = options[optionValue];
        const checkId = `eb-settings-config-target_bundles-${index}`;
        html += '<div class="eb-settings-checkbox-item">';
        html += `<input type="checkbox" id="${checkId}" name="config[target_bundles][${optionValue}]"`;
        html += ` value="${Drupal.checkPlain(optionValue)}">`;
        html += ` <label for="${checkId}">${Drupal.checkPlain(optionLabel)}</label>`;
        html += '</div>';
      });
    }

    bundleWrapper.innerHTML = html;
  };

})(Drupal, drupalSettings);
