/**
 * @file
 * Validation logic for EB UI module.
 *
 * Handles both client-side and server-side validation, error display,
 * and validation status indicators.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  /**
   * Initialize the ebUi namespace if not exists.
   */
  Drupal.ebAggrid = Drupal.ebAggrid || {};

  /**
   * Storage for validation errors.
   *
   * @type {Object}
   */
  Drupal.ebAggrid.validationErrors = {};
  Drupal.ebAggrid.serverValidationErrors = {};
  Drupal.ebAggrid.validationTimer = null;
  Drupal.ebAggrid.isValidating = false;

  /**
   * Validate a single row.
   *
   * @param {string} gridType
   *   The grid type.
   * @param {Object} node
   *   The row node.
   *
   * @return {Object}
   *   Object with field errors.
   */
  Drupal.ebAggrid.validateRow = (gridType, node) => {
    const errors = {};
    const { data } = node;
    const required = Drupal.ebAggrid.REQUIRED_FIELDS[gridType] || [];

    // Check required fields.
    required.forEach((field) => {
      const value = data[field];
      if (value === undefined || value === null || value === '') {
        errors[field] = Drupal.t('This field is required.');
      }
    });

    // Validate field_name format.
    if (data.field_name && !data.field_name.startsWith('field_')) {
      errors.field_name = Drupal.t('Field name must start with "field_".');
    }

    // Validate group_name format.
    if (data.group_name && !data.group_name.startsWith('group_')) {
      errors.group_name = Drupal.t('Group name must start with "group_".');
    }

    // Validate machine name format (bundle_id, menu_id).
    Drupal.ebAggrid.MACHINE_NAME_FIELDS.forEach((field) => {
      if (data[field] && !/^[a-z][a-z0-9_]*$/.test(data[field])) {
        errors[field] = Drupal.t('Must start with a letter and contain only lowercase letters, numbers, and underscores.');
      }
    });

    // Validate cardinality.
    if (data.cardinality !== undefined && data.cardinality !== '') {
      const cardVal = parseInt(data.cardinality, 10);
      if (isNaN(cardVal) || (cardVal !== -1 && cardVal < 1)) {
        errors.cardinality = Drupal.t('Must be -1 (unlimited) or a positive integer.');
      }
    }

    return errors;
  };

  /**
   * Validate all rows in a grid.
   *
   * @param {string} gridType
   *   The grid type.
   * @param {Object} api
   *   The grid API.
   */
  Drupal.ebAggrid.validateAllRows = (gridType, api) => {
    if (!api) {
      return;
    }

    // Initialize error storage.
    Drupal.ebAggrid.validationErrors[gridType] = {};

    // Validate each row.
    api.forEachNode((node) => {
      const errors = Drupal.ebAggrid.validateRow(gridType, node);
      if (Object.keys(errors).length > 0) {
        Drupal.ebAggrid.validationErrors[gridType][node.id] = errors;
      }
    });

    // Refresh all cells to update styling.
    api.refreshCells({ force: true });

    // Update tab error indicators.
    Drupal.ebAggrid.updateTabErrorIndicators();
  };

  /**
   * Validate cell value after change.
   *
   * @param {Object} event
   *   Cell value changed event.
   */
  Drupal.ebAggrid.validateCell = (event) => {
    const gridType = event.context?.gridType;
    if (!gridType) {
      return;
    }

    const rowId = event.node.id;

    // Validate entire row to catch cross-field dependencies.
    const errors = Drupal.ebAggrid.validateRow(gridType, event.node);

    // Initialize error storage.
    if (!Drupal.ebAggrid.validationErrors[gridType]) {
      Drupal.ebAggrid.validationErrors[gridType] = {};
    }

    // Update errors for this row.
    if (Object.keys(errors).length > 0) {
      Drupal.ebAggrid.validationErrors[gridType][rowId] = errors;
    }
    else {
      delete Drupal.ebAggrid.validationErrors[gridType][rowId];
    }

    // Refresh the row to update styling.
    event.api.refreshCells({
      rowNodes: [event.node],
      force: true
    });

    // Update row status with explicit gridType.
    if (typeof Drupal.ebAggrid.updateRowStatus === 'function') {
      Drupal.ebAggrid.updateRowStatus(event.api, event.node, gridType);
    }

    // Update tab error indicators.
    Drupal.ebAggrid.updateTabErrorIndicators();
  };

  /**
   * Trigger debounced server-side validation.
   *
   * @param {string} gridType
   *   The grid type that changed.
   */
  Drupal.ebAggrid.triggerDebouncedValidation = (gridType) => {
    // Clear existing timer.
    if (Drupal.ebAggrid.validationTimer) {
      clearTimeout(Drupal.ebAggrid.validationTimer);
    }

    const debounceMs = Drupal.ebAggrid.TIMING?.VALIDATION_DEBOUNCE_MS ?? 500;

    // Set new timer for debounced validation.
    Drupal.ebAggrid.validationTimer = setTimeout(() => {
      Drupal.ebAggrid.validateViaAjax(gridType);
    }, debounceMs);
  };

  /**
   * Validate grid data via AJAX.
   *
   * @param {string} gridType
   *   Optional grid type to validate. If null, validates all.
   *
   * @return {Promise}
   *   Promise that resolves with validation result.
   */
  Drupal.ebAggrid.validateViaAjax = (gridType) => {
    if (Drupal.ebAggrid.isValidating) {
      return Promise.resolve({ valid: true });
    }

    Drupal.ebAggrid.isValidating = true;
    Drupal.ebAggrid.updateValidationStatus('validating');

    // Collect all grid data.
    const allData = Drupal.ebAggrid.collectAllGridData();

    // Build YAML-like structure for validation endpoint.
    const yamlContent = JSON.stringify(allData);

    return Drupal.ebUi.fetchJson('/eb/api/validate', {
      method: 'POST',
      headers: Drupal.ebUi.getPostHeaders(),
      body: yamlContent
    })
      .then((result) => {
        Drupal.ebAggrid.isValidating = false;

        if (result.valid) {
          Drupal.ebAggrid.updateValidationStatus('valid');
          Drupal.ebAggrid.serverValidationErrors = {};
        }
        else {
          Drupal.ebAggrid.updateValidationStatus('error');
          Drupal.ebAggrid.mapServerErrors(result.errors || []);
        }

        // Refresh all grids to show error styling.
        Drupal.ebAggrid.refreshAllGrids();

        return result;
      })
      .catch((error) => {
        Drupal.ebAggrid.isValidating = false;
        Drupal.ebAggrid.updateValidationStatus('error');
        console.error('Validation error:', error);
        return { valid: false, errors: [error.message] };
      });
  };

  /**
   * Fetch extension warnings separately from eb_aggrid endpoint.
   *
   * @return {Promise}
   *   Promise that resolves with extension warnings array.
   */
  Drupal.ebAggrid.fetchExtensionWarnings = () => {
    // Collect all grid data.
    const allData = Drupal.ebAggrid.collectAllGridData();
    const jsonContent = JSON.stringify(allData);

    return Drupal.ebUi.fetchJson('/eb/api/extension-warnings', {
      method: 'POST',
      headers: Drupal.ebUi.getPostHeaders(),
      body: jsonContent
    })
      .then((result) => result.extension_warnings || [])
      .catch((error) => {
        console.error('Extension warnings fetch error:', error);
        return [];
      });
  };

  /**
   * Validate all grids (called by Validate All button).
   *
   * Calls validation endpoint, then fetches extension warnings separately
   * and merges them into the result.
   *
   * @return {Promise}
   *   Promise that resolves with validation result including extension_warnings.
   */
  Drupal.ebAggrid.validateAll = () => {
    return Drupal.ebAggrid.validateViaAjax(null)
      .then((validationResult) => {
        // Fetch extension warnings and merge into result.
        return Drupal.ebAggrid.fetchExtensionWarnings()
          .then((extensionWarnings) => ({
            ...validationResult,
            extension_warnings: extensionWarnings
          }));
      });
  };

  /**
   * Map server validation errors to grid cells.
   *
   * @param {Array} errors
   *   Array of error messages from server.
   */
  Drupal.ebAggrid.mapServerErrors = (errors) => {
    // Clear existing server errors.
    Drupal.ebAggrid.serverValidationErrors = {};

    // Parse error messages and map to cells.
    errors.forEach((error) => {
      // For now, store as general errors.
      if (!Drupal.ebAggrid.serverValidationErrors._general) {
        Drupal.ebAggrid.serverValidationErrors._general = [];
      }
      Drupal.ebAggrid.serverValidationErrors._general.push(error);
    });

    // Update tabs with error indicators.
    Drupal.ebAggrid.updateTabErrorIndicators();
  };

  /**
   * Update validation status indicator.
   *
   * @param {string} status
   *   Status: 'validating', 'valid', 'error', 'idle'.
   */
  Drupal.ebAggrid.updateValidationStatus = (status) => {
    const indicator = document.querySelector('.eb-validation-status');
    if (!indicator) {
      return;
    }

    indicator.className = `eb-validation-status eb-validation-status--${status}`;

    const statusText = {
      validating: Drupal.t('Validating...'),
      valid: Drupal.t('Valid'),
      error: Drupal.t('Has errors'),
      idle: ''
    };

    const icon = indicator.querySelector('.eb-validation-status__icon');
    const text = indicator.querySelector('.eb-validation-status__text');

    if (icon) {
      icon.className = `eb-validation-status__icon eb-validation-status__icon--${status}`;
    }
    if (text) {
      text.textContent = statusText[status] || '';
    }
  };

  /**
   * Update tab error indicators.
   */
  Drupal.ebAggrid.updateTabErrorIndicators = () => {
    const tabGridMap = Drupal.ebAggrid.TAB_GRID_MAP;

    Object.keys(tabGridMap).forEach((tabId) => {
      const gridType = tabGridMap[tabId];
      const tab = document.querySelector(`.eb-sheet-tab[data-tab="${tabId}"]`);
      if (!tab) {
        return;
      }

      let hasErrors = false;

      // Check client errors.
      const clientErrors = Drupal.ebAggrid.validationErrors[gridType];
      if (clientErrors) {
        Object.keys(clientErrors).forEach((rowId) => {
          if (Object.keys(clientErrors[rowId]).length > 0) {
            hasErrors = true;
          }
        });
      }

      // Check server errors.
      const serverErrors = Drupal.ebAggrid.serverValidationErrors[gridType];
      if (serverErrors) {
        hasErrors = true;
      }

      if (hasErrors) {
        tab.classList.add('eb-sheet-tab--has-error');
      }
      else {
        tab.classList.remove('eb-sheet-tab--has-error');
      }
    });
  };

  /**
   * Check if a cell has validation error.
   *
   * @param {Object} params
   *   Cell params.
   *
   * @return {boolean}
   *   True if cell has error.
   */
  Drupal.ebAggrid.hasCellError = (params) => {
    if (!params.context || !params.node || !params.colDef) {
      return false;
    }

    const gridType = params.context.gridType;
    const rowId = params.node.id;
    const field = params.colDef.field;

    const errors = Drupal.ebAggrid.validationErrors[gridType];
    return errors && errors[rowId] && errors[rowId][field];
  };

  /**
   * Get cell error message for tooltip.
   *
   * @param {Object} params
   *   Cell params.
   *
   * @return {string|null}
   *   Error message or null.
   */
  Drupal.ebAggrid.getCellErrorMessage = (params) => {
    if (!params.context || !params.node || !params.colDef) {
      return null;
    }

    const gridType = params.context.gridType;
    const rowId = params.node.id;
    const field = params.colDef.field;

    // Check client-side errors.
    const clientErrors = Drupal.ebAggrid.validationErrors[gridType];
    if (clientErrors?.[rowId]?.[field]) {
      return clientErrors[rowId][field];
    }

    // Check server-side errors.
    const serverErrors = Drupal.ebAggrid.serverValidationErrors[gridType];
    if (serverErrors?.[rowId]?.[field]) {
      return serverErrors[rowId][field];
    }

    return null;
  };

  /**
   * Group validation errors by category for display.
   *
   * @param {Array} errors
   *   Array of error strings.
   *
   * @return {Object}
   *   Errors grouped by category.
   */
  Drupal.ebAggrid.groupValidationErrors = (errors) => {
    const grouped = {};
    const errorCategories = Drupal.ebAggrid.ERROR_CATEGORIES;

    errors.forEach((error) => {
      let category = 'Other';
      let message = error;
      let context = '';

      // Extract context from brackets at the end [context info].
      const contextMatch = error.match(/^(.+?)\s*\[([^\]]+)\]$/);
      if (contextMatch) {
        message = contextMatch[1];
        context = contextMatch[2];
      }

      // Categorize by error type using patterns.
      let foundCategory = false;

      Object.keys(errorCategories).forEach((pattern) => {
        if (foundCategory || pattern === 'default') {
          return;
        }

        if (message.includes(pattern)) {
          const categoryConfig = errorCategories[pattern];

          if (typeof categoryConfig === 'string') {
            category = categoryConfig;
            foundCategory = true;
          }
          else if (typeof categoryConfig === 'object') {
            // Check sub-patterns.
            Object.keys(categoryConfig).forEach((subPattern) => {
              if (foundCategory || subPattern === 'default') {
                return;
              }

              if (message.includes(subPattern)) {
                category = categoryConfig[subPattern];
                foundCategory = true;
              }
            });

            // Use default for this pattern group.
            if (!foundCategory && categoryConfig.default) {
              category = categoryConfig.default;
              foundCategory = true;
            }
          }
        }
      });

      // Use global default if no match.
      if (!foundCategory) {
        category = errorCategories.default || 'Validation Errors';
      }

      if (!grouped[category]) {
        grouped[category] = [];
      }

      grouped[category].push({
        original: error,
        message,
        context
      });
    });

    return grouped;
  };

})(Drupal, drupalSettings);
