<?php

namespace Drupal\eb_aggrid\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\eb_ui\Service\GridProviderManagerInterface;

/**
 * Access checker for eb_aggrid routes.
 *
 * Performs two levels of access checking:
 * 1. Provider check: Ensures eb_aggrid is the active grid provider.
 * 2. Entity access: For edit routes, validates user can update the definition.
 *
 * This enables route priority (eb_aggrid takes precedence over eb_ui when
 * active) while also enforcing Tier 1 ownership-based permissions.
 */
class GridProviderAccess implements AccessInterface {

  /**
   * Constructor.
   *
   * @param \Drupal\eb_ui\Service\GridProviderManagerInterface $providerManager
   *   The grid provider manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(
    protected GridProviderManagerInterface $providerManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Checks access for eb_aggrid routes.
   *
   * For add routes (no definition_id), checks:
   * - eb_aggrid is active provider
   * - User has create access.
   *
   * For edit routes (with definition_id), checks:
   * - eb_aggrid is active provider
   * - User has update access for the specific definition
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   * @param string|null $definition_id
   *   The definition ID from the route parameter, or NULL for add routes.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account, ?string $definition_id = NULL): AccessResultInterface {
    $provider = $this->providerManager->getActiveProvider();

    // First check: aggrid must be the active provider.
    $is_active = $provider && $provider['id'] === 'aggrid';

    if (!$is_active) {
      return AccessResult::forbidden('AG-Grid provider is not active.')
        ->addCacheTags(['config:eb_ui.settings'])
        ->cachePerPermissions();
    }

    // Second check: entity access.
    $access_handler = $this->entityTypeManager->getAccessControlHandler('eb_definition');

    if (empty($definition_id)) {
      // Add route: check create access.
      $entity_access = $access_handler->createAccess(NULL, $account, [], TRUE);
    }
    else {
      // Edit route: check update access on the specific definition.
      $definition = $this->entityTypeManager
        ->getStorage('eb_definition')
        ->load($definition_id);

      if (!$definition) {
        return AccessResult::forbidden('Definition not found.')
          ->addCacheTags(['eb_definition_list', 'config:eb_ui.settings']);
      }

      $entity_access = $definition->access('update', $account, TRUE);
    }

    // Combine provider check with entity access.
    return $entity_access
      ->addCacheTags(['config:eb_ui.settings']);
  }

}
