<?php

namespace Drupal\eb_aggrid\Controller;

use Drupal\Core\Access\CsrfRequestHeaderAccessCheck;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Drupal\Core\Field\FormatterPluginManager;
use Drupal\Core\Field\WidgetPluginManager;
use Drupal\Core\Form\FormState;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\ElementInfoManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\eb\PluginManager\EbExtensionPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * AJAX controller for Entity Builder AG-GRID specific operations.
 *
 * This controller handles grid-specific AJAX requests from the eb_aggrid
 * JavaScript. Core validation, preview, and entity config endpoints are
 * provided by EbUiApiController in the eb_ui module.
 *
 * Endpoints:
 * - /eb/api/plugin-settings: Widget/formatter settings forms
 * - /eb/api/format-settings: Field group format settings forms
 * - /eb/api/extension-warnings: Detect disabled extension module warnings
 */
class EbAggridApiController extends ControllerBase {

  /**
   * Bytes per megabyte for unit conversion.
   */
  protected const BYTES_PER_MB = 1048576;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrfTokenGenerator
   *   The CSRF token generator service.
   * @param \Drupal\eb\PluginManager\EbExtensionPluginManager $extensionManager
   *   The extension plugin manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Field\WidgetPluginManager $widgetManager
   *   The widget plugin manager.
   * @param \Drupal\Core\Field\FormatterPluginManager $formatterManager
   *   The formatter plugin manager.
   * @param \Drupal\Core\Field\FieldTypePluginManagerInterface $fieldTypeManager
   *   The field type plugin manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Render\ElementInfoManagerInterface $elementInfo
   *   The element info manager.
   */
  public function __construct(
    ModuleHandlerInterface $moduleHandler,
    protected CsrfTokenGenerator $csrfTokenGenerator,
    protected EbExtensionPluginManager $extensionManager,
    ConfigFactoryInterface $configFactory,
    protected WidgetPluginManager $widgetManager,
    protected FormatterPluginManager $formatterManager,
    protected FieldTypePluginManagerInterface $fieldTypeManager,
    protected RendererInterface $renderer,
    protected ElementInfoManagerInterface $elementInfo,
  ) {
    $this->moduleHandler = $moduleHandler;
    $this->configFactory = $configFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    // @phpstan-ignore new.static
    return new static(
      $container->get('module_handler'),
      $container->get('csrf_token'),
      $container->get('plugin.manager.eb_extension'),
      $container->get('config.factory'),
      $container->get('plugin.manager.field.widget'),
      $container->get('plugin.manager.field.formatter'),
      $container->get('plugin.manager.field.field_type'),
      $container->get('renderer'),
      $container->get('element_info'),
    );
  }

  /**
   * Validate AJAX request for CSRF protection.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param bool $requireCsrfToken
   *   Whether to require CSRF token validation. Default TRUE for POST requests.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|null
   *   Error response if validation fails, NULL if valid.
   */
  protected function validateAjaxRequest(Request $request, bool $requireCsrfToken = TRUE): ?JsonResponse {
    // Check X-Requested-With header for AJAX requests.
    if ($request->headers->get('X-Requested-With') !== 'XMLHttpRequest') {
      return new JsonResponse([
        'valid' => FALSE,
        'success' => FALSE,
        'errors' => ['Invalid request: Missing X-Requested-With header'],
      ], 403);
    }

    // Validate user session.
    if ($this->currentUser()->isAnonymous()) {
      return new JsonResponse([
        'valid' => FALSE,
        'success' => FALSE,
        'errors' => ['Invalid request: Authentication required'],
      ], 403);
    }

    // Validate CSRF token for state-changing requests.
    if ($requireCsrfToken) {
      $csrf_token = $request->headers->get('X-CSRF-Token');
      if (!$csrf_token || !$this->csrfTokenGenerator->validate($csrf_token, CsrfRequestHeaderAccessCheck::TOKEN_KEY)) {
        return new JsonResponse([
          'valid' => FALSE,
          'success' => FALSE,
          'errors' => ['Invalid request: CSRF token validation failed'],
        ], 403);
      }
    }

    // Check content size.
    $content_length = $request->headers->get('Content-Length', '0');
    $max_content_size = $this->configFactory->get('eb_aggrid.settings')->get('max_content_size') * self::BYTES_PER_MB;
    if ((int) $content_length > $max_content_size) {
      return new JsonResponse([
        'valid' => FALSE,
        'success' => FALSE,
        'errors' => ['Content exceeds maximum allowed size'],
      ], 413);
    }

    return NULL;
  }

  /**
   * AJAX endpoint for rendering widget/formatter settings forms.
   *
   * Returns server-rendered settings form HTML for a widget or formatter
   * plugin. Uses Drupal's native settingsForm() method on the plugin.
   *
   * Route: POST /admin/eb/ajax/plugin-settings/{plugin_type}/{plugin_id}
   * Permission: 'administer entity builder'
   *
   * POST data (JSON):
   * - field_type: The field type ID (e.g., 'string', 'entity_reference').
   * - current_settings: Current settings to populate the form.
   * - entity_type: Optional entity type context.
   * - bundle: Optional bundle context.
   *
   * Response (200 OK):
   * {
   *   "success": true,
   *   "hasSettings": true,
   *   "html": "<form>...</form>"
   * }
   *
   * Response (no settings):
   * {
   *   "success": true,
   *   "hasSettings": false,
   *   "message": "No configurable settings for this widget."
   * }
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request.
   * @param string $plugin_type
   *   The plugin type ('widget' or 'formatter').
   * @param string $plugin_id
   *   The plugin ID (e.g., 'string_textfield', 'string').
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with rendered form HTML.
   */
  public function getPluginSettingsForm(Request $request, string $plugin_type, string $plugin_id): JsonResponse {
    // Validate AJAX request with CSRF.
    $csrf_error = $this->validateAjaxRequest($request, TRUE);
    if ($csrf_error !== NULL) {
      return $csrf_error;
    }

    // Parse POST data.
    $content = $request->getContent();
    $data = json_decode($content, TRUE) ?? [];

    $field_type = $data['field_type'] ?? '';
    $current_settings = $data['current_settings'] ?? [];
    $entity_type = $data['entity_type'] ?? 'node';
    $bundle = $data['bundle'] ?? 'article';

    if (empty($field_type)) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'field_type is required',
      ], 400);
    }

    // Validate plugin exists.
    $manager = $plugin_type === 'widget' ? $this->widgetManager : $this->formatterManager;

    if (!$manager->hasDefinition($plugin_id)) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => sprintf('Plugin "%s" not found for type "%s".', $plugin_id, $plugin_type),
      ], 404);
    }

    try {
      // Create a field definition for context.
      $field_definition = $this->createFieldDefinitionForPlugin(
        $field_type,
        $entity_type,
        $bundle,
        $data['field_config_settings'] ?? [],
        $data['field_storage_settings'] ?? []
      );

      // Create plugin instance.
      $plugin = $manager->createInstance($plugin_id, [
        'field_definition' => $field_definition,
        'settings' => $current_settings,
        'label' => '',
        'view_mode' => 'default',
        'third_party_settings' => [],
      ]);

      // Build the settings form.
      $form = [];
      $form_state = new FormState();
      $settings_form = $plugin->settingsForm($form, $form_state);

      // Check if there are settings.
      if (empty($settings_form)) {
        return new JsonResponse([
          'success' => TRUE,
          'hasSettings' => FALSE,
          'message' => $this->t('No configurable settings for this @type.', [
            '@type' => $plugin_type,
          ])->render(),
        ]);
      }

      // Ensure all details elements are expanded by default.
      $this->openAllDetailsElements($settings_form);

      // Wrap in a form structure for proper processing.
      $wrapper_form = [
        '#type' => 'container',
        '#tree' => TRUE,
        '#parents' => [],
        '#array_parents' => [],
        '#attributes' => [
          'class' => ['eb-plugin-settings-form', 'eb-plugin-settings-form--' . $plugin_type],
          'data-plugin-type' => $plugin_type,
          'data-plugin-id' => $plugin_id,
        ],
        'settings' => $settings_form,
      ];

      // Process form elements to expand composite types (radios, checkboxes).
      // This triggers #process callbacks that create individual elements.
      $this->processFormElements($wrapper_form, $form_state);

      // Render to HTML.
      $html = $this->renderer->renderRoot($wrapper_form);

      return new JsonResponse([
        'success' => TRUE,
        'hasSettings' => TRUE,
        'html' => (string) $html,
      ]);
    }
    catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => $e->getMessage(),
      ], 500);
    }
  }

  /**
   * Create a field definition for instantiating widget/formatter plugins.
   *
   * @param string $field_type
   *   The field type ID.
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle machine name.
   * @param array<string, mixed> $field_config_settings
   *   Optional field config settings.
   * @param array<string, mixed> $field_storage_settings
   *   Optional field storage settings.
   *
   * @return \Drupal\Core\Field\BaseFieldDefinition
   *   A configured field definition.
   */
  protected function createFieldDefinitionForPlugin(
    string $field_type,
    string $entity_type,
    string $bundle,
    array $field_config_settings = [],
    array $field_storage_settings = [],
  ): BaseFieldDefinition {
    $field_definition = BaseFieldDefinition::create($field_type)
      ->setName('field_eb_temp')
      ->setTargetEntityTypeId($entity_type)
      ->setTargetBundle($bundle);

    // Apply storage settings (especially important for entity_reference).
    if (!empty($field_storage_settings)) {
      foreach ($field_storage_settings as $key => $value) {
        $field_definition->setSetting($key, $value);
      }
    }

    // Apply field config settings.
    if (!empty($field_config_settings)) {
      foreach ($field_config_settings as $key => $value) {
        $field_definition->setSetting($key, $value);
      }
    }

    // For entity_reference, ensure target_type is set.
    if ($field_type === 'entity_reference' && empty($field_storage_settings['target_type'])) {
      $field_definition->setSetting('target_type', 'node');
    }

    return $field_definition;
  }

  /**
   * Recursively opens all details elements in a render array.
   *
   * This ensures that collapsible sections are visible when the form
   * is rendered via AJAX.
   *
   * @param array<string, mixed> &$element
   *   The render array to process.
   */
  protected function openAllDetailsElements(array &$element): void {
    // If this element is a details type, open it.
    if (isset($element['#type']) && $element['#type'] === 'details') {
      $element['#open'] = TRUE;
    }

    // Recursively process children.
    foreach ($element as $key => &$child) {
      if (is_array($child) && !str_starts_with((string) $key, '#')) {
        $this->openAllDetailsElements($child);
      }
    }
  }

  /**
   * Process form elements to expand composite types like radios/checkboxes.
   *
   * This method recursively processes form elements, adding element defaults
   * and executing #process callbacks. This is necessary for composite elements
   * like radios and checkboxes that need their #options expanded into
   * individual elements.
   *
   * @param array<string, mixed> $element
   *   The form element to process.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param array<int|string, string> $parents
   *   The parent keys for this element.
   *
   * @param-out array<string, mixed> $element
   */
  protected function processFormElements(array &$element, FormStateInterface $form_state, array $parents = []): void {
    // Set up element parents if not already set.
    if (!isset($element['#parents'])) {
      $element['#parents'] = $parents;
    }
    if (!isset($element['#array_parents'])) {
      $element['#array_parents'] = $parents;
    }

    // Apply element info defaults (includes #process callbacks).
    if (isset($element['#type']) && empty($element['#defaults_loaded'])) {
      $info = $this->elementInfo->getInfo($element['#type']);
      if ($info) {
        $element += $info;
        $element['#defaults_loaded'] = TRUE;
      }
    }

    // Generate #name from #parents for form elements.
    // This is normally done by FormBuilder::doBuildForm(), but since we're
    // rendering outside of a full form context, we need to do it manually.
    if (!empty($element['#input']) && !isset($element['#name']) && !empty($element['#parents'])) {
      $parents_copy = $element['#parents'];
      $element['#name'] = array_shift($parents_copy);
      if ($parents_copy) {
        $element['#name'] .= '[' . implode('][', $parents_copy) . ']';
      }
    }

    // Copy #default_value to #value if not already set.
    // This is normally done by FormBuilder during form processing,
    // but we need to do it manually when rendering outside a form context.
    if (!empty($element['#input']) && !isset($element['#value']) && isset($element['#default_value'])) {
      $element['#value'] = $element['#default_value'];
    }

    // Execute #process callbacks.
    if (!empty($element['#process']) && empty($element['#processed'])) {
      foreach ($element['#process'] as $callback) {
        // Create a complete_form reference for callbacks that need it.
        $complete_form = &$element;
        $element = call_user_func_array($callback, [&$element, $form_state, &$complete_form]);
      }
      $element['#processed'] = TRUE;
    }

    // Recursively process children.
    foreach ($element as $key => &$child) {
      if (is_array($child) && !str_starts_with((string) $key, '#')) {
        $child_parents = array_merge($parents, [$key]);
        $this->processFormElements($child, $form_state, $child_parents);
      }
    }
  }

  /**
   * AJAX endpoint for rendering field_group format settings forms.
   *
   * Returns server-rendered settings form HTML for a field_group format type.
   * Uses field_group's native settingsForm() method on the formatter plugin.
   *
   * Route: POST /admin/eb/ajax/format-settings/{format_type}
   * Permission: 'administer entity builder'
   *
   * POST data (JSON):
   * - current_settings: Current settings to populate the form.
   * - display_type: The display type ('form' or 'view').
   *
   * Response (200 OK):
   * {
   *   "success": true,
   *   "hasSettings": true,
   *   "html": "<form>...</form>"
   * }
   *
   * Response (no settings):
   * {
   *   "success": true,
   *   "hasSettings": false,
   *   "message": "No configurable settings for this format type."
   * }
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request.
   * @param string $format_type
   *   The field_group format type (e.g., 'fieldset', 'details', 'tabs').
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with rendered form HTML.
   */
  public function getFormatSettingsForm(Request $request, string $format_type): JsonResponse {
    // Validate AJAX request with CSRF.
    $csrf_error = $this->validateAjaxRequest($request, TRUE);
    if ($csrf_error !== NULL) {
      return $csrf_error;
    }

    // Check if field_group module is installed.
    if (!$this->moduleHandler->moduleExists('field_group')) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'The field_group module is not installed.',
      ], 400);
    }

    // Get the field_group formatter plugin manager.
    // Service is from optional contrib module, can't use constructor injection.
    // @phpstan-ignore globalDrupalDependencyInjection.useDependencyInjection
    $formatter_manager = \Drupal::service('plugin.manager.field_group.formatters');

    // Validate format_type exists.
    if (!$formatter_manager->hasDefinition($format_type)) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => sprintf('Format type "%s" not found.', $format_type),
      ], 404);
    }

    // Parse POST data.
    $content = $request->getContent();
    $data = json_decode($content, TRUE) ?? [];

    $current_settings = $data['current_settings'] ?? [];
    $display_type = $data['display_type'] ?? 'form';

    try {
      // Get plugin definition for default settings.
      $definition = $formatter_manager->getDefinition($format_type);
      $default_settings = $definition['default_formatter'] ?? [];

      // Merge current settings with defaults.
      $settings = array_merge($default_settings, $current_settings);

      // Create plugin instance with minimal group configuration.
      // Use label from current settings if available.
      $group_label = $settings['label'] ?? 'Temporary Group';
      $group = (object) [
        'group_name' => 'temp_group',
        'label' => $group_label,
        'children' => [],
        'weight' => 0,
        'format_type' => $format_type,
        'format_settings' => $settings,
        'region' => 'content',
        'mode' => 'default',
        'entity_type' => 'node',
        'bundle' => 'article',
        'parent_name' => '',
        'context' => $display_type,
      ];

      // Field_group formatter manager expects 'label' at top level
      // and other settings inside a 'settings' key.
      $formatter = $formatter_manager->getInstance([
        'format_type' => $format_type,
        'configuration' => [
          'label' => $group_label,
          'settings' => $settings,
        ],
        'group' => $group,
      ]);

      if (!$formatter) {
        return new JsonResponse([
          'success' => FALSE,
          'message' => sprintf('Could not create instance for format type "%s".', $format_type),
        ], 500);
      }

      // Build the settings form.
      $settings_form = $formatter->settingsForm();

      // Check if there are settings.
      if (empty($settings_form)) {
        return new JsonResponse([
          'success' => TRUE,
          'hasSettings' => FALSE,
          'message' => $this->t('No configurable settings for this format type.')->render(),
        ]);
      }

      // Ensure all details elements are expanded by default.
      $this->openAllDetailsElements($settings_form);

      // Wrap in a form structure for proper processing.
      $wrapper_form = [
        '#type' => 'container',
        '#tree' => TRUE,
        '#parents' => [],
        '#array_parents' => [],
        '#attributes' => [
          'class' => ['eb-plugin-settings-form', 'eb-plugin-settings-form--format'],
          'data-format-type' => $format_type,
        ],
        'settings' => $settings_form,
      ];

      // Process form elements to expand composite types.
      $form_state = new FormState();
      $this->processFormElements($wrapper_form, $form_state);

      // Render to HTML.
      $html = $this->renderer->renderRoot($wrapper_form);

      return new JsonResponse([
        'success' => TRUE,
        'hasSettings' => TRUE,
        'html' => (string) $html,
      ]);
    }
    catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => $e->getMessage(),
      ], 500);
    }
  }

  /**
   * AJAX endpoint for detecting extension warnings.
   *
   * Checks if the provided definition data contains extension-specific
   * configuration for modules that are not currently enabled.
   *
   * Route: POST /eb/api/extension-warnings
   * Permission: 'administer entity builder'
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request with JSON body containing grid data.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with extension_warnings array.
   */
  public function getExtensionWarnings(Request $request): JsonResponse {
    // Validate AJAX request with CSRF.
    $csrf_error = $this->validateAjaxRequest($request, TRUE);
    if ($csrf_error !== NULL) {
      return $csrf_error;
    }

    // Parse JSON content.
    $content = $request->getContent();
    $data = json_decode($content, TRUE);

    if (!is_array($data)) {
      return new JsonResponse([
        'extension_warnings' => [],
      ]);
    }

    // Detect extension warnings.
    $warnings = $this->detectExtensionWarnings($data);

    return new JsonResponse([
      'extension_warnings' => array_values($warnings),
    ]);
  }

  /**
   * Detect warnings for extension data when extension modules are disabled.
   *
   * Checks if the provided definition data contains extension-specific
   * configuration for modules that are not currently enabled. Returns
   * warnings to inform users that some data won't be applied.
   *
   * @param array<string, mixed> $data
   *   The definition data to check.
   *
   * @return array<string, string>
   *   Array of warnings keyed by extension name.
   */
  protected function detectExtensionWarnings(array $data): array {
    $warnings = [];

    // Check bundle definitions for pathauto and auto_entitylabel data.
    if (!empty($data['bundle_definitions'])) {
      $has_pathauto = FALSE;
      $has_auto_entitylabel = FALSE;

      foreach ($data['bundle_definitions'] as $bundle) {
        if (!empty($bundle['pathauto_pattern'])) {
          $has_pathauto = TRUE;
        }
        if (!empty($bundle['auto_entitylabel_status']) || !empty($bundle['auto_entitylabel_pattern'])) {
          $has_auto_entitylabel = TRUE;
        }
      }

      if ($has_pathauto && !$this->moduleHandler->moduleExists('eb_pathauto')) {
        $warnings['pathauto'] = (string) $this->t('Definition contains pathauto patterns, but the eb_pathauto module is not enabled. These patterns will not be applied.');
      }
      if ($has_auto_entitylabel && !$this->moduleHandler->moduleExists('eb_auto_entitylabel')) {
        $warnings['auto_entitylabel'] = (string) $this->t('Definition contains auto entity label settings, but the eb_auto_entitylabel module is not enabled. These settings will not be applied.');
      }
    }

    // Check for field group definitions.
    if (!empty($data['field_group_definitions']) && !$this->moduleHandler->moduleExists('eb_field_group')) {
      $warnings['field_group'] = (string) $this->t('Definition contains field group configurations, but the eb_field_group module is not enabled. These groups will not be created.');
    }

    // Check for field_definitions that reference form_group or view_group.
    if (!empty($data['field_definitions']) && !$this->moduleHandler->moduleExists('eb_field_group')) {
      foreach ($data['field_definitions'] as $field) {
        if (!empty($field['form_group']) || !empty($field['view_group'])) {
          $warnings['field_group'] = (string) $this->t('Definition contains field group assignments, but the eb_field_group module is not enabled. Fields will be created without group assignments.');
          break;
        }
      }
    }

    return $warnings;
  }

}
